//
// NAME: CIDTracer_Plane.Cpp
//
// DESCRIPTION:
//
//  This module implements the TRTPlane class, which represents an infinite
//  an in the 3D space. It has a center point vector and a radius, along
//  with the standard attributes provided by the parent class.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 03/05/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData2(TRTPlane,TRTGeometricObj)



// -----------------------------------------------------------------------------
//  CLASS: TRTPlane
// PREFIX: pln
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TRTPlane: Constructors and Destructors
// -----------------------------------------------------------------------------

TRTPlane::TRTPlane() :

    __vecNormal(0.0, 1.0, 0.0)
    , __f8Distance(0.0)
{
}

TRTPlane::TRTPlane(const TRTPlane& plnSrc) :

    TRTGeometricObj(plnSrc)
    , __vecNormal(plnSrc.__vecNormal)
    , __f8Distance(plnSrc.__f8Distance)
{
    // Don't flip sign of distance since it is already flipped in source
}

TRTPlane::TRTPlane( const   T3DVector&       vecNormal
                    , const tCIDLib::TFloat8 f8Distance) :
    __vecNormal(vecNormal)
    , __f8Distance(-f8Distance)
{
}


// -----------------------------------------------------------------------------
//  TRTPlane: Public, virtual methods
// -----------------------------------------------------------------------------

TRTPlane&  TRTPlane::operator=(const TRTPlane& plnSrc)
{
    if (this == &plnSrc)
        return *this;

    ((TParent&)*this) = (const TParent&)plnSrc;

    __vecNormal = plnSrc.__vecNormal;
    __f8Distance = plnSrc.__f8Distance;

    return *this;
}


// -----------------------------------------------------------------------------
//  TRTPlane: Public, Inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid  TRTPlane::Invert()
{
    TParent::Invert();

    __vecNormal.Negate();
    __f8Distance *= -1.0;
}


tCIDLib::TVoid  TRTPlane::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens     eToken;
    TString                 strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    // Next token must be an open parens
    prsiSrc.CheckOpenParen();

    // Deal with the valid tokens at this level
    if (eToken == TParseInfo::eToken_Normal)
    {
        prsiSrc.ParseVector(__vecNormal);
    }
     else if (eToken == TParseInfo::eToken_Distance)
    {
        prsiSrc.ParseFloat(__f8Distance);
        __f8Distance *= -1;
    }
     else
    {
        // Unget the token and pass it to our parent
        prsiSrc.UnGet("(");
        prsiSrc.UnGet(strToken);
        TParent::ParseMethod(prsiSrc);
        return;
    }

    // Has to end with a closing paren
    prsiSrc.CheckCloseParen();
}


tCIDLib::TVoid  TRTPlane::Transform(const TTransform& trnsApply)
{
    // Call the parent version
    TParent::Transform(trnsApply);

    // Transform our normal
    trnsApply.mtrxTrans.Transform(__vecNormal);

    //
    //  Make a vector from the distance and transform it.
    //
    T3DVector vecTmp(__f8Distance, __f8Distance, __f8Distance);
    trnsApply.mtrxTrans.Transform(vecTmp);
    __f8Distance = vecTmp.f8Magnitude();
}


// -----------------------------------------------------------------------------
//  TRTPlane: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TRTPlane::_AllIntersects(   const   TLightRay&  lrayTest
                            , TIntersectArray&  intraHits) const
{
    tCIDLib::TFloat8 f8Dist1, f8Dist2;

    if (_bTestIntersect(lrayTest, f8Dist1, f8Dist2))
        intraHits.Add(new TIntersect(lrayTest, this, f8Dist1));
}


tCIDLib::TBoolean  TRTPlane::_bPointInObj(const T3DVector& vecPoint) const
{
    tCIDLib::TFloat8 f8Tmp;

    f8Tmp = vecPoint.f8Dot(__vecNormal);

    if ((f8Tmp + __f8Distance) <= kCIDTracer_::f8SmallTolerance)
        return kCIDLib::True;
    else
        return kCIDLib::False;
}


tCIDLib::TBoolean 
TRTPlane::_bTestIntersect(  const   TLightRay&          lrayTest
                            ,       tCIDLib::TFloat8&   f8Dist1
                            ,       tCIDLib::TFloat8&   f8Dist2) const
{
    tCIDLib::TFloat8 f8Dir, f8Org;

    f8Dir = __vecNormal.f8Dot(lrayTest.vecDir());
    if ((f8Dir < kCIDTracer_::f8SmallTolerance)
    &&  (f8Dir > - kCIDTracer_::f8SmallTolerance))
    {
        return kCIDLib::False;
    }

    // Calc the dot product of the plane and ray origin
    f8Org = __vecNormal.f8Dot(lrayTest.vecOrg());
    f8Org += __f8Distance;
    f8Org *= -1.0;

    //
    //  Calculate the distance. If less than 0, then the object is behind
    //  the ray origin.
    //
    f8Dist1 = f8Org / f8Dir;
    f8Dist2 = f8Dist1;

    if ((f8Dist1 >= kCIDTracer_::f8SmallTolerance)
    &&  (f8Dist1 <= kCIDTracer_::f8MaxDistance))
    {
        return kCIDLib::True;
    }
    return kCIDLib::False;
}


tCIDLib::TVoid  TRTPlane::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the attribute info into the string
    strmToWriteTo   << L"Dist: " << __f8Distance
                    << L", Normal: " << __vecNormal;
}


tCIDLib::TVoid  TRTPlane::_Rotate(   const   T3DVector&   vecRotate
                                            , const TTransform& trnsRotate)
{
    // Call the parent version
    TParent::_Rotate(vecRotate, trnsRotate);

    // Apply it to the plane normal
    trnsRotate.mtrxTrans.Transform(__vecNormal);
}


tCIDLib::TVoid  TRTPlane::_Scale(const   T3DVector&   vecScale
                                        , const TTransform& trnsScale)
{
    // Call the parent version
    TParent::_Scale(vecScale, trnsScale);

    tCIDLib::TFloat8 f8Len;

    __vecNormal /= vecScale;
    f8Len = __vecNormal.f8Magnitude();
    __vecNormal *= (1.0 / f8Len);
    __f8Distance /= f8Len;
}


tCIDLib::TVoid  TRTPlane::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Call our parent's version
    TParent::_StreamFrom(strmToReadFrom);

    // Get the center point vector out
    strmToReadFrom >> __vecNormal;

    // Get the distance out
    strmToReadFrom >> __f8Distance;
}

tCIDLib::TVoid  TRTPlane::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent's version
    TParent::_StreamTo(strmToWriteTo);

    // Write out the center point vector
    strmToWriteTo << __vecNormal;

    // Write out the distance
    strmToWriteTo << __f8Distance;
}


tCIDLib::TVoid 
TRTPlane::_Translate(   const   T3DVector&   vecTrans
                        , const TTransform& trnsTranslate)
{
    // Call the parent version
    TParent::_Translate(vecTrans, trnsTranslate);

    // Get a vector that will provide the translation
    T3DVector     vecTmp(vecTrans);
    vecTmp *= __vecNormal;

    // And subtract the sum of the magnitudes to get the distance
    __f8Distance -= vecTmp.f8MagSum();
}
