//
// NAME: CIDTracer_Parse.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDTracer_Parse.Cpp module. It defines the
//  TParseInfo and TRTParseErr.
//
//  TParseInfo is a wrapper around a TStreamParser object. The methods of
//  this class understand how to get data from the stream parser for specific
//  purposes.
//
//  TRTParseErr is an object that is thrown if an error occurs during
//  parsing of the scene file.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/16/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Forward references
// -----------------------------------------------------------------------------
class   TViewGeometry;


// -----------------------------------------------------------------------------
//  Typedef our bag of import names
// -----------------------------------------------------------------------------
typedef TBag<TString>   TBagOfImportNames;


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//   CLASS: TParseInfo
//  PREFIX: prsi
//
//  This is a local class that is just used to make the parsing operation
//  reentrant. We need somewhere to hold the instance data.
// -----------------------------------------------------------------------------
class CIDTRACEREXP TParseInfo : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Public, class specific types
        //
        //  eTokens
        //      This enum lists the possible language tokens. They are
        //      listed in generally larger orders of magnitude.
        //
        //  eTokenModes
        //      These are the mode flags that can be passed to the
        //      eNextToken() method. It lets that method do a lot of grunt
        //      work for the caller.
        // --------------------------------------------------------------------
        enum    eTokens
        {
            // Syntax tokens
            eToken_CloseBrace
            , eToken_CloseBracket
            , eToken_CloseParen
            , eToken_Color
            , eToken_Comma
            , eToken_Comment
            , eToken_DoubleColon
            , eToken_Import
            , eToken_Object
            , eToken_OpenBrace
            , eToken_OpenBracket
            , eToken_OpenParen
            , eToken_Period
            , eToken_SemiColon

            // Common texture attribute access methods
            , eToken_Ambient
            , eToken_Diffuse
            , eToken_Brilliance
            , eToken_Metallic
            , eToken_Phong
            , eToken_PhongSize
            , eToken_Randomness
            , eToken_Reflection
            , eToken_Refraction
            , eToken_RefractionIndex
            , eToken_Roughness
            , eToken_Specular

            // Noise textures methods
            , eToken_BumpAmount
            , eToken_Frequency
            , eToken_Phase
            , eToken_Turbulence

            // Checker/stripe texture methods
            , eToken_AltColor

            // Banded texture methods
            , eToken_AppendBand
            , eToken_PrependBand

            // Common object attribute access methods
            , eToken_AppendTexture
            , eToken_Bounding
            , eToken_Invert
            , eToken_PrependTexture
            , eToken_Texture
            , eToken_Visibility

            // Sphere methods
            , eToken_Center
            , eToken_Radius

            // Plane methods
            , eToken_Distance
            , eToken_Normal

            // Quadric methods
            , eToken_Constant
            , eToken_QuadricValues
            , eToken_Clip1
            , eToken_Clip2

            // CSG methods
            , eToken_CSGOperation

            // Mathematical mthods that can be applied to objects/txtrs
            , eToken_Rotate
            , eToken_Scale
            , eToken_Translate

            // Mathematical objects
            , eToken_Matrix
            , eToken_Vector

            // View methods
            , eToken_AddObject
            , eToken_AddLight
            , eToken_Scene
            , eToken_EyePosition
            , eToken_FogColor
            , eToken_Lens
            , eToken_LookAt
            , eToken_Depth
            , eToken_Quality
            , eToken_RightVector
            , eToken_Sample
            , eToken_UpVector

            // Miscellaneous
            , eToken_Unknown
            , eToken_End
        };

        enum    eTokenModes
        {
            eTokenMode_None             = 0UL
            , eTokenMode_UnknownOk      = 0x00000001UL
            , eTokenMode_EndOk          = 0x00000002UL
            , eTokenMode_UnknownOrEndOk = 0x00000003UL
        };


    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TParseInfo
        (
                    TViewGeometry&          viewTarget
        );

        ~TParseInfo();


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bCheckBraceOrSemiColon();

        tCIDLib::TBoolean bCheckBraceOrComma();

        tCIDLib::TBoolean bNameSpaceImported
        (
            const   TString&                strNameSpace
        )   const;

        tCIDLib::TCard4 c4CurLine();

        tCIDLib::TVoid CheckComma();

        tCIDLib::TVoid CheckCloseBrace();

        tCIDLib::TVoid CheckCloseBracket();

        tCIDLib::TVoid CheckCloseParen();

        tCIDLib::TVoid CheckOpenBrace();

        tCIDLib::TVoid CheckOpenBracket();

        tCIDLib::TVoid CheckOpenParen();

        tCIDLib::TVoid CheckPeriod();

        tCIDLib::TVoid CheckSemiColon();

        eTokens eLookUpToken
        (
            const   TString&                strText
        );

        tCIDLib::TVoid LookUpColor
        (
            const   TString&                strName
            , const TString&                strNameSpace
            ,       TFRGBClr&               frgbTarget
        );

        eTokens eNextToken
        (
                    TString&                strToken
            , const eTokenModes             eMode = eTokenMode_None
            , const TString&                strNoSyntax = NUL_TString
        );

        tCIDLib::TVoid GetName
        (
                    TString&                strName
        );

        tCIDLib::TVoid GetName
        (
                    TString&                strName
            ,       TString&                strNameSpace
        );

        tCIDLib::TVoid ParseBoolean
        (
                    tCIDLib::TBoolean&         bTarget
        );

        tCIDLib::TVoid ParseCard
        (
                    tCIDLib::TCard4&        c4Target
        );

        tCIDLib::TVoid ParseColor
        (
                    TFRGBClr&               frgbClr
        );

        tCIDLib::TVoid ParseColorMethod
        (
                    TFRGBClr* const         pfrgbTarget
        );

        tCIDLib::TVoid ParseFile
        (
            const   TString&                strFile
            , const TString&                strIncludePath
        );

        tCIDLib::TVoid ParseFloat
        (
                    tCIDLib::TFloat8&       f8Target
        );

        tCIDLib::TVoid ParseMethod
        (
            const   TString&                strName
            , const TString&                strNameSpace
        );

        tCIDLib::TVoid ParseVector
        (
                    T3DVector&              vecTarget
        );

        tCIDLib::TVoid ParseVectorMethod
        (
                    T3DVector* const        pvecTarget
        );

        TFRGBClr* pfrgbFind
        (
            const   TString&                strName
            , const TString&                strNameSpace
        );

        TRTObject* prtobjFind
        (
            const   TString&                strName
            , const TString&                strNameSpace
        );

        TRTObject* prtobjParseObject
        (
                    TString&                strName
        );

        TRTTexture* ptxtrFind
        (
            const   TString&                strName
            , const TString&                strNameSpace
        );

        TRTTexture* ptxtrParseTexture
        (
                    TString&                strName
        );

        T3DVector* pvecFind
        (
            const   TString&                strName
            , const TString&                strNameSpace
        );

        const TString& strCurFile() const;

        const TString& strCurNameSpace() const;

        tCIDLib::TVoid ThrowErr
        (
            const   tCIDLib::TCard4         c4Err
            , const TString&                strToken
            , const MFormattable&           fmtblToken1 = NUL_MFormattable
            , const MFormattable&           fmbtlToken2 = NUL_MFormattable
        );

        tCIDLib::TVoid UnGet
        (
            const   TString&                strToUnGet
        );

        tCIDLib::TVoid UnGet
        (
            const   tCIDLib::Tch* const     pszToUnGet
        );


    private     :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TParseInfo();

        TParseInfo(const TParseInfo&);

        TParseInfo& operator=(const TParseInfo&);


        // ---------------------------------------------------------------------
        //  Private, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid __ParseFile
        (
            const   TString&                strFile
            , const TString&                strIncludePath
        );


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __pbagImports
        //      The name spaces imported so far. This keeps us from parsing
        //      name spaces multiple times in multi-file scenes where the
        //      same name spaces can be imported multiple times. The
        //      bNameSpaceImported() method will search it.
        //
        //  __ppsrsInput
        //      This is the input stream parser object. It is allocated as
        //      needed for each level of recursion, which replaces the old
        //      one on exit.
        //
        //  __prtlstColors
        //  __prtlstObjects
        //  __prtlstTextures
        //  __prtlstVectors
        //      The lists of colors, lights, objects, textures, and vectors
        //      that have been parsed out of the file.
        //
        //  __strCurFile
        //      The name of the file that is currently being parsed. This
        //      is managed by ParseFile() who keeps the name of the
        //      current file here. When an exception is thrown we need to
        //      be able to give the name of the file where the error was
        //      (because the parsing is recursive.)
        //
        //  __strCurNameSpace
        //      The current name space that we are in at this time. It is
        //      set initially to "Main" for the main file.
        //
        //  __viewTarget
        //      This is a reference to the TViewGeometry object that we will
        //      be filling in.
        // ---------------------------------------------------------------------
        TBagOfImportNames*      __pbagImports;
        TStreamParser*          __pprsrInput;
        TRTClrList*             __prtlstColors;
        TRTObjList*             __prtlstObjects;
        TRTTxtrList*            __prtlstTextures;
        TVectorList*            __prtlstVectors;
        TString                 __strCurFile;
        TString                 __strCurNameSpace;
        TViewGeometry&          __viewTarget;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TParseInfo,TObject);
};



// -----------------------------------------------------------------------------
//   CLASS: TRTParseErr
//  PREFIX: rterr
// -----------------------------------------------------------------------------
class CIDTRACEREXP TRTParseErr
{
    public      :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TRTParseErr
        (
            const   TRTParseErr&            rterrToCopy
        );

        TRTParseErr
        (
            const   tCIDLib::TCard4         c4Line
            , const tCIDLib::TCard4         c4ErrMsgId
            , const TString&                strFile
            , const TString&                strObjName
            , const TString&                strMsg
        );


        // ---------------------------------------------------------------------
        //  Public data members
        //
        //  c4LineNum
        //      The number of the line that the error occured on.
        //
        //  c4MsgId
        //      This is the string id of the error message.
        //
        //  strFile
        //      This is file where the error occured.
        //
        //  strMsg
        //      This is the loaded message with tokens replaced.
        //
        //  strObjFail
        //      This is the name of the object/texture on which the error
        //      occured.
        // ---------------------------------------------------------------------
        tCIDLib::TCard4         c4LineNum;
        tCIDLib::TCard4         c4MsgId;
        TString                 strFile;
        TString                 strMsg;
        TString                 strObjFail;


    private     :
        // ---------------------------------------------------------------------
        //  Hidden constructors/operators
        // ---------------------------------------------------------------------
        TRTParseErr();

        tCIDLib::TVoid operator=
        (
            const   TRTParseErr&            rterrSrc
        );
};

#pragma pack(pop)



// -----------------------------------------------------------------------------
//  Public, non-virtual methods
// -----------------------------------------------------------------------------
inline const TString& TParseInfo::strCurFile() const
{
    return __strCurFile;
}

inline const TString& TParseInfo::strCurNameSpace() const
{
    return __strCurNameSpace;
}
