//
// NAME: CIDTracer_Parse.Cpp
//
// DESCRIPTION:
//
//  This module implements the TParseInfo class, which performs scene file
//  parsing.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/02/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility specific includes
//
//  The local lookup tables are included via CIDTracer_Parse_.Hpp with it sees
//  the CIDTracer_PARSE_CPP token defined.
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"
#include    "CIDTracer_Parse_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData(TParseInfo,TObject)


// -----------------------------------------------------------------------------
//  Local constant data
//
//  __c4MaxNameLen
//      The maximum length of a user definable symbol name.
// -----------------------------------------------------------------------------
static const tCIDLib::TCard4    __c4MaxNameLen = 32;


// -----------------------------------------------------------------------------
//  Local data
//
//  __ahshTokens
//      These are arrays of hash values for the token list. This list is a
//      constant lists of the names of stuff we need to treat specially in
//      the scene files. It is initialized by _bInitParser() below.
// -----------------------------------------------------------------------------
static tCIDLib::THashVal    __ahshTokens[__c4TokenCount];



// -----------------------------------------------------------------------------
//  Intra-facility modules
// -----------------------------------------------------------------------------

tCIDLib::TVoid _InitTermParser( const   tCIDLib::EInitTerm      eInitTerm
                                , const tCIDLib::EGlobalStates  eGlobals
                                , const TModule&                modInit
                                , const tCIDLib::TCard4         c4MaxChars
                                ,       tCIDLib::Tch* const     pszFailReason)
{
    if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
    &&  (eGlobals == tCIDLib::EGlobalState_Before))
    {
        for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4TokenCount; c4Ind++)
        {
            __ahshTokens[c4Ind] = TRawStr::hshHashStr
            (
                __atokList[c4Ind].szName
                , kCIDTracer::c4HashModulus
            );
        }
    }
}



// -----------------------------------------------------------------------------
//   CLASS: TRTParseErr
//  PREFIX: rterr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TRTParseErr: Constructors and destructors
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TRTParseErr
//
// DESCRIPTION:
//
// ---------------------------------------
//   INPUT: rterrSrc is the source error object to copy
//          c4Line is the line number of the error in the source file
//          strFile is the name of the file being parsed.
//          strObjName is the text that failed.
//
//  OUTPUT: None
//
//  RETURN: None
//
 TRTParseErr::TRTParseErr(const TRTParseErr& rterrSrc) :

    c4LineNum(rterrSrc.c4LineNum)
    , c4MsgId(rterrSrc.c4MsgId)
    , strFile(rterrSrc.strFile)
    , strMsg(rterrSrc.strMsg)
    , strObjFail(rterrSrc.strObjFail)
{
}

 TRTParseErr::TRTParseErr(   const   tCIDLib::TCard4 c4Line
                                    , const tCIDLib::TCard4 c4ErrMsgId
                                    , const TString&        strFile
                                    , const TString&        strObjName
                                    , const TString&        strMsgText) :
    c4LineNum(c4Line)
    , c4MsgId(c4ErrMsgId)
    , strFile(strFile)
    , strMsg(strMsgText)
    , strObjFail(strObjName)
{
}




// -----------------------------------------------------------------------------
//   CLASS: TParseInfo
//  PREFIX: pinfo
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TParseInfo
//                       ~ParseInfo
//
// DESCRIPTION:
//
//  This constructor sets up the parse info object, which has everything we
//  need to parse the file and create the view object. Once the object is
//  created, the client must call ParseFile() to kick off the parse. The
//  current name space is set initially to Main, which is the name space for
//  the main module of a scene.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: viewTarget is the target TViewGeometry object to fill in with
//              the info from the parsed file.
//
//  RETURN: None
//
 TParseInfo::TParseInfo(TViewGeometry& viewTarget) :

    __pbagImports(0)
    , __pprsrInput(0)
    , __prtlstColors(0)
    , __prtlstObjects(0)
    , __prtlstTextures(0)
    , __prtlstVectors(0)
    , __strCurNameSpace("Main")
    , __viewTarget(viewTarget)
{
    // Allocate all of the collection members
    __pbagImports       = new TBagOfImportNames;
    __prtlstColors      = new TRTClrList;
    __prtlstObjects     = new TRTObjList;
    __prtlstTextures    = new TRTTxtrList;
    __prtlstVectors     = new TVectorList;
}

 TParseInfo::~TParseInfo()
{
    // Clean up all of the collection members
    delete __prtlstColors;
    delete __prtlstObjects;
    delete __prtlstTextures;
    delete __prtlstVectors;
    delete __pbagImports;
    delete __pprsrInput;
}


// -----------------------------------------------------------------------------
//  TParseInfo:: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: bCheckBraceOrSemiColon
//                       bCheckBraceOrComma
//
// DESCRIPTION:
//
//  This method will check to see that the next token is either a closing
//  brace or a semicolon or a closing brace or a comma. If not it throws and
//  error.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: eTrue if a closing brace, else eFalse if a semicolon or comma
//
tCIDLib::TBoolean  TParseInfo::bCheckBraceOrSemiColon()
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    eTokens eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    if (eToken == eToken_CloseBrace)
        return kCIDLib::True;

    if (eToken == eToken_SemiColon)
        return kCIDLib::False;

    ThrowErr(kTracerErrs::errcParse_ExpectedBraceOrSemi, strToken);

    // Make compiler happy
    return kCIDLib::False;
}

tCIDLib::TBoolean  TParseInfo::bCheckBraceOrComma()
{
    TString  strToken(kCIDLib::pszEmptyZStr, 256);

    eTokens eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    if (eToken == eToken_CloseBrace)
        return kCIDLib::True;

    if (eToken == eToken_Comma)
        return kCIDLib::False;

    ThrowErr(kTracerErrs::errcParse_ExpectedBraceOrComma, strToken);

    // Make compiler happy
    return kCIDLib::False;
}


//
// FUNCTION/METHOD NAME: bNameSpaceImported
//
// DESCRIPTION:
//
//  Looks for the passed name space in the list of already imported spaces.
// ---------------------------------------
//   INPUT: strNameSpace is the name of the space to look up.
//
//  OUTPUT: None
//
//  RETURN: eTrue if this space is already imported, else eFalse.
//
tCIDLib::TBoolean 
TParseInfo::bNameSpaceImported(const TString& strNameSpace) const
{
    TBagOfImportNames::TCursor  cursSearch(__pbagImports);

    if (!cursSearch.bReset())
        return kCIDLib::False;

    do
    {
        if (cursSearch.objCur() == strNameSpace)
            return kCIDLib::True;
    }   while (cursSearch.bNext());

    return kCIDLib::False;
}


//
// FUNCTION/METHOD NAME: c4CurLine
//
// DESCRIPTION:
//
//  Just passes the call on to the input parser member.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: The current input line reported by the input parser.
//
tCIDLib::TCard4  TParseInfo::c4CurLine()
{
    return __pprsrInput->c4CurLine();
}


//
// FUNCTION/METHOD NAME: CheckCloseBrace
//                       CheckCloseBracket
//                       CheckCloseParen
//                       CheckComma
//                       CheckOpenBrace
//                       CheckOpenBracket
//                       CheckOpenParen
//                       CheckPeriod
//                       CheckSemiColon
//
// DESCRIPTION:
//
//  This method will read the next token and make sure that it is a
//  particular token. If not, then it will throw the appropriate error.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid  TParseInfo::CheckCloseBrace()
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    TParseInfo::eTokens eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    if (eToken != TParseInfo::eToken_CloseBrace)
        ThrowErr(kTracerErrs::errcParse_ExpectedCloseBrace, strToken);
}

tCIDLib::TVoid  TParseInfo::CheckCloseBracket()
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    TParseInfo::eTokens eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    if (eToken != TParseInfo::eToken_CloseBracket)
        ThrowErr(kTracerErrs::errcParse_ExpectedCloseBracket, strToken);
}

tCIDLib::TVoid  TParseInfo::CheckCloseParen()
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    TParseInfo::eTokens eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    if (eToken != TParseInfo::eToken_CloseParen)
        ThrowErr(kTracerErrs::errcParse_ExpectedCloseParen, strToken);
}

tCIDLib::TVoid  TParseInfo::CheckComma()
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    if (eNextToken(strToken, eTokenMode_UnknownOk) != TParseInfo::eToken_Comma)
        ThrowErr(kTracerErrs::errcParse_ExpectedComma, strToken);
}

tCIDLib::TVoid  TParseInfo::CheckOpenBrace()
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    TParseInfo::eTokens eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    if (eToken != TParseInfo::eToken_OpenBrace)
        ThrowErr(kTracerErrs::errcParse_ExpectedOpenBrace, strToken);
}

tCIDLib::TVoid  TParseInfo::CheckOpenBracket()
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    TParseInfo::eTokens eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    if (eToken != TParseInfo::eToken_OpenBracket)
        ThrowErr(kTracerErrs::errcParse_ExpectedOpenBracket, strToken);
}

tCIDLib::TVoid  TParseInfo::CheckOpenParen()
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    TParseInfo::eTokens eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    if (eToken != TParseInfo::eToken_OpenParen)
        ThrowErr(kTracerErrs::errcParse_ExpectedOpenParen, strToken);
}

tCIDLib::TVoid  TParseInfo::CheckPeriod()
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    if (eNextToken(strToken, eTokenMode_UnknownOk) != TParseInfo::eToken_Period)
        ThrowErr(kTracerErrs::errcParse_ExpectedPeriod, strToken);
}

tCIDLib::TVoid  TParseInfo::CheckSemiColon()
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    if (eNextToken(strToken, eTokenMode_UnknownOk) != TParseInfo::eToken_SemiColon)
        ThrowErr(kTracerErrs::errcParse_ExpectedSemiColon, strToken);
}


//
// FUNCTION/METHOD NAME: eLookUpToken
//
// DESCRIPTION:
//
//  These methods will try to find token with the passed name. If it is not
//  found, then an exception is thrown. In this case we are looking the list
//  of predefined names, not the list of user defined names from the file
//  being parsed.
// ---------------------------------------
//   INPUT: strName is the text to look up.
//
//  OUTPUT: None
//
//  RETURN: The enumerated representation of the scene component
//
TParseInfo::eTokens 
TParseInfo::eLookUpToken(const TString& strName)
{
    tCIDLib::THashVal hshTarget =
                        strName.hshCalcHash(kCIDTracer::c4HashModulus);

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4TokenCount; c4Ind++)
    {
        if (hshTarget != __ahshTokens[c4Ind])
            continue;

        if (strName == __atokList[c4Ind].szName)
            return __atokList[c4Ind].eToken;
    }
    return TParseInfo::eToken_Unknown;
}


//
// FUNCTION/METHOD NAME: eNextToken
//
// DESCRIPTION:
//
//  This function will return the next token from the input stream parser.
//  It handles throwing away comments.
// ---------------------------------------
//   INPUT: bUnknownOk indicates whether getting a non-keyword is ok
//          eMode indicates some special circumstances that allows this
//              method to handle some grunt work.
//          strNoSyntax is an optional string of characters that are usually
//              syntax chars but that should be treated normally in this
//              case. The default is NUL_TString.
//
//  OUTPUT: strToken is the string to fill in with the token text
//
//  RETURN: The TParseInfo::eTokens representation of the token.
//
TParseInfo::eTokens  
TParseInfo::eNextToken(         TString&    strToken
                        , const eTokenModes eMode
                        , const TString&    strNoSyntax)
{
    tCIDLib::TBoolean   bLoop   = kCIDLib::True;
    eTokens             eToken;

    while (bLoop)
    {
        __pprsrInput->GetNextToken(kCIDLib::szWhitespace, strToken, strNoSyntax);

        if (strToken.bEmpty())
            return eToken_End;

        // Look up the token
        eToken = eLookUpToken(strToken);

        // If unknown, then throw an exception if caller says so
        if ((eToken == eToken_Unknown) &&  !(eMode & eTokenMode_UnknownOk))
        {
            ThrowErr(kTracerErrs::errcParse_BadToken, strToken);
        }
         else if ((eToken == eToken_End) && !(eMode & eTokenMode_EndOk))
        {
            ThrowErr(kTracerErrs::errcParse_UnexpectedEOF, kCIDLib::pszEmptyZStr);
        }
         else if (eToken == eToken_Comment)
        {
            // Flush the rest of the line
            __pprsrInput->FlushLine();
        }
         else
        {
            // Something reasonable so return it
            break;
        }
    }
    return eToken;
}


//
// FUNCTION/METHOD NAME: GetName
//
// DESCRIPTION:
//
//  These functions will parse names and namespace/name combinations from the
//  input stream.. It will make sure that the token is not larger than the
//  max symbol name or a keyword.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: strName is filled in with the name token.
//          strNameSpace is filled with the namespace token
//
//  RETURN: None
//
tCIDLib::TVoid  TParseInfo::GetName(TString& strName)
{
    // Get the next token
    TParseInfo::eTokens eToken = eNextToken(strName, eTokenMode_UnknownOk);

    // If the token was not Unknown, then name is illegal
    if (eToken != TParseInfo::eToken_Unknown)
        ThrowErr(kTracerErrs::errcParse_ReservedToken, strName);

    if (strName.c4Length() > __c4MaxNameLen)
    {
        ThrowErr
        (
            kTracerErrs::errcParse_NameTooLong
            , strName
            , TCardinal(__c4MaxNameLen)
        );
    }
}

tCIDLib::TVoid
TParseInfo::GetName(TString& strName, TString&    strNameSpace)
{
    //
    //  Get the first token, which might be the name or namespace, but guess
    //  that it will be a name for now.
    //
    GetName(strName);

    //
    //  If the next token is not a double colon, then its gotta be relative
    //  to the current namespace. So we just put the token back, set the name
    //  space to the current one, and return.
    //
    TString  strToken(kCIDLib::pszEmptyZStr, 128);
    TParseInfo::eTokens eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    if (eToken != TParseInfo::eToken_DoubleColon)
    {
        UnGet(strToken);
        strNameSpace = __strCurNameSpace;
        return;
    }

    // We guessed wrong, move the name to the name space and get the name
    strNameSpace = strName;
    GetName(strName);
}


//
// FUNCTION/METHOD NAME: LookUpColor
//
// DESCRIPTION:
//
//  This function will look up the passed name in the list of defined
//  colors. If found, it will be copied to the target color object. If
//  not, then an exception will be thrown.
// ---------------------------------------
//   INPUT: strName is the name to look up
//          strNameSpace is the name space to which the color belongs.
//
//  OUTPUT: frgbTarget is the color object to fill in, if the color name
//              is found.
//
//  RETURN: None
//
tCIDLib::TVoid 
TParseInfo::LookUpColor(const   TString& strName
                        , const TString& strNameSpace
                        ,       TFRGBClr&   frgbTarget)
{
    TFRGBClr* const pfrgbTmp = pfrgbFind(strName, strNameSpace);

    if (!pfrgbTmp)
    {
        ThrowErr
        (
            kTracerErrs::errcParse_ColorNotFound
            , strName
            , strNameSpace
            , strName
        );
    }

    // Copy this color to the target
    frgbTarget = *pfrgbTmp;
}


//
// FUNCTION/METHOD NAME: ParseBoolean
//
// DESCRIPTION:
//
//  This method will parse the next token as a boolean.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: bTarget is filled in with the result
//
//  RETURN: None
//
tCIDLib::TVoid  TParseInfo::ParseBoolean(tCIDLib::TBoolean& bTarget)
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    TParseInfo::eTokens eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    // It should not match any keyword
    if (eToken != TParseInfo::eToken_Unknown)
        ThrowErr(kTracerErrs::errcParse_NotAllowedHere, strToken);

    // Upper case it for easy comparison
    strToken.ToUpper();

    if (strToken == L"TRUE")
        bTarget = kCIDLib::True;
     else if (strToken == "FALSE")
        bTarget = kCIDLib::False;
     else
        ThrowErr(kTracerErrs::errcParse_ExpectedBool, strToken);
}


//
// FUNCTION/METHOD NAME: ParseCard
//
// DESCRIPTION:
//
//  This function will parse a cardinal value and place the result
//  into c4Target.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: c4Target is filled in with the value.
//
//  RETURN: None
//
tCIDLib::TVoid  TParseInfo::ParseCard(tCIDLib::TCard4& c4Target)
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    TParseInfo::eTokens eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    // It should not match any keyword
    if (eToken != TParseInfo::eToken_Unknown)
        ThrowErr(kTracerErrs::errcParse_NotAllowedHere, strToken);

    // Try to convert the string to a value
    tCIDLib::TBoolean  bErr = kCIDLib::False;
    try
    {
        c4Target = strToken.c4Val(tCIDLib::ERadix_Dec);
    }

    catch (...)
    {
        ThrowErr(kTracerErrs::errcParse_BadFloat, strToken);
    }
}


//
// FUNCTION/METHOD NAME: ParseColor
//
// DESCRIPTION:
//
//  This method will parse a color. There are two scenarios here. If there
//  is one token, then the closed paren, its assumed to be the name of a
//  previously defined color. Otherwise, its assumed to the 3 or 4 floats
//  of an inline color definition.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: frgbTarget is a color object to fill in
//
//  RETURN: None
//
tCIDLib::TVoid  TParseInfo::ParseColor(TFRGBClr& frgbTarget)
{
    tCIDLib::TBoolean   bInPlace = kCIDLib::False;
    TParseInfo::eTokens eToken;
    TString             strToken(kCIDLib::pszEmptyZStr, 128);

    //
    //  Get the next token. Force the period char out of the normal list of
    //  syntax chars, in case we get a number.
    //
    eToken = eNextToken(strToken, eTokenMode_UnknownOk, ".");

    //
    //  First check for the obvious stuff. If we get here and the first
    //  parm is Color, the its an 'in place' color to a method that takes
    //  a color
    //
    if (eToken == eToken_Color)
    {
        bInPlace = kCIDLib::True;
        CheckOpenParen();
    }
     else
    {
        //
        //  Ok, so its either a named color or a set of color values.
        //  So look for the double colon. If not found it cannot be
        //  a named color.
        //
        TString  strToken2(kCIDLib::pszEmptyZStr, 128);
        eToken = eNextToken(strToken2, eTokenMode_UnknownOk);

        UnGet(strToken2);
        UnGet(strToken);

        if (eToken == eToken_DoubleColon)
        {
            TString   strName;
            TString   strNameSpace;
            GetName(strName, strNameSpace);

            TFRGBClr* const pfrgbTmp = pfrgbFind(strName, strNameSpace);

            if (pfrgbTmp)
            {
                frgbTarget = *pfrgbTmp;
                return;
            }
        }
    }

    tCIDLib::TFloat8    f8Tmp;

    ParseFloat(f8Tmp);
    frgbTarget.f8Red(f8Tmp);
    CheckComma();

    ParseFloat(f8Tmp);
    frgbTarget.f8Green(f8Tmp);
    CheckComma();

    ParseFloat(f8Tmp);
    frgbTarget.f8Blue(f8Tmp);

    //
    //  Get the next token. It has to be either a comma before the alpha
    //  or the closing paren.
    //
    eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    // It has to be a comma after the blue
    if (eToken == TParseInfo::eToken_Comma)
    {
        ParseFloat(f8Tmp);
        frgbTarget.f8Alpha(f8Tmp);
    }
     else
    {
        UnGet(strToken);
    }

    // If it was an 'in place' color, gotta have the close paren
    if (bInPlace)
        CheckCloseParen();
}


//
// FUNCTION/METHOD NAME: ParseColorMethod
//
// DESCRIPTION:
//
//  This is called to parse a method that is called against a color
//  object.
// ---------------------------------------
//   INPUT: pfrgbTarget is a pointer to the target color object against
//              which the method was called.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid 
TParseInfo::ParseColorMethod(TFRGBClr* const pfrgbTarget)
{
    // <TBD> Parse the damn thing
}


//
// FUNCTION/METHOD NAME: ParseFile
//
// DESCRIPTION:
//
//  This method will call the recursive parser (which will parse the file
//  and fill in the list of defined objects), and then add the defined objects
//  to the view object. The template objects, textures, and texture templates are
//  just flushed out.
// ---------------------------------------
//   INPUT: strFile is the initial file to start parsing
//          strIncludePath is the include path for included files.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid 
TParseInfo::ParseFile(  const   TString& strFile
                        , const TString& strIncludePath)
{
    //
    //  Load up the intrinsic name spaces. This is how we make the built
    //  in objects, textures, and lights available to the user.
    //
    __prtlstObjects->Add(new TRTBox, L"Box", L"BaseShapes");
    __prtlstObjects->Add(new TLightSource, L"Light", L"BaseShapes");
    __prtlstObjects->Add(new TRTMetaObj, L"MetaObj", L"BaseShapes");
    __prtlstObjects->Add(new TRTQuadric, L"Quadric", L"BaseShapes");
    __prtlstObjects->Add(new TRTPlane, L"Plane", L"BaseShapes");
    __prtlstObjects->Add(new TRTSolidGeom, L"SolidGeom", L"BaseShapes");
    __prtlstObjects->Add(new TRTSphere, L"Sphere", L"BaseShapes");

    __prtlstTextures->Add(new TTxtrAgate, L"Agate", L"BaseTextures");
    __prtlstTextures->Add(new TTxtrBanded, L"Banded", L"BaseTextures");
    __prtlstTextures->Add(new TTxtrBumps, L"Bumps", L"BaseTextures");
    __prtlstTextures->Add(new TTxtrChecker, L"Checkered", L"BaseTextures");
    __prtlstTextures->Add(new TTxtrDents, L"Dents", L"BaseTextures");
    __prtlstTextures->Add(new TTxtrGranite, L"Granite", L"BaseTextures");
    __prtlstTextures->Add(new TTxtrMarble, L"Marble", L"BaseTextures");
    __prtlstTextures->Add(new TTxtrSpots, L"Spots", L"BaseTextures");
    __prtlstTextures->Add(new TTxtrStripe, L"Striped", L"BaseTextures");
    __prtlstTextures->Add(new TRTTexture, L"Surface", L"BaseTextures");
    __prtlstTextures->Add(new TTxtrWaves, L"Waves", L"BaseTextures");
    __prtlstTextures->Add(new TTxtrWood, L"Wood", L"BaseTextures");
    __prtlstTextures->Add(new TTxtrWrinkles, L"Wrinkles", L"BaseTextures");

    __prtlstVectors->Add(new T3DVector(0, -1, 0), L"Down", L"BaseVectors");
    __prtlstVectors->Add(new T3DVector(-1, 0, 0), L"Left", L"BaseVectors");
    __prtlstVectors->Add(new T3DVector(1, 0, 0), L"Right", L"BaseVectors");
    __prtlstVectors->Add(new T3DVector(0, 1, 0), L"Up", L"BaseVectors");
    __prtlstVectors->Add(new T3DVector(0, 0, 1), L"Forward", L"BaseVectors");
    __prtlstVectors->Add(new T3DVector(0, 0, -1), L"Backward", L"BaseVectors");

    // Do the recursive parsing
    __ParseFile(strFile, strIncludePath);
}


//
// FUNCTION/METHOD NAME: ParseFloat
//
// DESCRIPTION:
//
//  This function will parse a floating point value and place the result
//  into f8Target.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: f8Target is filled in with the value.
//
//  RETURN: None
//
tCIDLib::TVoid  TParseInfo::ParseFloat(tCIDLib::TFloat8& f8Target)
{
    TString  strToken(kCIDLib::pszEmptyZStr, 128);

    //
    //  Get the next token. In this case, force the period character out
    //  of the normal list of syntax chars.
    //
    TParseInfo::eTokens eToken = eNextToken
    (
        strToken
        , eTokenMode_UnknownOk
        , L"."
    );

    // It should not match any keyword
    if (eToken != TParseInfo::eToken_Unknown)
        ThrowErr(kTracerErrs::errcParse_NotAllowedHere, strToken);

    // Try to convert the string to a value
    try
    {
        f8Target = strToken.f8Val();
    }

    catch (...)
    {
        ThrowErr(kTracerErrs::errcParse_BadFloat, strToken);
    }
}


//
// FUNCTION/METHOD NAME: ParseMethod
//
// DESCRIPTION:
//
//  This method will parse a method on the given object. It is assumed that
//  the caller parsed the name and namespace, so the next token should be the
//  period, followed by the method and its parms. We have to find out what
//  kind of object it is, then call the appropriate parsing method.
// ---------------------------------------
//   INPUT: strName is the of the object
//          strNameSpace is the name space to which it belongs.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid 
TParseInfo::ParseMethod(const   TString& strName
                        , const TString& strNameSpace)
{
    // Look first in the object list
    TRTObject* const prtobjParse = prtobjFind(strName, strNameSpace);
    if (prtobjParse)
    {
        prtobjParse->ParseMethod(*this);
        return;
    }

    // Ok, try a texture
    TRTTexture* const ptxtrParse = ptxtrFind(strName, strNameSpace);
    if (ptxtrParse)
    {
        ptxtrParse->ParseMethod(*this);
        return;
    }

    // All right, wrong again, check for a color
    TFRGBClr* const pfrgbParse = pfrgbFind(strName, strNameSpace);
    if (pfrgbParse)
    {
        ParseColorMethod(pfrgbParse);
        return;
    }

    // One last chance, see if its a vector
    T3DVector* const pvecParse = pvecFind(strName, strNameSpace);
    if (pvecParse)
    {
        ParseVectorMethod(pvecParse);
        return;
    }

    ThrowErr
    (
        kTracerErrs::errcParse_ObjectNotFound
        , strName
        , strNameSpace
        , strName
    );
}


//
// FUNCTION/METHOD NAME: ParseVector
//
// DESCRIPTION:
//
//  This method will parse a vector and put the results into vecTarget. If
//  an error occurs, it throws an exception.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: vecTarget is filled in with the vector info.
//
//  RETURN: None
//
tCIDLib::TVoid  TParseInfo::ParseVector(T3DVector& vecTarget)
{
    tCIDLib::TBoolean   bInPlace = kCIDLib::False;
    TParseInfo::eTokens eToken;
    tCIDLib::TFloat8    f8X, f8Y, f8Z;
    TString             strToken(kCIDLib::pszEmptyZStr, 128);

    //
    //  Get the next token, and see if it starts with Vector. If so, then
    //  its an 'in place' vector definition. If not, then try to look up
    //  the token as a named vector. If not found, assume it is just three
    //  float values (as parameters to some method that takes it that way.)
    //
    eToken = eNextToken(strToken, eTokenMode_UnknownOk, L".");

    if (eToken == eToken_Vector)
    {
        bInPlace = kCIDLib::True;
        CheckOpenParen();
    }
     else
    {
        //
        //  Ok, so its either a named vector or a set of vector values.
        //  So get the next token. If its a comma, then its a set of
        //  values, else its a named vector.
        //
        TString  strToken2(kCIDLib::pszEmptyZStr, 128);
        eToken = eNextToken(strToken2, eTokenMode_UnknownOk);

        // Either way, put both of them back
        UnGet(strToken2);
        UnGet(strToken);

        if (eToken != eToken_Comma)
        {
            TString   strName;
            TString   strNameSpace;
            GetName(strName, strNameSpace);

            T3DVector* const pvecTmp = pvecFind(strName, strNameSpace);

            if (!pvecTmp)
            {
                ThrowErr
                (
                    kTracerErrs::errcParse_VectorNotFound
                    , strName
                    , strNameSpace
                    , strName
                );
            }

            vecTarget = *pvecTmp;
            return;
        }    
    }

    // Get the x component
    ParseFloat(f8X);
    CheckComma();

    // Get the y component
    ParseFloat(f8Y);
    CheckComma();

    // Get the z component
    ParseFloat(f8Z);

    // If it was an 'in place' vector, gotta have the close paren
    if (bInPlace)
        CheckCloseParen();

    vecTarget.Set(f8X, f8Y, f8Z);
}


//
// FUNCTION/METHOD NAME: ParseVectorMethod
//
// DESCRIPTION:
//
//  This is called to parse a method that is called against a vector object.
// ---------------------------------------
//   INPUT: pvecTarget is a pointer to the target vector object against
//              which the method was called.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid 
TParseInfo::ParseVectorMethod(T3DVector* const pvecTarget)
{
    // <TBD> Parse the damn thing
}


//
// FUNCTION/METHOD NAME: pfrgbFind
//                       prtobjFind
//                       pvecFind
//
// DESCRIPTION:
//
//  This method will look up the passed name in the list of named scene
//  elements.
// ---------------------------------------
//   INPUT: strName is the name of the scene element to look for.
//          strNameSpace is the name of the namespace in which it lives.
//
//  OUTPUT: None
//
//  RETURN: A pointer to the object if found, else 0.
//
TFRGBClr*  TParseInfo::pfrgbFind(const   TString& strName
                                        , const TString& strNameSpace)
{
    return __prtlstColors->pobjFind(strName, strNameSpace);
}

TRTObject* TParseInfo::prtobjFind(  const   TString& strName
                                    , const TString& strNameSpace)
{
    return __prtlstObjects->pobjFind(strName, strNameSpace);
}

T3DVector* TParseInfo::pvecFind( const   TString& strName
                                , const TString& strNameSpace)
{
    return __prtlstVectors->pobjFind(strName, strNameSpace);
}


//
// FUNCTION/METHOD NAME: prtobjParseObject
//
// DESCRIPTION:
//
//  This method will parse an object from the input stream. It is assumed
//  that the Object keyword has just been parsed and we have to take it from
//  there.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: strName is filled with the name of the object
//
//  RETURN: A pointer to the new object
//
TRTObject*  TParseInfo::prtobjParseObject(TString& strName)
{
    eTokens     eToken;
    TRTObject*  prtobjNew = 0;
    TString     strNameSpace(kCIDLib::pszEmptyZStr, 128);
    TString     strToken(kCIDLib::pszEmptyZStr, 128);

    //
    //  The next token must the name of the source object from which this
    //  one is derived.
    //
    GetName(strName, strNameSpace);

    // See if this guy exists
    TRTObject* const prtobjTmp = prtobjFind(strName, strNameSpace);

    if (!prtobjTmp)
    {
        ThrowErr
        (
            kTracerErrs::errcParse_ObjectNotFound
            , strName
            , strNameSpace
            , strName
        );
    }

    // Duplicate the object
    prtobjNew = ::pDupObject<TRTObject>(*prtobjTmp);

    //
    //  The next thing will either be the name of the object or the
    //  opening bracket of its initialization block
    //
    eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    if ((eToken == eToken_SemiColon) || (eToken == eToken_CloseParen))
    {
        UnGet(strToken);

        // There was no name so clean it out
        strName.Clear();

        // Replicate the object and return it
        return prtobjNew;
    }

    //
    //  If not an opening brace it must be the name
    //
    if (eToken != eToken_OpenBrace)
    {
        UnGet(strToken);
        GetName(strName);

        eToken = eNextToken(strToken, eTokenMode_UnknownOk);
        if (eToken != eToken_OpenBrace)
        {
            UnGet(strToken);
            return prtobjNew;
        }
    }

    //
    //  Loop until we get the terminating brace, parsing methods. That
    //  is all that can be done in here.
    //
    while (1)
    {
        eToken = eNextToken(strToken, eTokenMode_UnknownOk);

        if (eToken == eToken_CloseBrace)
            break;

        // Unget the token and parse it as a method
        UnGet(strToken);
        prtobjNew->ParseMethod(*this);

        // Has to be a semi-colon at the end
        CheckSemiColon();
    }
    return prtobjNew;
}


//
// FUNCTION/METHOD NAME: ptxtrFind
//
// DESCRIPTION:
//
//  This method will look up the passed name as a texture.
// ---------------------------------------
//   INPUT: strName is the name of the object to find
//          strNameSpace is name of the namespace to which it belongs
//
//  OUTPUT: None
//
//  RETURN: 0 if not found, else a pointer to the texture object.
//
TRTTexture*  TParseInfo::ptxtrFind( const   TString& strName
                                    , const TString& strNameSpace)
{
    return __prtlstTextures->pobjFind(strName, strNameSpace);
}


//
// FUNCTION/METHOD NAME: ptxtrParseTexture
//
// DESCRIPTION:
//
//  This method will parse a texture and return the new object. It is
//  assumed that the Texture keyword has already been parsed.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: strName is filled with the name
//          strNameSpace is filled with the name space
//
//  RETURN: A pointer to the new texture object.
//
TRTTexture*  TParseInfo::ptxtrParseTexture(TString& strName)
{
    eTokens     eToken;
    TRTTexture* ptxtrNew;
    TString     strNameSpace;
    TString     strToken(kCIDLib::pszEmptyZStr, 128);

    //
    //  The next token must the name of the source object from which this
    //  one is derived.
    //
    GetName(strName, strNameSpace);

    // See if this guy exists
    TRTTexture* const ptxtrTmp = ptxtrFind(strName, strNameSpace);

    if (!ptxtrTmp)
    {
        ThrowErr
        (
            kTracerErrs::errcParse_TextureNotFound
            , strName
            , strNameSpace
            , strName
        );
    }

    // Replicate the new object
    ptxtrNew = ::pDupObject<TRTTexture>(*ptxtrTmp);

    //
    //  The next thing will either be the name of the object or the
    //  opening bracket of its initialization block
    //
    eToken = eNextToken(strToken, eTokenMode_UnknownOk);

    if ((eToken == eToken_SemiColon) || (eToken == eToken_CloseParen))
    {
        UnGet(strToken);

        // There was no name so clean it out
        strName.Clear();

        return ptxtrNew;
    }

    //
    //  If not an opening brace it must be the name
    //
    if (eToken != eToken_OpenBrace)
    {
        UnGet(strToken);
        GetName(strName);

        eToken = eNextToken(strToken, eTokenMode_UnknownOk);
        if (eToken != eToken_OpenBrace)
        {
            UnGet(strToken);
            return ptxtrNew;
        }
    }

    //
    //  Loop until we get the terminating brace, parsing methods. That
    //  is all that can be done in here.
    //
    while (1)
    {
        eToken = eNextToken(strToken, eTokenMode_UnknownOk);

        if (eToken == eToken_CloseBrace)
            break;

        // Unget the token and parse it as a method
        UnGet(strToken);
        ptxtrNew->ParseMethod(*this);

        // Has to be a semi-colon at the end
        CheckSemiColon();
    }
    return ptxtrNew;
}


//
// FUNCTION/METHOD NAME: ThrowErr
//
// DESCRIPTION:
//
//  This method will create an TRTParseErr object and throw it.
// ---------------------------------------
//   INPUT: c4Err is the error that occured. It should be backed by a string
//              in the resource file.
//          strToken text that cause the error.
//          objToken1 and objToken2 are the optional replacement tokens.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid 
TParseInfo::ThrowErr(   const   tCIDLib::TCard4 c4Err
                        , const TString&        strToken
                        , const MFormattable&   fmtblToken1
                        , const MFormattable&   fmtblToken2)
{
    // Load up the message
    TString  strMsg(c4Err, facCIDTracer, 256, fmtblToken1, fmtblToken2);

    // Create the error object and throw it
    TRTParseErr  rterrThrow
    (
        __pprsrInput->c4CurLine()
        , c4Err
        , __strCurFile
        , strToken
        , strMsg
    );
    throw rterrThrow;
}


//
// FUNCTION/METHOD NAME: UnGet
//
// DESCRIPTION:
//
//  Just passes the call on to the input stream parser, which will unget the
//  passed token.
// ---------------------------------------
//   INPUT: strToUnGet is the token to unget.
//          pszToUnGet is the token as a raw asci string.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TParseInfo::UnGet(const TString& strToUnGet)
{
    __pprsrInput->UnGetToken(strToUnGet);
}

tCIDLib::TVoid TParseInfo::UnGet(const tCIDLib::Tch* const pszToUnGet)
{
    __pprsrInput->UnGetToken(pszToUnGet);
}


// -----------------------------------------------------------------------------
//  TParseInfo: Private, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __ParseFile
//
// DESCRIPTION:
//
//  This method will do the whole file parse and either successfully
//  parse the file or throw an exception. The highest level tokens are
//  handled here, with calls made to the appropriate parties to handle
//  each defined object, color, etc... Recursive processing of include
//  files is handled here as well.
// ---------------------------------------
//   INPUT: strFileName is the name of the file to parse.
//          strIncludePath is the search path for include files.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid 
TParseInfo::__ParseFile(const   TString& strFileName
                        , const TString& strIncludePath)
{
    //
    //  The first thing to do is to see whether the file exists. If not,
    //  then throw an exception.
    //
    TFileSys    flsysParse;
    TFindBuf    fndbFile;
    if (!flsysParse.bExistsInPath
    (
        strFileName
        , strIncludePath
        , tCIDLib::EFileAttr_All
        , tCIDLib::EFileAttr_None
        , fndbFile))
    {
        //
        //  Have to throw this one manually, instead of via ThrowErr(),
        //  because we have not set set up the file name and stream
        //  parser members.
        //
        TRTParseErr rterrFreakOut
        (
            0
            , kTracerErrs::errcParse_FileNotFound
            , strFileName
            , kCIDLib::pszEmptyZStr
            , L"Could not find file"
        );
        throw rterrFreakOut;
    }

    // Save the old input parser and input file name
    TStreamParser*  pprsrOld = __pprsrInput;
    TString         strNameOld(__strCurFile);

    // Put next block in a try block so we can replace the parser on an err
    try
    {
        // Store away the file name, saving the current one
        __strCurFile = fndbFile.pathFileName();

        //
        //  Create a new parser for this level. Assume UNICode format for
        //  now. We will check below for a UNICode marker and change the
        //  format if required.
        //
        TTextFileStream* pstrmNew = new TTextFileStream
        (
            __strCurFile
            , tCIDLib::ETextFmt_UNICode
            , tCIDLib::EAccess_Read
            , tCIDLib::ECreateAct_OpenIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_None
        );
        TJanitor<TTextFileStream> janStream(pstrmNew);

        //
        //  We need to first see if this file is in UNICode or ASCII
        //  text format. We read in the first character and see if it
        //  is the UNICode marker. If it is, we set the text stream
        //  input format accordingly. Otherwise, we reset the stream.
        //
        tCIDLib::Tch chMarker;
        chMarker = pstrmNew->chGet();

        if ((chMarker != kCIDLib::chMarker)
        &&  (chMarker != kCIDLib::chSwappedMarker))
        {
            pstrmNew->eTextFormat(tCIDLib::ETextFmt_ASCII);
            pstrmNew->Reset();
        }

        //
        //  Now create the stream parser for this level and give it the
        //  text stream. We tell it to adopt the stream so we don't worry
        //  about it anymore.
        //
        __pprsrInput = new TStreamParser(pstrmNew, tCIDLib::EAdoptOpt_Adopt);

        // Orphan the stream from the janitor
        janStream.Orphan();

        // Set up the syntax tokens
        for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4SyntaxCount; c4Ind++)
            __pprsrInput->AddSyntaxToken(__aszTokens[c4Ind]);

        //
        //  Enter the main parsing loop. Here we get the next token, which
        //  should always be a comment or keyword at this level. Once we see
        //  what the keyword is, that tells us what local function we have to
        //  call to parse that type of thingy.
        //
        tCIDLib::TBoolean   bLoop   = kCIDLib::True;
        eTokens             eToken;
        TString             strName;
        TString             strNameSpace;
        TString             strToken(kCIDLib::pszEmptyZStr, 128);
        while (bLoop)
        {
            // Get the next token
            eToken = eNextToken(strToken, eTokenMode_UnknownOrEndOk);

            //
            //  According to the token do the right thing. If it is not a
            //  valid token for this level of the parse, then abort.
            //
            if (eToken == eToken_Color)
            {
                // The next parm must be the name of the color
                GetName(strName, strNameSpace);

                // Make sure that it does not exist
                if (pfrgbFind(strName, strNameSpace))
                {
                    ThrowErr
                    (
                        kTracerErrs::errcParse_AlreadyExists
                        , strName
                        , strNameSpace
                        , strName
                    );
                }

                // Next must be an opening paren
                CheckOpenParen();

                TFRGBClr    frgbTmp;

                // Parse into the temp color
                ParseColor(frgbTmp);

                // Has to be a closing paren and semicolon
                CheckCloseParen();
                CheckSemiColon();

                // Ok it went well, so we can save it away
                __prtlstColors->Add
                (
                    new TFRGBClr(frgbTmp)
                    , strName
                    , strNameSpace
                );
            }
             else if (eToken == eToken_Import)
            {
                //
                //  Next should be the name of the name space to import, and
                //  a semicolon.
                //
                GetName(strName);
                CheckSemiColon();

                //
                //  If we have already parsed this namespace, then don't
                //  do it again.
                //
                if (!bNameSpaceImported(strName))
                {
                    // Add the name to the list
                    __pbagImports->Add(strName);

                    // And parse this name space file
                    strToken = strName;
                    strToken.Append(".NameSpace");

                    //
                    //  Parse the file recursively. We need to save away the
                    //  current name space and put it back afterwards.
                    //
                    TString   strSaveSpace = __strCurNameSpace;
                    __strCurNameSpace = strName;
                    __ParseFile(strToken, strIncludePath);
                    __strCurNameSpace = strSaveSpace;
                }
            }
             else if (eToken == eToken_Object)
            {
                TRTObject* prtobjNew = prtobjParseObject(strName);
                __prtlstObjects->Add(prtobjNew, strName, __strCurNameSpace);
            }
             else if (eToken == eToken_Texture)
            {
                TRTTexture* const ptxtrNew = ptxtrParseTexture(strName);
                __prtlstTextures->Add(ptxtrNew, strName, __strCurNameSpace);
            }
             else if (eToken == eToken_Scene)
            {
                // The next thing must be the period
                CheckPeriod();

                // Ok, parse the method
                __viewTarget.ParseMethod(*this);

                // Has to be a trailing semicolon
                CheckSemiColon();
            }
             else if (eToken == eToken_End)
            {
                break;
            }
             else
            {
                // Must be a method called against an object
                UnGet(strToken);
                GetName(strName, strNameSpace);
                CheckPeriod();
                ParseMethod(strName, strNameSpace);
                CheckSemiColon();
            }
        }
    }

    catch(...)
    {
        // Delete the current input parser
        delete __pprsrInput;

        // Put back the old parser
        __pprsrInput = pprsrOld;

        // Put the old file name back
        __strCurFile = strNameOld;

        // And rethrow the exception
        throw;
    }

    // Delete the current input parser
    delete __pprsrInput;

    // Put back the old parser
    __pprsrInput = pprsrOld;

    // Put the old file name back
    __strCurFile = strNameOld;
}
