//
// NAME: CIDTracer_Object.Hpp
//
// DESCRIPTION: 
//
//  This module implements the TRTObject, and TRTGeometricObj classes.
//  These are the basis of all of the types of objects that go into a scene.
//  Only derivatives of TRTGeometricObj are actually placed into an
//  intersection list. Other derivatives are just for bounding, CSG, and
//  composite object control, they are not visible in the scene themselves.
//  Therefore they are derived straight from TRTObject.
//
//  TRTObject is the basic ray tracing object. It provides the virtual API
//  that all other scene components derive from, but it cannot be instantiated
//  directly. All derivatives of TRTObject must be test whether a ray
//  intersects them (or possibly any of their child objects.) It mixes in
//  MDuplicable and MStreamable so that all of its derivatives will
//  have to provide these capabilities.
//
//  TRTGeometricObj is the basis of all of the geometry classes, which are the
//  actual visible objects in a scene. All derivatives of TRTGeometricObj must
//  be able to calculate a surface normal at any point on their surface. They
//  must also be able to calculate their surface color at any point, which is
//  determined by their textures.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 03/05/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//
//  1)  Because of the nature of the ray tracing framework, all of the
//      derivatives of RTObject and TRTGeometricObj should support dynamic
//      type creation.
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: TRTObject
// PREFIX: rtobj
// -----------------------------------------------------------------------------
class CIDTRACEREXP TRTObject :

    public TObject, public MStreamable, public MDuplicable, public MFormattable
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and Destructors
        // ---------------------------------------------------------------------
        TRTObject();

        TRTObject
        (
            const   TRTObject&              rtobjToCopy
        );

        ~TRTObject();


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TRTObject& operator=
        (
            const   TRTObject&              rtobjToAssign
        );

        // ---------------------------------------------------------------------
        //  Public, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TVoid AppendTexture
        (
                    TRTTexture* const       ptxtrAppend
            , const TString&                strName = NUL_TString
            , const TString&                strNameSpace = NUL_TString
        ) = 0;

        virtual tCIDLib::TVoid Invert() = 0;

        virtual tCIDLib::TVoid ParseMethod
        (
                    TParseInfo&             prsrSrc
        );

        virtual tCIDLib::TVoid PrependTexture
        (
                    TRTTexture* const       ptxtrPrepend
            , const TString&                strName = NUL_TString
            , const TString&                strNameSpace = NUL_TString
        ) = 0;

        virtual tCIDLib::TVoid Transform
        (
            const   TTransform&             trnsToApply
        ) = 0;

        virtual tCIDLib::TVoid TransformTextures
        (
            const   TTransform&             trnsToApply
        ) = 0;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid AdoptBoundingObj
        (
                    TRTObject* const        prtobjNew
        );

        tCIDLib::TVoid AllIntersects
        (
            const   TLightRay&              lrayToTest
            ,       TIntersectArray&        intraIntersects
        )   const;

        tCIDLib::TBoolean bInBoundingObj
        (
            const   T3DVector&              vecToTest
        )   const;

        tCIDLib::TBoolean bPointInObj
        (
            const   T3DVector&              vecToTest
        )   const;

        tCIDLib::TBoolean bTestBoundingObj
        (
            const   TLightRay&              lrayToTest
        )   const;

        tCIDLib::TBoolean bTestIntersect
        (
            const   TLightRay&              lrayToTest
            ,       tCIDLib::TFloat8&       f8Distance1
            ,       tCIDLib::TFloat8&       f8Distance2
        )   const;

        tCIDLib::TBoolean bVisible() const;

        tCIDLib::TBoolean bVisible
        (
            const   tCIDLib::TBoolean       bNewState
        );

        tCIDLib::TVoid Rotate
        (
            const   T3DVector&              vecRotate
        );

        tCIDLib::TVoid Rotate
        (
                    tCIDLib::TFloat8        f8X
            ,       tCIDLib::TFloat8        f8Y
            ,       tCIDLib::TFloat8        f8Z
        );

        tCIDLib::TVoid Scale
        (
            const   T3DVector&              vecScale
        );

        tCIDLib::TVoid Scale
        (
                    tCIDLib::TFloat8        f8X
            ,       tCIDLib::TFloat8        f8Y
            ,       tCIDLib::TFloat8        f8Z
        );

        tCIDLib::TVoid Translate
        (
            const   T3DVector&              vecTrans
        );

        tCIDLib::TVoid Translate
        (
                    tCIDLib::TFloat8        f8X
            ,       tCIDLib::TFloat8        f8Y
            ,       tCIDLib::TFloat8        f8Z
        );


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TVoid _AllIntersects
        (
            const   TLightRay&              lrayToTest
            ,       TIntersectArray&        intraIntersects
        )   const = 0;

        virtual tCIDLib::TBoolean _bPointInObj
        (
            const   T3DVector&              vecToTest
        )   const = 0;

        virtual tCIDLib::TBoolean _bTestIntersect
        (
            const   TLightRay&              lrayToTest
            ,       tCIDLib::TFloat8&       f8Distance1
            ,       tCIDLib::TFloat8&       f8Distance2
        )   const = 0;

        virtual tCIDLib::TVoid _Rotate
        (
            const   T3DVector&              vecRotate
            , const TTransform&             trnsRotate
        ) = 0;

        virtual tCIDLib::TVoid _Scale
        (
            const   T3DVector&              vecScale
            , const TTransform&             trnsScale
        ) = 0;

        virtual tCIDLib::TVoid _Translate
        (
            const   T3DVector&              vecTrans
            , const TTransform&             trnsTranslate
        ) = 0;


        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private :
        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __bVisible
        //      This flag indicates whether the object is visible or not.
        //      TRTObject::bTestIntersect() will check this flag and the
        //      bounding object. All derived objects should call it first.
        //
        //  __prtobjBounding
        //      This is a pointer to a bounding object. Each object can
        //      have a bounding object that will be used to reduce the
        //      overhead of hit testing. Unless the user sets it, it will
        //      remain 0.
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean       __bVisible;
        TRTObject*              __prtobjBounding;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TRTObject,TObject)
};


// -----------------------------------------------------------------------------
//  Typedef our list of objects
// -----------------------------------------------------------------------------
typedef TRTListNode<TRTObject>      TRTObjNode;
typedef TRTList<TRTObject>          TRTObjList;


// -----------------------------------------------------------------------------
//  CLASS: TRTGeometricObj
// PREFIX: rtgeo
//
//  This class is the basis for all of the geometry classes, which are the
//  visible shapes in scenes.
// -----------------------------------------------------------------------------
class CIDTRACEREXP TRTGeometricObj : public TRTObject
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and Destructors
        // ---------------------------------------------------------------------
        TRTGeometricObj();

        TRTGeometricObj
        (
            const   TRTGeometricObj&        rtgeoToCopy
        );

        ~TRTGeometricObj();


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TRTGeometricObj& operator=
        (
            const   TRTGeometricObj&        rtgeoToAssign
        );

        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid AppendTexture
        (
                    TRTTexture* const       ptxtrToAppend
            , const TString&                strName = NUL_TString
            , const TString&                strNameSpace = NUL_TString
        );

        tCIDLib::TVoid Invert();

        tCIDLib::TVoid ParseMethod
        (
                    TParseInfo&             prsrSrc
        );

        tCIDLib::TVoid PrependTexture
        (
                    TRTTexture* const       ptxtrToPrepend
            , const TString&                strName = NUL_TString
            , const TString&                strNameSpace = NUL_TString
        );

        tCIDLib::TVoid Transform
        (
            const   TTransform&             trnsToApply
        );

        tCIDLib::TVoid TransformTextures
        (
            const   TTransform&             trnsToApply
        );


        // ---------------------------------------------------------------------
        //  Public, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TVoid CalcSurfaceNormal
        (
            const   TIntersect* const       pintrSrc
            ,       T3DVector&              vecNormal
        )   const = 0;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bInverted() const;

        const TRTTxtrList& rtlstTextures() const;


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _Rotate
        (
            const   T3DVector&              vecRotate
            , const TTransform&             trnsRotate
        );

        tCIDLib::TVoid _Scale
        (
            const   T3DVector&              vecScale
            , const TTransform&             trnsScale
        );

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;

        tCIDLib::TVoid _Translate
        (
            const   T3DVector&              vecTrans
            , const TTransform&             trnsTranslate
        );


    private :
        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __bInverted
        //      This flag is used to indicate that an object is inverted.
        //      Inversion is important in CSG objects.
        //
        //  __prtlstTextures
        //      This is a list of the textures that are associated with this
        //      object.
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean       __bInverted;
        TRTTxtrList*            __prtlstTextures;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TRTGeometricObj,TRTObject)
};

#pragma pack(pop)



// -----------------------------------------------------------------------------
//  TRTObject: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean TRTObject::bVisible() const
{
    return __bVisible;
}

inline tCIDLib::TBoolean
TRTObject::bVisible(const tCIDLib::TBoolean bNewState)
{
    __bVisible = bNewState;
    return __bVisible;
}

inline tCIDLib::TVoid
TRTObject::Rotate(  tCIDLib::TFloat8    f8X
                    , tCIDLib::TFloat8  f8Y
                    , tCIDLib::TFloat8  f8Z)
{
    Rotate(T3DVector(f8X, f8Y, f8Z));
}

inline tCIDLib::TVoid
TRTObject::Scale(tCIDLib::TFloat8   f8X
                , tCIDLib::TFloat8  f8Y
                , tCIDLib::TFloat8  f8Z)
{
    Scale(T3DVector(f8X, f8Y, f8Z));
}

inline tCIDLib::TVoid
TRTObject::Translate(   tCIDLib::TFloat8    f8X
                        , tCIDLib::TFloat8  f8Y
                        , tCIDLib::TFloat8  f8Z)
{
    Translate(T3DVector(f8X, f8Y, f8Z));
}



// -----------------------------------------------------------------------------
//  TRTGeometricObj: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean TRTGeometricObj::bInverted() const
{
    return __bInverted;
}

inline const TRTTxtrList& TRTGeometricObj::rtlstTextures() const
{
    return *__prtlstTextures;
}
