//
// NAME: CIDTracer_Object.Cpp
//
// DESCRIPTION:
//
//  This module implements the TRTObject, and TRTGeometricObj classes. These
//  are the most basic scene object classes of the ray tracing system.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 03/05/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our class' RTTI stuff
// -----------------------------------------------------------------------------
RTTIData(TRTObject,TObject)
RTTIData(TRTGeometricObj,TRTObject)


// -----------------------------------------------------------------------------
//  CLASS: TRTObject
// PREFIX: rtobj
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TRTObject:: Constructors and Destructors
// -----------------------------------------------------------------------------

TRTObject::TRTObject() :

    __bVisible(kCIDLib::True)
    , __prtobjBounding(0)
{
}

TRTObject::TRTObject(const TRTObject& rtobjSrc) :

    __bVisible(rtobjSrc.__bVisible)
    , __prtobjBounding(0)
{
    // If there is a bounding object, replicate it
    if (rtobjSrc.__prtobjBounding)
        __prtobjBounding = ::pDupObject<TRTObject>(*rtobjSrc.__prtobjBounding);
}

TRTObject::~TRTObject()
{
    delete __prtobjBounding;
}


// -----------------------------------------------------------------------------
//  TRTObject: Public operators
// -----------------------------------------------------------------------------

TRTObject& TRTObject::operator=(const TRTObject& rtobjSrc)
{
    if (this == &rtobjSrc)
        return *this;

    // Handle parent first
    TParent::operator=(rtobjSrc);

    // If there is a bounding object, then delete it
    if (rtobjSrc.__prtobjBounding)
    {
        delete __prtobjBounding;
        __prtobjBounding = 0;
    }

    // Copy the visible flag
    __bVisible = rtobjSrc.__bVisible;

    // If there is a bounding object, replicate it
    if (rtobjSrc.__prtobjBounding)
        __prtobjBounding = ::pDupObject<TRTObject>(*rtobjSrc.__prtobjBounding);

    return *this;
}


// -----------------------------------------------------------------------------
//  TRTObject: Public, virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TRTObject::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens eToken;
    TString             strName;
    TString             strNameSpace;
    TString             strToken(kCIDLib::pszEmptyZStr, 128);

    //
    //  Ok, none of our derived classes wants to handle this method, so
    //  we need to deal with it.
    //
    eToken = prsiSrc.eNextToken(strToken);

    // Must be followed by an open parens
    prsiSrc.CheckOpenParen();

    if ((eToken == TParseInfo::eToken_AppendTexture)
    ||  (eToken == TParseInfo::eToken_PrependTexture))
    {
        TRTTexture* const ptxtrNew = prsiSrc.ptxtrParseTexture(strName);

        if (eToken == TParseInfo::eToken_AppendTexture)
            AppendTexture(ptxtrNew, strName, prsiSrc.strCurNameSpace());
        else
            PrependTexture(ptxtrNew, strName, prsiSrc.strCurNameSpace());
    }
     else if (eToken == TParseInfo::eToken_Bounding)
    {
        // Parse in a new object
        TRTObject* const prtobjNew = prsiSrc.prtobjParseObject(strName);
        AdoptBoundingObj(prtobjNew);
    }
     else if (eToken == TParseInfo::eToken_Visibility)
    {
        tCIDLib::TBoolean  bTmp;
        prsiSrc.ParseBoolean(bTmp);
        bVisible(bTmp);
    }
     else if ((eToken == TParseInfo::eToken_Scale)
          ||  (eToken == TParseInfo::eToken_Rotate)
          ||  (eToken == TParseInfo::eToken_Translate))
    {
        T3DVector vecTmp;
        prsiSrc.ParseVector(vecTmp);
        vecTmp.ClipToDegrees();

        if (eToken == TParseInfo::eToken_Scale)
        {
            // Scaling must be uniform for spheres
            if (bIsDescendantOf(TRTSphere::clsThis))
            {
                if ((vecTmp.f8X() != vecTmp.f8Y())
                ||  (vecTmp.f8X() != vecTmp.f8Z()))
                {
                    prsiSrc.ThrowErr
                    (
                        kTracerErrs::errcSphere_NonUniformScale
                        , kCIDLib::pszEmptyZStr
                        , vecTmp
                    );
                }
            }
            Scale(vecTmp);
        }
         else if (eToken == TParseInfo::eToken_Rotate)
        {
            Rotate(vecTmp);
        }
         else if (eToken == TParseInfo::eToken_Translate)
        {
            Translate(vecTmp);
        }
    }
     else
    {
        //
        //  Freak out, we cannot handle this method. Some derived class
        //  has not done its job.
        //
        prsiSrc.ThrowErr(kTracerErrs::errcParse_UnknownMethod, strToken);
    }

    // Gotta be a close parens
    prsiSrc.CheckCloseParen();
}


tCIDLib::TVoid TRTObject::Transform(const TTransform& trnsApply)
{
    if (__prtobjBounding)
        __prtobjBounding->Transform(trnsApply);
}


// -----------------------------------------------------------------------------
//  TRTObject: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TRTObject::AdoptBoundingObj(TRTObject* const prtobjNew)
{
    // Delete any current object
    delete __prtobjBounding;
    __prtobjBounding = 0;

    // Save the new one
    __prtobjBounding = prtobjNew;
}


tCIDLib::TVoid
TRTObject::AllIntersects(   const   TLightRay&          lrayTest
                            ,       TIntersectArray&    intraHits) const
{
    // If there is a bounding object, then test it
    if (__prtobjBounding)
    {
        if (!bTestBoundingObj(lrayTest))
            return;
    }

    _AllIntersects(lrayTest, intraHits);
}


tCIDLib::TBoolean TRTObject::bInBoundingObj(const T3DVector& vecTest) const
{
    if (__prtobjBounding)
        return __prtobjBounding->bPointInObj(vecTest);

    return kCIDLib::True;
}

tCIDLib::TBoolean
TRTObject::bTestBoundingObj(const TLightRay& lrayTest) const
{
    if (__prtobjBounding)
    {
        tCIDLib::TFloat8    f8Dist1, f8Dist2;
        return __prtobjBounding->bTestIntersect
        (
            lrayTest
            , f8Dist1
            , f8Dist2
        );
    }
    return kCIDLib::True;
}


tCIDLib::TBoolean TRTObject::bPointInObj(const T3DVector& vecPoint) const
{
// <TBD> Figure out why this is commented out!!
//    if (!bInBoundingObj(vecPoint))
//        return kCIDLib::False;

    return _bPointInObj(vecPoint);
}


tCIDLib::TBoolean
TRTObject::bTestIntersect(  const   TLightRay&          lrayTest
                            ,       tCIDLib::TFloat8&   f8Dist1
                            ,       tCIDLib::TFloat8&   f8Dist2) const
{
    if (!__bVisible)
        return kCIDLib::False;

    if (__prtobjBounding)
    {
        //
        //  Test for intersection, just throwing away the distances since
        //  they don't matter anymore.
        //
        tCIDLib::TFloat8 f8Tmp1, f8Tmp2;
        if (!__prtobjBounding->bTestIntersect(lrayTest, f8Tmp1, f8Tmp2))
            return kCIDLib::False;
    }

    // Call the virtual version to run the hierarchy.
    return _bTestIntersect(lrayTest, f8Dist1, f8Dist2);
}


tCIDLib::TVoid TRTObject::Rotate(const T3DVector& vecRotate)
{
    // Get a rotation transform
    TTransform   trnsRotate;
    trnsRotate.Rotate(vecRotate);

    // Do the bounding object first
    if (__prtobjBounding)
        __prtobjBounding->Rotate(vecRotate);

    _Rotate(vecRotate, trnsRotate);
}

tCIDLib::TVoid TRTObject::Scale(const T3DVector& vecScale)
{
    // Get a scaling transform
    TTransform   trnsScale;
    trnsScale.Scale(vecScale);

    if (__prtobjBounding)
        __prtobjBounding->Scale(vecScale);

    _Scale(vecScale, trnsScale);
}

tCIDLib::TVoid TRTObject::Translate(const T3DVector& vecTrans)
{
    // Get a scaling transform
    TTransform   trnsTranslate;
    trnsTranslate.Translate(vecTrans);

    if (__prtobjBounding)
        __prtobjBounding->Translate(vecTrans);

    _Translate(vecTrans, trnsTranslate);
}

// -----------------------------------------------------------------------------
//  TRTObject: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TRTObject::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo   << L"Type: " << clsIsA()
                    << L", Visible: " << __bVisible
                    << L"Bound Obj: ";

    if (__prtobjBounding)
        strmToWriteTo << __prtobjBounding->clsIsA();
    else
        strmToWriteTo << L"None";
}


tCIDLib::TVoid TRTObject::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Free current bounding object if any
    if (__prtobjBounding)
    {
        delete __prtobjBounding;
        __prtobjBounding = 0;
    }
    
    // Pull out the visible flag
    strmToReadFrom >> __bVisible;

    // Pull out the 'bounding object present' flag
    tCIDLib::TCard1 c1GotBounding;
    strmToReadFrom >> c1GotBounding;

    // If we had one, then we gotta read it back in
    if (c1GotBounding)
        ::PolymorphicRead(__prtobjBounding, strmToReadFrom);
}

tCIDLib::TVoid TRTObject::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Write out the visible flag. Bools are written as 1 byte
    strmToWriteTo << __bVisible;

    // Write out a byte that indicates whether there is a bounding object
    tCIDLib::TCard1 c1Tmp = 0;
    if (__prtobjBounding)
        c1Tmp = 1;
    strmToWriteTo << c1Tmp;

    // Stream out the bounding object's class if we have one
    if (__prtobjBounding)
        ::PolymorphicWrite(__prtobjBounding, strmToWriteTo);
}



// -----------------------------------------------------------------------------
//  CLASS: TRTGeometricObj
// PREFIX: rtobj
// -----------------------------------------------------------------------------

TRTGeometricObj::TRTGeometricObj() :

    __bInverted(kCIDLib::False)
    , __prtlstTextures(0)
{
    __prtlstTextures = new TRTTxtrList;
}


TRTGeometricObj::TRTGeometricObj(const TRTGeometricObj& rtgeoToCopy) :

    TRTObject(rtgeoToCopy)
    , __bInverted(rtgeoToCopy.__bInverted)
    , __prtlstTextures(0)
{
    __prtlstTextures = new TRTTxtrList(*rtgeoToCopy.__prtlstTextures);
}


TRTGeometricObj::~TRTGeometricObj()
{
    // Flush the texture list and destroy it
    __prtlstTextures->Flush();
    delete __prtlstTextures;
}


// -----------------------------------------------------------------------------
//  TRTGeometricObj: Public operators
// -----------------------------------------------------------------------------

TRTGeometricObj& TRTGeometricObj::operator=(const TRTGeometricObj& rtgeoToAssign)
{
    // Copy our parent's stuff first
    TParent::operator=(rtgeoToAssign);

    // Copy over the inverted state
    __bInverted = rtgeoToAssign.__bInverted;

    // Flush this object's texture list
    __prtlstTextures->Flush();

    // Copy the source textures
    *__prtlstTextures = *rtgeoToAssign.__prtlstTextures;

    return *this;
}


// -----------------------------------------------------------------------------
//  TRTGeometricObj: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TRTGeometricObj::AppendTexture( TRTTexture* const   ptxtrToAppend
                                , const TString&  strName
                                , const TString&  strNameSpace)
{
    __prtlstTextures->Append(ptxtrToAppend, strName, strNameSpace);
}


tCIDLib::TVoid TRTGeometricObj::Invert()
{
    __bInverted = kCIDLib::True;
}


tCIDLib::TVoid TRTGeometricObj::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens     eToken;
    TString                 strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    // The next token must be the open paren
    prsiSrc.CheckOpenParen();

    // Deal with the valid methods at this level
    if (eToken == TParseInfo::eToken_Invert)
    {
        tCIDLib::TBoolean  bTmp;
        prsiSrc.ParseBoolean(bTmp);

        if (bTmp)
            Invert();
    }
     else
    {
        // Unget the token and pass it to our parent
        prsiSrc.UnGet("(");
        prsiSrc.UnGet(strToken);
        TParent::ParseMethod(prsiSrc);
        return;
    }

    // Has to end with a closing paren
    prsiSrc.CheckCloseParen();
}


tCIDLib::TVoid
TRTGeometricObj::PrependTexture(        TRTTexture* const   ptxtrToPrepend
                                , const TString&          strName
                                , const TString&          strNameSpace)
{
    __prtlstTextures->Add(ptxtrToPrepend, strName, strNameSpace);
}


tCIDLib::TVoid TRTGeometricObj::Transform(const TTransform& trnsApply)
{
    // Call the parent version
    TParent::Transform(trnsApply);

    TransformTextures(trnsApply);
}


tCIDLib::TVoid
TRTGeometricObj::TransformTextures(const TTransform& trnsApply)
{
    TRTTxtrNode* pnodeCur = __prtlstTextures->pnodeHead();
    while (pnodeCur)
    {
        pnodeCur->pobjData()->Transform(trnsApply);
        pnodeCur = pnodeCur->pnodeNext();
    }
}


// -----------------------------------------------------------------------------
//  TRTGeometricObj: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TRTGeometricObj::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Call our parent first
    TParent::_FormatTo(strmToWriteTo);
    strmToWriteTo   << L", Inverted: " << __bInverted
                    << L", Texture Count: "
                    << __prtlstTextures->c4NodeCount();
}


tCIDLib::TVoid
TRTGeometricObj::_Rotate(   const   T3DVector&  vecRotate
                            , const TTransform& trnsRotate)
{
    TransformTextures(trnsRotate);
}

tCIDLib::TVoid
TRTGeometricObj::_Scale(const T3DVector& vecScale, const TTransform& trnsScale)
{
    TransformTextures(trnsScale);
}

tCIDLib::TVoid
TRTGeometricObj::_Translate(const   T3DVector&   vecTrans
                            , const TTransform& trnsTranslate)
{
    TransformTextures(trnsTranslate);
}


tCIDLib::TVoid
TRTGeometricObj::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    //
    //  Flush our texture list, in the default delete mode which is to
    //  destroy them all.
    //
    __prtlstTextures->Flush();

    // Call our parent to let him stream in first
    TParent::_StreamFrom(strmToReadFrom);

    // Stream back in our inverted flag
    strmToReadFrom >> __bInverted;

    // See how many textures we wrote out
    tCIDLib::TCard4 c4Count;
    strmToReadFrom >> c4Count;

    //
    //  Now iterate the texture list and stream them in polymorphically.
    //  We add each new one to the list.
    //
    TRTTexture* ptxtrNew;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
    {
        ::PolymorphicRead(ptxtrNew, strmToReadFrom);
        __prtlstTextures->Append(ptxtrNew);
    }
}

tCIDLib::TVoid
TRTGeometricObj::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent first
    TParent::_StreamTo(strmToWriteTo);

    //
    //  Stream our our inverted flag. Note that the stream will save it as
    //  a single byte.
    //
    strmToWriteTo << __bInverted;

    // Write out how many textures we have
    strmToWriteTo << __prtlstTextures->c4NodeCount();

    // Now iterate the texture list and stream them polymorphically
    TRTTxtrNode* pnodeCur = __prtlstTextures->pnodeHead();
    while (pnodeCur)
        ::PolymorphicWrite(pnodeCur->pobjData(), strmToWriteTo);
}
