//
// NAME: CIDTracer_NoiseTextures.Cpp
//
// DESCRIPTION:
//
//  This module implements the TTxtrNoise class, which is the basis for all
//  of the noise based textures. It only has a couple of data members but
//  provides a couple of important virtual methods which derivatives
//  can override.
//
//  Also defined here are some simple derivatives of TTxtrNoise. TTxtrBanded
//  is an important one that is the basis for many small derivatives.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 05/15/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard RTTI macros
// -----------------------------------------------------------------------------
RTTIData2(TTxtrNoise,TRTTexture)
RTTIData2(TTxtrBumps,TTxtrNoise)
RTTIData2(TTxtrDents,TTxtrNoise)
RTTIData2(TTxtrWaves,TTxtrNoise)
RTTIData2(TTxtrWrinkles,TTxtrNoise)


// -----------------------------------------------------------------------------
//  Local constant values
//
//  __ac2CRCTbl
//      This is a table of CRC values. Since they are all known at compile
//      time, it is hard coded.
//
//  __c4MaxTblSz
//      This is the size of the __af8RTable[] array.
// -----------------------------------------------------------------------------
const   tCIDLib::TCard2  __ac2CRCTable[256] =
{
     0x0000, 0xC0C1, 0xC181, 0x0140, 0xC301, 0x03C0, 0x0280, 0xC241
   , 0xC601, 0x06C0, 0x0780, 0xC741, 0x0500, 0xC5C1, 0xC481, 0x0440
   , 0xCC01, 0x0CC0, 0x0D80, 0xCD41, 0x0F00, 0xCFC1, 0xCE81, 0x0E40
   , 0x0A00, 0xCAC1, 0xCB81, 0x0B40, 0xC901, 0x09C0, 0x0880, 0xC841
   , 0xD801, 0x18C0, 0x1980, 0xD941, 0x1B00, 0xDBC1, 0xDA81, 0x1A40
   , 0x1E00, 0xDEC1, 0xDF81, 0x1F40, 0xDD01, 0x1DC0, 0x1C80, 0xDC41
   , 0x1400, 0xD4C1, 0xD581, 0x1540, 0xD701, 0x17C0, 0x1680, 0xD641
   , 0xD201, 0x12C0, 0x1380, 0xD341, 0x1100, 0xD1C1, 0xD081, 0x1040
   , 0xF001, 0x30C0, 0x3180, 0xF141, 0x3300, 0xF3C1, 0xF281, 0x3240
   , 0x3600, 0xF6C1, 0xF781, 0x3740, 0xF501, 0x35C0, 0x3480, 0xF441
   , 0x3C00, 0xFCC1, 0xFD81, 0x3D40, 0xFF01, 0x3FC0, 0x3E80, 0xFE41
   , 0xFA01, 0x3AC0, 0x3B80, 0xFB41, 0x3900, 0xF9C1, 0xF881, 0x3840
   , 0x2800, 0xE8C1, 0xE981, 0x2940, 0xEB01, 0x2BC0, 0x2A80, 0xEA41
   , 0xEE01, 0x2EC0, 0x2F80, 0xEF41, 0x2D00, 0xEDC1, 0xEC81, 0x2C40
   , 0xE401, 0x24C0, 0x2580, 0xE541, 0x2700, 0xE7C1, 0xE681, 0x2640
   , 0x2200, 0xE2C1, 0xE381, 0x2340, 0xE101, 0x21C0, 0x2080, 0xE041
   , 0xA001, 0x60C0, 0x6180, 0xA141, 0x6300, 0xA3C1, 0xA281, 0x6240
   , 0x6600, 0xA6C1, 0xA781, 0x6740, 0xA501, 0x65C0, 0x6480, 0xA441
   , 0x6C00, 0xACC1, 0xAD81, 0x6D40, 0xAF01, 0x6FC0, 0x6E80, 0xAE41
   , 0xAA01, 0x6AC0, 0x6B80, 0xAB41, 0x6900, 0xA9C1, 0xA881, 0x6840
   , 0x7800, 0xB8C1, 0xB981, 0x7940, 0xBB01, 0x7BC0, 0x7A80, 0xBA41
   , 0xBE01, 0x7EC0, 0x7F80, 0xBF41, 0x7D00, 0xBDC1, 0xBC81, 0x7C40
   , 0xB401, 0x74C0, 0x7580, 0xB541, 0x7700, 0xB7C1, 0xB681, 0x7640
   , 0x7200, 0xB2C1, 0xB381, 0x7340, 0xB101, 0x71C0, 0x7080, 0xB041
   , 0x5000, 0x90C1, 0x9181, 0x5140, 0x9301, 0x53C0, 0x5280, 0x9241
   , 0x9601, 0x56C0, 0x5780, 0x9741, 0x5500, 0x95C1, 0x9481, 0x5440
   , 0x9C01, 0x5CC0, 0x5D80, 0x9D41, 0x5F00, 0x9FC1, 0x9E81, 0x5E40
   , 0x5A00, 0x9AC1, 0x9B81, 0x5B40, 0x9901, 0x59C0, 0x5880, 0x9841
   , 0x8801, 0x48C0, 0x4980, 0x8941, 0x4B00, 0x8BC1, 0x8A81, 0x4A40
   , 0x4E00, 0x8EC1, 0x8F81, 0x4F40, 0x8D01, 0x4DC0, 0x4C80, 0x8C41
   , 0x4400, 0x84C1, 0x8581, 0x4540, 0x8701, 0x47C0, 0x4680, 0x8641
   , 0x8201, 0x42C0, 0x4380, 0x8341, 0x4100, 0x81C1, 0x8081, 0x4040
};
const   tCIDLib::TCard4     __c4HashTblSz   = kCIDLib::c4MemPageSize;
const   tCIDLib::TCard4     __c4MaxTblSz    = 267;
const   tCIDLib::TCard4     __c4NumWaves    = 10;
const   tCIDLib::TCard4     __c4SinTblSz    = 1000;
const   tCIDLib::TFloat8    __f8MinX        = -10000.0;
const   tCIDLib::TFloat8    __f8MinY        = __f8MinX;
const   tCIDLib::TFloat8    __f8MinZ        = __f8MinX;
const   tCIDLib::TFloat8    __f8NumWaves    = 10;
const   tCIDLib::TFloat8    __f8RandDiv     = tCIDLib::TFloat8(0x7FFF);
const   tCIDLib::TFloat8    __f8RealScale   = 2.0 / 65535.0;
const   tCIDLib::TInt2      __i2RandMask    = 0x7FFF;



// -----------------------------------------------------------------------------
//  Local macros
// -----------------------------------------------------------------------------
#define f8IncSum(m,s,x,y,z) ((s)*(__af8RTable[m]*0.5 \
                                + __af8RTable[m+1]*(x) \
                                + __af8RTable[m+2]*(y) \
                                + __af8RTable[m+3]*(z)))

#define f8SCurve(a)         ((a)*(a)*(3.0-2.0*(a)))

#define i2Hash3D(a,b,c)     __ai2HashTable[ \
                                (tCIDLib::TInt2)(__ai2HashTable[ \
                                (tCIDLib::TInt2)(__ai2HashTable[ \
                                (tCIDLib::TInt2)((a) & 0xFFFL)] \
                                ^ ((b) & 0xFFFL))] ^ ((c) & 0xFFFL))]



// -----------------------------------------------------------------------------
//  Local data
//
//  __af8FreqTbl
//      This is a table of random values used by the wave textures, to randomize
//      the periodicities of the waves.
//
//  __af8RTable
//      This is a table of 
//
//  __af8SinTable
//      This is an array of sine values, which are predone to avoid the 
//      overhead of the sine calculation.
//
//  __ai2HashTable
//      This is a table of randomly generated numbers used by the noise
//      functions.
//
//  __avecWaveSources
//      This is an array of vectors used by the wave textures.
// -----------------------------------------------------------------------------
tCIDLib::TFloat8    __af8FreqTbl[__c4NumWaves];
tCIDLib::TFloat8    __af8RTable[__c4MaxTblSz];
tCIDLib::TFloat8    __af8SineTable[__c4SinTblSz];
tCIDLib::TInt2      __ai2HashTable[__c4HashTblSz];
T3DVector           __avecWaveSources[__c4NumWaves];



// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __i2CRC16
//
// DESCRIPTION:
//
//  This function will calculate a CRC on the data in the passed buffer.
// ---------------------------------------
//   INPUT: pc1Buf is the buffer to calc the CRC on
//          c4Size is the size of the buffer.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TInt2 __i2CRC16(        tCIDLib::TCard1*    pc1Buf
                                , const tCIDLib::TCard4     c4Size)
{
    tCIDLib::TCard2 c2CRC = 0;

    tCIDLib::TCard4 c4CountDown = c4Size;
    while (c4CountDown--)
    {
        c2CRC = (c2CRC >> 8)
                ^ __ac2CRCTable[(tCIDLib::TCard1)(c2CRC ^ *pc1Buf)];
        pc1Buf++;
    }
    return c2CRC;
}


//
// FUNCTION/METHOD NAME: __i2RTblCalc
//
// DESCRIPTION:
//
//  This method will calculate a CRC on the passed buffer, which is
//  intepreted as a vector.
// ---------------------------------------
//   INPUT: pc1Data is the data buffer pointer.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TInt2 __i2RTblCalc(tCIDLib::TCard1* const pc1Data)
{
    tCIDLib::TFloat8*   pf8Data = (tCIDLib::TFloat8*)pc1Data;

    *pf8Data *= 0.12345;
    pf8Data++;
    *pf8Data *= 0.12345;
    pf8Data++;
    *pf8Data *= 0.12345;
    return __i2CRC16(pc1Data, sizeof(tCIDLib::TFloat8)*3);
}


//
// FUNCTION/METHOD NAME: __InitTables
//
// DESCRIPTION:
//
//  This method is called from the intra-facility init function,
//  _bInitTxtrTables(), below. We just init all of the local texture tables.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __InitTables()
{
    tCIDLib::TInt2   i2Ind, i2Val, i2Tmp;
    tCIDLib::TFloat8 af8Info[3];

    // Seed the random number generator
    TMathLib::SeedRandom(0);

    for (i2Ind = 0; i2Ind < __c4HashTblSz; i2Ind++)
        __ai2HashTable[i2Ind] = i2Ind;

    for (i2Ind = __c4HashTblSz-1; i2Ind >= 0; i2Ind--)
    {
        i2Val = tCIDLib::TInt2(TMathLib::c4RandomNum() % __c4HashTblSz);
        i2Tmp = __ai2HashTable[i2Ind];
        __ai2HashTable[i2Ind] = __ai2HashTable[i2Val];
        __ai2HashTable[i2Val] = i2Tmp;
    }

    for (i2Ind = 0; i2Ind < __c4MaxTblSz; i2Ind++)
    {
        af8Info[0] = tCIDLib::TFloat8(i2Ind);
        af8Info[1] = tCIDLib::TFloat8(i2Ind);
        af8Info[2] = tCIDLib::TFloat8(i2Ind);
        __af8RTable[i2Ind] =
        (
            tCIDLib::TCard2(__i2RTblCalc((tCIDLib::TCard1*)af8Info))
            * __f8RealScale - 1.0
        );
    }

    for (i2Ind = 0; i2Ind < __c4SinTblSz; i2Ind++)
    {
        __af8SineTable[i2Ind] = TMathLib::f8Sine
        (
            tCIDLib::TFloat8(i2Ind)/tCIDLib::TFloat8(__c4SinTblSz)
            * (kCIDLib::f8PI * 2.0)
        );
    }

    for (i2Ind = 0; i2Ind < __c4NumWaves; i2Ind++)
    {
        TTxtrNoise::_VecNoise
        (
            __avecWaveSources[i2Ind]
            , tCIDLib::TFloat8(i2Ind)
            , 0.0
            , 0.0
        );
        __avecWaveSources[i2Ind].Normalize();
        __af8FreqTbl[i2Ind] = 
        (
            (TMathLib::c4RandomNum() & __i2RandMask)
            / __f8RandDiv + 0.01
        );
    }
}


// -----------------------------------------------------------------------------
//  Intra-facility functions
// -----------------------------------------------------------------------------

tCIDLib::TVoid
_InitTermNoiseTextures( const   tCIDLib::EInitTerm      eInitTerm
                        , const tCIDLib::EGlobalStates  eGlobals
                        , const TModule&                modInit
                        , const tCIDLib::TCard4         c4MaxChars
                        ,       tCIDLib::Tch* const     pszFailReason)
{
    if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
    &&  (eGlobals == tCIDLib::EGlobalState_Before))
    {
        __InitTables();
    }
}


//
// FUNCTION/METHOD NAME: _c2Random3D
//
// DESCRIPTION:
//
//  Computes a 3D noise value used by the supersampling code.
// ---------------------------------------
//   INPUT: i2X, i2Y are the current pixel col/row of the image.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TCard2 _c2Random3D(const   tCIDLib::TInt2 i2X
                            , const tCIDLib::TInt2 i2Y)
{
    tCIDLib::TInt2 i2HashVal1   = __ai2HashTable[tCIDLib::TInt2(i2X & 0x0FFF)];
    tCIDLib::TInt2 i2HashVal2   = tCIDLib::TInt2(i2HashVal1^i2Y) & 0x0FFF;
    tCIDLib::TInt2 i2Index      = __ai2HashTable[i2HashVal2] & 0x00FF;
    return __ac2CRCTable[i2Index];
}


// -----------------------------------------------------------------------------
//  CLASS: TTxtrNoise
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrNoise: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrNoise::TTxtrNoise() :

    TRTTexture()
    , __f8BumpAmount(0.0)
    , __f8Frequency(1.0)
    , __f8Phase(0.0)
    , __f8TurbVal(0.0)
{
}

TTxtrNoise::TTxtrNoise(const TTxtrNoise& txtrToCopy) :

    TRTTexture(txtrToCopy)
    , __f8BumpAmount(txtrToCopy.__f8BumpAmount)
    , __f8Frequency(txtrToCopy.__f8Frequency)
    , __f8Phase(txtrToCopy.__f8Phase)
    , __f8TurbVal(txtrToCopy.__f8TurbVal)
{
}


// -----------------------------------------------------------------------------
//  TTxtrNoise: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TTxtrNoise::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens     eToken;
    tCIDLib::TFloat8        f8Tmp;
    TString                 strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    // Next token must be the open paren
    prsiSrc.CheckOpenParen();

    // Deal with the valid tokens at this level
    if (eToken == TParseInfo::eToken_Turbulence)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0.0) || (f8Tmp > 1.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(1.0)
            );
        }
        __f8TurbVal = f8Tmp;
    }
     else if (eToken == TParseInfo::eToken_Frequency)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0.0) || (f8Tmp > 1.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(1.0)
            );
        }
        __f8Frequency = f8Tmp;
    }
     else if (eToken == TParseInfo::eToken_BumpAmount)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0.0) || (f8Tmp > 1.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(1.0)
            );
        }
        __f8BumpAmount = f8Tmp;
    }
     else if (eToken == TParseInfo::eToken_Phase)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0.0) || (f8Tmp > 1.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(1.0)
            );
        }
        __f8Phase = f8Tmp;
    }
     else
    {
        // Unget the token and pass it to parent
        prsiSrc.UnGet("(");
        prsiSrc.UnGet(strToken);
        TParent::ParseMethod(prsiSrc);
        return;
    }

    // Has to end with a closing paren
    prsiSrc.CheckCloseParen();
}


// -----------------------------------------------------------------------------
//  TTxtrNoise: Public, virtual methods
// -----------------------------------------------------------------------------

TTxtrNoise&  TTxtrNoise::operator=(const TTxtrNoise& txtrToAssign)
{
    if (this == &txtrToAssign)
        return *this;

    // Copy over our parent's stuff
    TParent::operator=(txtrToAssign);

    // And now copy our fields
    __f8BumpAmount  = txtrToAssign.__f8BumpAmount;
    __f8Frequency   = txtrToAssign.__f8Frequency;
    __f8Phase       = txtrToAssign.__f8Phase;
    __f8TurbVal     = txtrToAssign.__f8TurbVal;

    return *this;
}


// -----------------------------------------------------------------------------
//  TTxtrNoise: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid  TTxtrNoise::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the attribute info into the string
    TParent::_FormatTo(strmToWriteTo);
    strmToWriteTo   << L", Bump: " << __f8BumpAmount
                    << L", Freq: " << __f8Frequency
                    << L", Phase: " << __f8Phase
                    << L", Turb: " << __f8TurbVal;
}


tCIDLib::TVoid  TTxtrNoise::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Call our parent's version
    TParent::_StreamFrom(strmToReadFrom);

    // STream back in our member
    strmToReadFrom >> __f8BumpAmount;
    strmToReadFrom >> __f8Frequency;
    strmToReadFrom >> __f8Phase;
    strmToReadFrom >> __f8TurbVal;
}

tCIDLib::TVoid  TTxtrNoise::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent's version
    TParent::_StreamTo(strmToWriteTo);

    // Write out our members
    strmToWriteTo << __f8BumpAmount;
    strmToWriteTo << __f8Frequency;
    strmToWriteTo << __f8Phase;
    strmToWriteTo << __f8TurbVal;
}


// -----------------------------------------------------------------------------
//  TTxtrNoise: Protected, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TFloat8 
TTxtrNoise::_f8Cycloidal(const tCIDLib::TFloat8 f8Value)
{
    tCIDLib::TInt2 i2Ind;

    if (f8Value >= 0.0)
    {
        i2Ind = tCIDLib::TInt2
        (
            (f8Value - TMathLib::f8Floor(f8Value)) * __c4SinTblSz
        );
        return __af8SineTable[i2Ind];
    }
     else
    {
        i2Ind = tCIDLib::TInt2
        (
            (0.0 - (f8Value + TMathLib::f8Floor(0.0 - f8Value)))
            * __c4SinTblSz
        );
        return 0.0 - __af8SineTable[i2Ind];
    }
}


tCIDLib::TFloat8  TTxtrNoise::_f8Noise(const T3DVector& vecPos)
{
    return _f8Noise(vecPos.f8X(), vecPos.f8Y(), vecPos.f8Z());
}

tCIDLib::TFloat8 
TTxtrNoise::_f8Noise(   const   tCIDLib::TFloat8    f8X
                        , const tCIDLib::TFloat8    f8Y
                        , const tCIDLib::TFloat8    f8Z)
{
    tCIDLib::TInt2      i2M;
    tCIDLib::TFloat8    f8Sum;

    tCIDLib::TFloat8 f8XVal = f8X - __f8MinX;
    tCIDLib::TFloat8 f8YVal = f8Y - __f8MinY;
    tCIDLib::TFloat8 f8ZVal = f8Z - __f8MinZ;

    tCIDLib::TInt4   i4X = (tCIDLib::TInt4)f8XVal;
    tCIDLib::TInt4   i4Y = (tCIDLib::TInt4)f8YVal;
    tCIDLib::TInt4   i4Z = (tCIDLib::TInt4)f8ZVal;

    tCIDLib::TInt4   i4JX = i4X + 1;
    tCIDLib::TInt4   i4JY = i4Y + 1;
    tCIDLib::TInt4   i4JZ = i4Z + 1;

    tCIDLib::TFloat8 f8SX = f8SCurve(f8XVal - i4X);
    tCIDLib::TFloat8 f8SY = f8SCurve(f8YVal - i4Y);
    tCIDLib::TFloat8 f8SZ = f8SCurve(f8ZVal - i4Z);

    tCIDLib::TFloat8 f8TX = 1.0 - f8SX;
    tCIDLib::TFloat8 f8TY = 1.0 - f8SY;
    tCIDLib::TFloat8 f8TZ = 1.0 - f8SZ;

    i2M = i2Hash3D(i4X, i4Y, i4Z) & 0xFF;
    f8Sum = f8IncSum(i2M, f8TX*f8TY*f8TZ, f8XVal-i4X, f8YVal-i4Y, f8ZVal-i4Z);

    i2M = i2Hash3D(i4JX, i4Y, i4Z) & 0xFF;
    f8Sum += f8IncSum(i2M, f8SX*f8TY*f8TZ, f8XVal-i4JX, f8YVal-i4Y, f8ZVal-i4Z);

    i2M = i2Hash3D(i4X, i4JY, i4Z) & 0xFF;
    f8Sum += f8IncSum(i2M, f8TX*f8SY*f8TZ, f8XVal-i4X, f8YVal-i4JY, f8ZVal-i4Z);

    i2M = i2Hash3D(i4JX, i4JY, i4Z) & 0xFF;
    f8Sum += f8IncSum(i2M, f8SX*f8SY*f8TZ, f8XVal-i4JX, f8YVal-i4JY, f8ZVal-i4Z);

    i2M = (i2Hash3D(i4X, i4Y, i4JZ) & 0xFF);
    f8Sum += f8IncSum(i2M, f8TX*f8TY*f8SZ, f8XVal-i4X, f8YVal-i4Y, f8ZVal-i4JZ);

    i2M = (i2Hash3D(i4JX, i4Y, i4JZ) & 0xFF);
    f8Sum += f8IncSum(i2M, f8SX*f8TY*f8SZ, f8XVal-i4JX, f8YVal-i4Y, f8ZVal-i4JZ);

    i2M = i2Hash3D(i4X, i4JY, i4JZ) & 0xFF;
    f8Sum += f8IncSum(i2M, f8TX*f8SY*f8SZ, f8XVal-i4X, f8YVal-i4JY, f8ZVal-i4JZ);

    i2M = i2Hash3D(i4JX, i4JY, i4JZ) & 0xFF;
    f8Sum += f8IncSum(i2M, f8SX*f8SY*f8SZ, f8XVal-i4JX, f8YVal-i4JY, f8ZVal-i4JZ);

    f8Sum += 0.5;

    if (f8Sum < 0.0)
        f8Sum = 0.0;
    if (f8Sum > 1.0)
        f8Sum = 1.0;

    return (f8Sum);
}


tCIDLib::TFloat8  TTxtrNoise::_f8Turbulence(const T3DVector& vecPos)
{
    return _f8Turbulence(vecPos.f8X(), vecPos.f8Y(), vecPos.f8Z());
}

tCIDLib::TFloat8 
TTxtrNoise::_f8Turbulence(  const   tCIDLib::TFloat8    f8X
                            , const tCIDLib::TFloat8    f8Y
                            , const tCIDLib::TFloat8    f8Z)
{
    tCIDLib::TFloat8    f8PixelSize = 0.1;
    tCIDLib::TFloat8    f8T         = 0.0;
    tCIDLib::TFloat8    f8Scale, f8Value;

    for (f8Scale = 1.0; f8Scale > f8PixelSize; f8Scale *= 0.5)
    {
        f8Value = _f8Noise(f8X/f8Scale, f8Y/f8Scale, f8Z/f8Scale);
        f8T += TMathLib::f8Abs(f8Value) * f8Scale;
    }
    return f8T;
}


tCIDLib::TFloat8 
TTxtrNoise::_f8TriangleWave(const tCIDLib::TFloat8 f8Value)
{
    tCIDLib::TFloat8    f8Offset;

    if (f8Value >= 0.0)
        f8Offset = f8Value - TMathLib::f8Floor(f8Value);
    else
        f8Offset = f8Value - (-1.0 - TMathLib::f8Floor(TMathLib::f8Abs(f8Value)));

    if (f8Offset >= 0.5)
        return (2.0 * (1.0 - f8Offset));
    else
        return (2.0 * f8Offset);
}


tCIDLib::TVoid 
TTxtrNoise::_VecNoise(T3DVector& vecTarget, const T3DVector& vecPoint)
{
    _VecNoise(vecTarget, vecPoint.f8X(), vecPoint.f8Y(), vecPoint.f8Z());
}

tCIDLib::TVoid 
TTxtrNoise::_VecNoise(          T3DVector&           vecTarget
                        , const tCIDLib::TFloat8    f8X
                        , const tCIDLib::TFloat8    f8Y
                        , const tCIDLib::TFloat8    f8Z)
{
    tCIDLib::TInt2      i2M;
    tCIDLib::TFloat8    f8PX;
    tCIDLib::TFloat8    f8PY;
    tCIDLib::TFloat8    f8PZ;
    tCIDLib::TFloat8    f8S;

    tCIDLib::TFloat8 f8XVal = f8X - __f8MinX;
    tCIDLib::TFloat8 f8YVal = f8Y - __f8MinY;
    tCIDLib::TFloat8 f8ZVal = f8Z - __f8MinZ;

    tCIDLib::TInt4   i4X = (tCIDLib::TInt4)f8XVal;
    tCIDLib::TInt4   i4Y = (tCIDLib::TInt4)f8YVal;
    tCIDLib::TInt4   i4Z = (tCIDLib::TInt4)f8ZVal;

    tCIDLib::TInt4   i4JX = i4X + 1;
    tCIDLib::TInt4   i4JY = i4Y + 1;
    tCIDLib::TInt4   i4JZ = i4Z + 1;

    tCIDLib::TFloat8 f8SX = f8SCurve(f8XVal - i4X);
    tCIDLib::TFloat8 f8SY = f8SCurve(f8YVal - i4Y);
    tCIDLib::TFloat8 f8SZ = f8SCurve(f8ZVal - i4Z);

    tCIDLib::TFloat8 f8TX = 1.0 - f8SX;
    tCIDLib::TFloat8 f8TY = 1.0 - f8SY;
    tCIDLib::TFloat8 f8TZ = 1.0 - f8SZ;

    i2M = i2Hash3D(i4X, i4Y, i4Z ) & 0xFF;
    f8PX = f8XVal-i4X;  f8PY = f8YVal-i4Y;  f8PZ = f8ZVal-i4Z;
    f8S = f8TX*f8TY*f8TZ;
    vecTarget.Set
    (
        f8IncSum(i2M,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+4,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+8,f8S,f8PX,f8PY,f8PZ)
    );

    i2M = i2Hash3D(i4JX, i4Y, i4Z ) & 0xFF;
    f8PX = f8XVal-i4JX;
    f8S = f8SX*f8TY*f8TZ;
    vecTarget.Adjust
    (
        f8IncSum(i2M,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+4,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+8,f8S,f8PX,f8PY,f8PZ)
    );

    i2M = i2Hash3D(i4JX, i4JY, i4Z ) & 0xFF;
    f8PY = f8YVal-i4JY;
    f8S = f8SX*f8SY*f8TZ;
    vecTarget.Adjust
    (
        f8IncSum(i2M,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+4,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+8,f8S,f8PX,f8PY,f8PZ)
    );

    i2M = i2Hash3D(i4X, i4JY, i4Z ) & 0xFF;
    f8PX = f8XVal-i4X;
    f8S = f8TX*f8SY*f8TZ;
    vecTarget.Adjust
    (
        f8IncSum(i2M,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+4,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+8,f8S,f8PX,f8PY,f8PZ)
    );

    i2M = i2Hash3D(i4X, i4JY, i4JZ ) & 0xFF;
    f8PZ = f8ZVal-i4JZ;
    f8S = f8TX*f8SY*f8SZ;
    vecTarget.Adjust
    (
        f8IncSum(i2M,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+4,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+8,f8S,f8PX,f8PY,f8PZ)
    );

    i2M = i2Hash3D(i4JX, i4JY, i4JZ ) & 0xFF;
    f8PX = f8XVal-i4JX;
    f8S = f8SX*f8SY*f8SZ;
    vecTarget.Adjust
    (
        f8IncSum(i2M,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+4,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+8,f8S,f8PX,f8PY,f8PZ)
    );

    i2M = i2Hash3D(i4JX, i4Y, i4JZ ) & 0xFF;
    f8PY = f8YVal-i4Y;
    f8S = f8SX*f8TY*f8SZ;
    vecTarget.Adjust
    (
        f8IncSum(i2M,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+4,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+8,f8S,f8PX,f8PY,f8PZ)
    );

    i2M = i2Hash3D(i4X, i4Y, i4JZ ) & 0xFF;
    f8PX = f8XVal-i4X;
    f8S = f8TX*f8TY*f8SZ;
    vecTarget.Adjust
    (
        f8IncSum(i2M,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+4,f8S,f8PX,f8PY,f8PZ)
        , f8IncSum(i2M+8,f8S,f8PX,f8PY,f8PZ)
    );
}


tCIDLib::TVoid 
TTxtrNoise::_VecTurbulence(         T3DVector&          vecTarget
                            , const tCIDLib::TFloat8    f8X
                            , const tCIDLib::TFloat8    f8Y
                            , const tCIDLib::TFloat8    f8Z)
{
    tCIDLib::TFloat8 f8PixelSize = 0.01;
    T3DVector         vecTmp;

    vecTarget.Set(0.0, 0.0, 0.0);
    vecTmp.Set(0.0, 0.0, 0.0);

    for (tCIDLib::TFloat8 f8Scale = 1.0; f8Scale > f8PixelSize; f8Scale *= 0.5)
    {
        _VecNoise(vecTmp, f8X/f8Scale, f8Y/f8Scale, f8Z/f8Scale);
        vecTarget.AddScaled(vecTmp, f8Scale);
    }
}




// -----------------------------------------------------------------------------
//  CLASS: TTxtrBumps
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrBumps: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrBumps::TTxtrBumps() :

    TTxtrNoise()
{
}

TTxtrBumps::TTxtrBumps(const TTxtrBumps& txtrToCopy) :

    TTxtrNoise(txtrToCopy)
{
}


// -----------------------------------------------------------------------------
//  TTxtrBumps: Public operators
// -----------------------------------------------------------------------------

TTxtrBumps&  TTxtrBumps::operator=(const TTxtrBumps& txtrToAssign)
{
    if (this == &txtrToAssign)
        return *this;

    // Copy over our parent's stuff
    TParent::operator=(txtrToAssign);

    return *this;
}


// -----------------------------------------------------------------------------
//  TTxtrBumps: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean 
TTxtrBumps::bAdjustNormalAt(const   T3DVector&   vecPoint
                            ,       T3DVector&   vecNormal) const
{
    if (f8BumpAmount() == 0.0)
        return kCIDLib::False;

    T3DVector    vecBumpTurb;
    T3DVector    vecTmp(vecPoint);

    // Translate the point via the inverse transform
    trnsTexture().mtrxInvTrans.Transform(vecTmp);

    tCIDLib::TFloat8    f8X = vecTmp.f8X();
    tCIDLib::TFloat8    f8Y = vecTmp.f8Y();
    tCIDLib::TFloat8    f8Z = vecTmp.f8Z();

    _VecNoise(vecBumpTurb, f8X, f8Y, f8Z);
    vecBumpTurb *= f8BumpAmount();
    vecNormal += vecBumpTurb;
    vecNormal.Normalize();

    return kCIDLib::True;
}





// -----------------------------------------------------------------------------
//  CLASS: TTxtrDents
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrDents: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrDents::TTxtrDents() :

    TTxtrNoise()
{
}

TTxtrDents::TTxtrDents(const TTxtrDents& txtrToCopy) :

    TTxtrNoise(txtrToCopy)
{
}


// -----------------------------------------------------------------------------
//  TTxtrDents: Public operators
// -----------------------------------------------------------------------------

TTxtrDents&  TTxtrDents::operator=(const TTxtrDents& txtrToAssign)
{
    if (this == &txtrToAssign)
        return *this;

    // Copy over our parent's stuff
    TParent::operator=(txtrToAssign);

    return *this;
}


// -----------------------------------------------------------------------------
//  TTxtrDents: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean 
TTxtrDents::bAdjustNormalAt(const   T3DVector&   vecPoint
                            ,       T3DVector&   vecNormal) const
{
    if (f8BumpAmount() == 0.0)
        return kCIDLib::False;

    T3DVector         vecStuccoTurb;
    tCIDLib::TFloat8 f8Noise;

    T3DVector    vecTmp(vecPoint);

    // Translate the point via the inverse transform
    trnsTexture().mtrxInvTrans.Transform(vecTmp);

    tCIDLib::TFloat8    f8X = vecTmp.f8X();
    tCIDLib::TFloat8    f8Y = vecTmp.f8Y();
    tCIDLib::TFloat8    f8Z = vecTmp.f8Z();

    f8Noise = _f8Noise(f8X, f8Y, f8Z);
    f8Noise =  f8Noise * f8Noise * f8Noise * f8BumpAmount();

    _VecNoise(vecStuccoTurb, f8X, f8Y, f8Z);
    vecStuccoTurb *= f8Noise;
    vecNormal += vecStuccoTurb;
    vecNormal.Normalize();

    return kCIDLib::True;
}





// -----------------------------------------------------------------------------
//  CLASS: TTxtrWaves
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrWaves: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrWaves::TTxtrWaves() :

    TTxtrNoise()
{
}

TTxtrWaves::TTxtrWaves(const TTxtrWaves& txtrToCopy) :

    TTxtrNoise(txtrToCopy)
{
}


// -----------------------------------------------------------------------------
//  TTxtrWaves: Public operators
// -----------------------------------------------------------------------------

TTxtrWaves&  TTxtrWaves::operator=(const TTxtrWaves& txtrToAssign)
{
    if (this == &txtrToAssign)
        return *this;

    // Copy over our parent's stuff
    TParent::operator=(txtrToAssign);

    return *this;
}

// -----------------------------------------------------------------------------
//  TTxtrWaves: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean 
TTxtrWaves::bAdjustNormalAt(const   T3DVector&   vecPoint
                            ,       T3DVector&   vecNormal) const
{
    tCIDLib::TInt4   i4Ind;
    tCIDLib::TFloat8 f8Length, f8Scalar, f8Index, f8SinValue;
    T3DVector         vecTmp;
    T3DVector         vecTrans(vecPoint);

    // Translate the point via the inverse transform
    trnsTexture().mtrxInvTrans.Transform(vecTrans);

    for (i4Ind = 0; i4Ind < __c4NumWaves; i4Ind++)
    {
        vecTmp = vecTrans;
        vecTmp -= __avecWaveSources[i4Ind];
        f8Length = vecTmp.f8Dot(vecTmp);
        if (f8Length == 0.0)
            f8Length = 1.0;
        f8Length = TMathLib::f8SqrRoot(f8Length);
        f8Index = (f8Length * f8Frequency() * __af8FreqTbl[i4Ind]) + f8Phase();
        f8SinValue = _f8Cycloidal(f8Index);
        f8Scalar = f8SinValue * f8BumpAmount() / __af8FreqTbl[i4Ind];
        vecTmp *= (f8Scalar / f8Length / __f8NumWaves);
        vecNormal += vecTmp;
    }
    vecNormal.Normalize();

    return kCIDLib::True;
}




// -----------------------------------------------------------------------------
//  CLASS: TTxtrWrinkles
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrWrinkles: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrWrinkles::TTxtrWrinkles() :

    TTxtrNoise()
{
}

TTxtrWrinkles::TTxtrWrinkles(const TTxtrWrinkles& txtrToCopy) :

    TTxtrNoise(txtrToCopy)
{
}


// -----------------------------------------------------------------------------
//  TTxtrWrinkles: Public operators
// -----------------------------------------------------------------------------

TTxtrWrinkles& TTxtrWrinkles::operator=(const TTxtrWrinkles& txtrToAssign)
{
    if (this == &txtrToAssign)
        return *this;

    // Copy over our parent's stuff
    TParent::operator=(txtrToAssign);

    return *this;
}

// -----------------------------------------------------------------------------
//  TTxtrWrinkles: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean 
TTxtrWrinkles::bAdjustNormalAt( const   T3DVector&   vecPoint
                                ,       T3DVector&   vecNormal) const
{
    if (f8BumpAmount() == 0.0)
        return kCIDLib::False;

    tCIDLib::TFloat8    f8Scale = 1.0;
    T3DVector            vecResult;
    T3DVector            vecTmp(vecPoint);
    T3DVector            vecValue;

    // Translate the point via the inverse transform
    trnsTexture().mtrxInvTrans.Transform(vecTmp);

    tCIDLib::TFloat8    f8X = vecTmp.f8X();
    tCIDLib::TFloat8    f8Y = vecTmp.f8Y();
    tCIDLib::TFloat8    f8Z = vecTmp.f8Z();

    for (tCIDLib::TInt4 i4Ind = 0; i4Ind < 10; f8Scale *= 2.0, i4Ind++)
    {
        _VecNoise(vecValue, f8X * f8Scale, f8Y * f8Scale, f8Z * f8Scale);
        vecValue /= f8Scale;
        vecValue.Abs();
        vecResult += vecValue;
    }

    vecResult *= f8BumpAmount();
    vecNormal += vecResult;
    vecNormal.Normalize();

    return kCIDLib::True;
}

