//
// NAME: CIDTracer_MetaObj.Cpp
//
// DESCRIPTION:
//
//  This module implements the TRTMetaObj class, which is the basis of all
//  of the 'object container' classes such as bounding, composite, and
//  CSG classes.
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 06/01/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData2(TRTMetaObj,TRTObject)



// -----------------------------------------------------------------------------
//  CLASS: TRTMetaObj
// PREFIX: meta
// -----------------------------------------------------------------------------

TRTMetaObj::TRTMetaObj() :

    TRTObject()
    , __prtlstChildren(0)
{
    __prtlstChildren = new TRTObjList;
}

TRTMetaObj::TRTMetaObj(const TRTMetaObj& metaSrc) :

    TRTObject(metaSrc)
    , __prtlstChildren(0)
{
    __prtlstChildren = new TRTObjList(*metaSrc.__prtlstChildren);
}

 TRTMetaObj::~TRTMetaObj()
{
    __prtlstChildren->Flush();
    delete __prtlstChildren;
}


// -----------------------------------------------------------------------------
//  TRTMetaObj: Public, virtual methods
// -----------------------------------------------------------------------------

TRTMetaObj&  TRTMetaObj::operator=(const TRTMetaObj& metaSrc)
{
    if (this == &metaSrc)
        return *this;

    // Copy our parent's members
    ((TParent&)*this) = (const TParent&)metaSrc;

    // And flush and copy the source list of children
    __prtlstChildren->Flush();
    *__prtlstChildren = *metaSrc.__prtlstChildren;

    return *this;
}


// -----------------------------------------------------------------------------
//  TRTMetaObj: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TRTMetaObj::AppendTexture(          TRTTexture* const   ptxtrToAppend
                            , const TString&            strName
                            , const TString&            strNameSpace)
{
    TRTObjNode* pnodeCur = __prtlstChildren->pnodeHead();

    while (pnodeCur)
    {
        pnodeCur->pobjData()->AppendTexture
        (
            ::pDupObject<TRTTexture>(*ptxtrToAppend)
            , strName
            , strNameSpace
        );
        pnodeCur = pnodeCur->pnodeNext();
    }

    // And now delete the texture
    delete ptxtrToAppend;
}


tCIDLib::TVoid  TRTMetaObj::Invert()
{
    TRTObjNode* pnodeCur = __prtlstChildren->pnodeHead();

    while (pnodeCur)
    {
        pnodeCur->pobjData()->Invert();
        pnodeCur = pnodeCur->pnodeNext();
    }
}


tCIDLib::TVoid  TRTMetaObj::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens eToken;
    TString             strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    // The next token has to be an open parens
    prsiSrc.CheckOpenParen();

    // Deal with the valid tokens at this level
    if (eToken == TParseInfo::eToken_AddObject)
    {
        TRTObject*  prtobjNew = 0;
        TString   strName;
        TString   strNameSpace;

        prtobjNew = prsiSrc.prtobjParseObject(strName);

        // Add the new object as a child
        AdoptChild(prtobjNew, strName, prsiSrc.strCurNameSpace());
    }
     else
    {
        // Unget the tokens and pass it to our parent
        prsiSrc.UnGet("(");
        prsiSrc.UnGet(strToken);
        TParent::ParseMethod(prsiSrc);
        return;
    }

    // Has to end with a closing paren
    prsiSrc.CheckCloseParen();
}


tCIDLib::TVoid 
TRTMetaObj::PrependTexture(         TRTTexture* const   ptxtrToPrepend
                            , const TString&            strName
                            , const TString&            strNameSpace)
{
    TRTObjNode* pnodeCur = __prtlstChildren->pnodeHead();

    while (pnodeCur)
    {
        pnodeCur->pobjData()->PrependTexture
        (
            ::pDupObject<TRTTexture>(*ptxtrToPrepend)
            , strName
            , strNameSpace
        );
        pnodeCur = pnodeCur->pnodeNext();
    }

    // And now delete the texture
    delete ptxtrToPrepend;
}


tCIDLib::TVoid  TRTMetaObj::Transform(const TTransform& trnsApply)
{
    // Call our parent
    TParent::Transform(trnsApply);

    TRTObjNode* pnodeCur = __prtlstChildren->pnodeHead();
    while (pnodeCur)
    {
        pnodeCur->pobjData()->Transform(trnsApply);
        pnodeCur = pnodeCur->pnodeNext();
    }
}


tCIDLib::TVoid 
TRTMetaObj::TransformTextures(const TTransform& trnsApply)
{
    TRTObjNode* pnodeCur = __prtlstChildren->pnodeHead();
    while (pnodeCur)
    {
        pnodeCur->pobjData()->TransformTextures(trnsApply);
        pnodeCur = pnodeCur->pnodeNext();
    }
}


// -----------------------------------------------------------------------------
//  TRTMetaObj: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TRTMetaObj::AdoptChild(         TRTObject* const    prtobjNew
                        , const TString&          strName
                        , const TString&          strNameSpace)
{
    //
    //  Make sure that this is not a duplicate name. Object and texture names
    //  must be unique within a particular scope.
    //
    if (&strName && &strNameSpace)
    {
        if (__prtlstChildren->pobjFind(strName, strNameSpace))
        {
            // <TBD> Log a duplicate name error
        }
    }

    // Allowed derived classes to preprocess
    __NewObject(prtobjNew, __prtlstChildren->c4NodeCount());

    // MUST append because the order of insertion is important
    __prtlstChildren->Append(prtobjNew, strName, strNameSpace);
}



// -----------------------------------------------------------------------------
//  TRTMetaObj: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TRTMetaObj::_AllIntersects( const   TLightRay&  lrayTest
                            , TIntersectArray&  intraHits) const
{
    TRTObjNode* pnodeCur = __prtlstChildren->pnodeHead();
    while (pnodeCur)
    {
        TRTObject* const prtobjCur = pnodeCur->pobjData();

        // Add this child's intersections
        prtobjCur->AllIntersects(lrayTest, intraHits);

        pnodeCur = pnodeCur->pnodeNext();
    }
}


tCIDLib::TBoolean 
TRTMetaObj::_bPointInObj(const T3DVector& vecPoint) const
{
    TRTObjNode* pnodeCur = __prtlstChildren->pnodeHead();
    while (pnodeCur)
    {
        if (pnodeCur->pobjData()->bPointInObj(vecPoint))
            return kCIDLib::True;

        pnodeCur = pnodeCur->pnodeNext();
    }
    return kCIDLib::False;
}


tCIDLib::TBoolean 
TRTMetaObj::_bTestIntersect(const   TLightRay&          lrayTest
                            ,       tCIDLib::TFloat8&   f8Dist1
                            ,       tCIDLib::TFloat8&   f8Dist2) const
{
    TIntersectArray intraTmp;

    // Get all of the intersections into a local list
    _AllIntersects(lrayTest, intraTmp);

    tCIDLib::TCard4  c4Cnt = intraTmp.c4HitCount();
    if (!c4Cnt)
        return kCIDLib::False;

    // Sort it
    if (c4Cnt > 1)
        intraTmp.Sort();

    // Send back the first one for f8Dist1
    TIntersect*  pintrTmp = intraTmp[0];
    f8Dist1 = pintrTmp->f8Distance;

    if (f8Dist1 < kCIDTracer_::f8SmallTolerance)
        return kCIDLib::False;

    // If at least 2, then do second distance
    if (c4Cnt > 1)
    {
        pintrTmp = intraTmp[c4Cnt-1];
        f8Dist2 = pintrTmp->f8Distance;

        if (f8Dist2 < kCIDTracer_::f8SmallTolerance)
            f8Dist2 = f8Dist1;
    }
     else
    {
        f8Dist2 = f8Dist1;
    }
    return kCIDLib::True;
}


tCIDLib::TVoid TRTMetaObj::_FormatTo(TTextStream& strmToWriteTo) const
{
    TParent::_FormatTo(strmToWriteTo);
    strmToWriteTo << L", Child count: " << __prtlstChildren->c4NodeCount();
}


tCIDLib::TVoid  TRTMetaObj::_Rotate( const   T3DVector&   vecRotate
                                            , const TTransform& trnsRotate)
{
    TRTObjNode* pnodeCur = __prtlstChildren->pnodeHead();
    while (pnodeCur)
    {
        pnodeCur->pobjData()->Rotate(vecRotate);
        pnodeCur = pnodeCur->pnodeNext();
    }
}


tCIDLib::TVoid  TRTMetaObj::_Scale(  const   T3DVector&   vecScale
                                            , const TTransform& trnsScale)
{
    TRTObjNode* pnodeCur = __prtlstChildren->pnodeHead();
    while (pnodeCur)
    {
        pnodeCur->pobjData()->Scale(vecScale);
        pnodeCur = pnodeCur->pnodeNext();
    }
}


tCIDLib::TVoid  TRTMetaObj::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Flush any existing children of the list
    __prtlstChildren->Flush();

    // Call our parent's version
    TParent::_StreamFrom(strmToReadFrom);

    // Read in the count of child objects we wrote out
    tCIDLib::TCard4 c4Count;
    strmToReadFrom >> c4Count;

    //
    //  Loop through the children and stream them back in polymorphically,
    //  adding them to the list of children.
    //
    TRTObject* prtobjNew;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
    {
        ::PolymorphicRead(prtobjNew, strmToReadFrom);
        __prtlstChildren->Append(prtobjNew);
    }
}

tCIDLib::TVoid  TRTMetaObj::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent's version
    TParent::_StreamTo(strmToWriteTo);

    // Write out the number of child objects we are storing
    strmToWriteTo << __prtlstChildren->c4NodeCount();

    // Loop through the children and stream them out polymorphic
    TRTObjNode* pnodeCur = __prtlstChildren->pnodeHead();
    while (pnodeCur)
    {
        ::PolymorphicWrite(pnodeCur->pobjData(), strmToWriteTo);
        pnodeCur = pnodeCur->pnodeNext();
    }
}


tCIDLib::TVoid 
TRTMetaObj::_Translate( const   T3DVector&   vecTrans
                        , const TTransform& trnsTranslate)
{
    TRTObjNode* pnodeCur = __prtlstChildren->pnodeHead();
    while (pnodeCur)
    {
        pnodeCur->pobjData()->Translate(vecTrans);
        pnodeCur = pnodeCur->pnodeNext();
    }
}
