//
// NAME: CIDTracer_List_.Hpp
//
// DESCRIPTION: 
//
//  This is the header file for the CIDTracer_List.Cpp module. This module
//  implements a specialized container class named TRTObjList. This class is
//  a high performance container for use by the Ray Tracer. Since there
//  are a lot of uses of containers in the ray tracer, and since its
//  performance requirements are way above and beyond that of most other code,
//  and since its use of the container is very simple this became necessary
//  (and desirable really.) The flexibility of the general purpose container
//  classes (unused and unwanted) put them out of contention.
//
//  This container class makes the following assumptions:
//
//  1)  Each node is an object of type T
//  2)  Speed of iteration is a fundamental requirement for good performance
//      because lists of objects and textures are iterated potentially
//      tens of millions of times during a reasonably sized scene and
//      hundreds of millions of times in a really big, super-sampled
//      scene.
//  3)  Every node has an optional name and a hash of that name for fast
//      location of the object.
//  4)  They are all used to hold heterogenous groups of objects derived
//          from the instantiation type.
//  5)  It maintains both the head and tail so that appends are as efficient
//          as prepends.
//  6)  It supports Add()/Append semantics and Push/Pop stack like semnatics
//          since those are the two schemes needed. Push() is just a call to
//          the Add() method.
//
//  For these reasons, the nodes are of a very simple template class. It is
//  in the form of a singly linked list, since there is a need for open ended
//  expansion and fast iteration without range checking and it only needs to
//  be iterated one direction.
//
//  The use of a template class is desirable because of the fact that these
//  lists have to hold textures, colors, and scene objects. These are only
//  related at the lowest class levels and therefore a non-template container
//  would have to use TObject objects requiring lots of casting and more
//  virtual linkages.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/04/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// ----------------------------------------------------------------------------
//  Forward reference the bag and bag cursor classes
// ----------------------------------------------------------------------------
template <class T> class TRTList;


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: TRTListNode
// PREFIX: lnod
// -----------------------------------------------------------------------------
template <class T> class TRTListNode : public TObject
{
    public              :
        // ---------------------------------------------------------------------
        // Constructors and Destructors
        // ---------------------------------------------------------------------
        TRTListNode(        T* const    pobjData
                    , const TString&    strName
                    , const TString&    strNameSpace) :

            __hshName(strName.hshCalcHash(kCIDTracer::c4HashModulus))
            , __hshNameSpace(strNameSpace.hshCalcHash(kCIDTracer::c4HashModulus))
            , __pnodeNext(0)
            , __pobjData(pobjData)
            , __strName(strName)
            , __strNameSpace(strNameSpace)
        {
        }

        TRTListNode(const TRTListNode<T>& lnodToCopy) :

            __hshName(lnodToCopy.__hshName)
            , __hshNameSpace(lnodToCopy.__hshNameSpace)
            , __pnodeNext(0)
            , __pobjData(0)
            , __strName(lnodToCopy.__strName)
            , __strNameSpace(lnodToCopy.__strNameSpace)
        {
            __pobjData = ::pDupObject<T>(*lnodToCopy.__pobjData);
        }

        ~TRTListNode()
        {
            delete __pobjData;
            __pobjData = 0;
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual 
        // ---------------------------------------------------------------------
        T* pobjData() const
        {
            return __pobjData;
        }

        TRTListNode<T>* pnodeNext() const
        {
            return __pnodeNext;
        }

        const TString& strName() const
        {
            return __strName;
        }

        const TString& strNameSpace() const
        {
            return __strNameSpace;
        }


    private :
        // ---------------------------------------------------------------------
        //  Make the releated list template a friend
        // ---------------------------------------------------------------------
        friend class TRTList<T>;


        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TRTListNode();

        tCIDLib::TVoid operator=(const TRTListNode<T>&);


        // ---------------------------------------------------------------------
        //  Protected, non-virtual methods
        //
        //  __hshName
        //  __hshNameSpace
        //      The hash of the name of the object and the namespace in which
        //      it lives, for fast lookup of the object. they are  set during
        //      construction.
        //
        //  __pnodeNext
        //      A pointer to the next node, or 0 if this is the last node.
        //
        //  __pobjData
        //      A pointer to the object that represents the data of this node.
        //
        //  __strName
        //  __strNameSpace
        //      The name of this object and he name space in which this
        //      object lives.
        // ---------------------------------------------------------------------
        tCIDLib::THashVal       __hshName;
        tCIDLib::THashVal       __hshNameSpace;
        TRTListNode<T>*         __pnodeNext;
        T*                      __pobjData;
        TString                 __strName;
        TString                 __strNameSpace;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TRTListNode<T>,TObject)
};

template <class T> const TClass TRTListNode<T>::clsThis;



// -----------------------------------------------------------------------------
//  CLASS: TRTList
// PREFIX: rtlst
// -----------------------------------------------------------------------------
template <class T> class TRTList : public TObject
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and Destructors
        // ---------------------------------------------------------------------
        TRTList(const tCIDLib::EDeleteModes eDelMode = tCIDLib::EDel_Delete) :

            __c4NodeCount(0)
            , __eDelMode(eDelMode)
            , __pnodeHead(0)
            , __pnodeTail(0)
        {
        }

        TRTList(const TRTList<T>& rtlstToCopy) :

            __c4NodeCount(rtlstToCopy.__c4NodeCount)
            , __eDelMode(rtlstToCopy.__eDelMode)
            , __pnodeHead(0)
            , __pnodeTail(0)
        {
            TRTListNode<T>* pnodeCur = rtlstToCopy.__pnodeHead;

            // Duplicate any nodes
            while (pnodeCur)
            {
                Append(new TRTListNode<T>(*pnodeCur));
                pnodeCur = pnodeCur->__pnodeNext;
            }
        }

        ~TRTList()
        {
            Flush();
        }


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TRTList<T>& operator=(const TRTList<T>& rtlstToAssign)
        {
            // Flush this list
            Flush();

            TRTListNode<T>* pnodeCur = rtlstToAssign.__pnodeHead;

            // Duplicate any nodes
            while (pnodeCur)
            {
                Append(new TRTListNode<T>(*pnodeCur));
                pnodeCur = pnodeCur->__pnodeNext;
            }

            return *this;
        }


        // --------------------------------------------------------------------
        //  Public, non-inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Add(TRTListNode<T>* const pnodeToAdd)
        {
            // Save the current head (or 0 if none)
            TRTListNode<T>* pnodeOldHead = __pnodeHead;

            //
            //  If the list is empty, then this node will be the last one
            //  too.
            //
            if (!__pnodeHead)
                __pnodeTail = pnodeToAdd;

            // Make the new guy the head and point his next at old head
            __pnodeHead = pnodeToAdd;
            __pnodeHead->__pnodeNext = pnodeOldHead;

            __c4NodeCount++;
        }

        tCIDLib::TVoid Add(         T* const    pobjToAdd
                            , const TString&    strName = NUL_TString
                            , const TString&    strNameSpace = NUL_TString)
        {
            // Save the current head (or 0 if none)
            TRTListNode<T>* pnodeOldHead = __pnodeHead;

            TString   strActualName;
            TString   strActualNameSpace;

            if (&strName)
                strActualName = strName;

            if (&strNameSpace)
                strActualNameSpace = strNameSpace;

            //
            //  Create a new node and make it the head and point his next
            //  at old head.
            //
            __pnodeHead = new TRTListNode<T>
            (
                pobjToAdd
                , strActualName
                , strActualNameSpace
            );
            __pnodeHead->__pnodeNext = pnodeOldHead;

            //
            //  If the list is empty, then this node will be the last one
            //  too.
            //
            if (!pnodeOldHead)
                __pnodeTail = __pnodeHead;

            __c4NodeCount++;
        }

        tCIDLib::TVoid Append(TRTListNode<T>* const pnodeToAppend)
        {
            //
            //  If there is a current tail, then make the last node's next
            //  node the new node. if the list is empty, then just set the
            //  head and tail to the new node.
            //
            if (__pnodeTail)
                __pnodeTail->__pnodeNext = pnodeToAppend;
             else
                __pnodeHead = pnodeToAppend;

            // This new node becomes the tail node
            __pnodeTail = pnodeToAppend;

            // Make sure the appended node's next pointer is 0
            pnodeToAppend->__pnodeNext = 0;

            __c4NodeCount++;
        }

        tCIDLib::TVoid Append(          T* const    pobjToAppend
                                , const TString&    strName = NUL_TString
                                , const TString&    strNameSpace = NUL_TString)
        {
            TString   strActualName;
            TString   strActualNameSpace;

            if (&strName)
                strActualName = strName;

            if (&strNameSpace)
                strActualNameSpace = strNameSpace;

            // Create a new node for this object
            TRTListNode<T>* pnodeNew = new TRTListNode<T>
            (
                pobjToAppend
                , strActualName
                , strActualNameSpace
            );

            //
            //  If there is a tail node, then set its next node to the
            //  new node. If the list is empty, then just point the head
            //  and tail to the new node.
            //
            if (__pnodeTail)
                __pnodeTail->__pnodeNext = pnodeNew;
            else
                __pnodeHead = pnodeNew;

            // This new node becomes the tail node
            __pnodeTail = pnodeNew;

            __c4NodeCount++;
        }

        tCIDLib::TVoid
        Flush(const tCIDLib::EDeleteModes eDelMode = tCIDLib::EDel_Default)
        {
            // Save the head
            TRTListNode<T>*  pnodeCur = __pnodeHead;

            // Zero the head and tail out and the node count
            __pnodeHead = 0;
            __pnodeTail = 0;
            __c4NodeCount = 0;

            // And loop through them, deleting as we go
            while (pnodeCur)
            {
                TRTListNode<T>*  pnodeNext = pnodeCur->__pnodeNext;

                //
                //  Figure out the actual delete mode. if the passed mode
                //  is default, then use the list's default mode. Else use
                //  the passed mode.
                //
                tCIDLib::EDeleteModes eActual = __eDelMode;
                if (eDelMode != tCIDLib::EDel_Default)
                    eActual = eDelMode;

                //
                //  If the actual delete mode is not delete, then zero the
                //  object pointer before we delete the node. This orphans
                //  the object.
                //
                if (eActual != tCIDLib::EDel_Delete)
                    pnodeCur->__pobjData = 0;

                // And now we can delete the node
                delete pnodeCur;

                // And get the next node now
                pnodeCur = pnodeNext;
            }
        }

        tCIDLib::TCard4 c4NodeCount() const
        {
            return __c4NodeCount;
        }

        TRTListNode<T>* pnodeHead() const
        {
            return __pnodeHead;
        }

        TRTListNode<T>* pnodeTail() const
        {
            return __pnodeTail;
        }

        T* pobjFind(const TString& strName, const TString& strNameSpace)
        {
            TRTListNode<T>* pnodeCur = __pnodeHead;

            if (!pnodeCur)
                return 0;

            // Calculate the hash
            tCIDLib::THashVal hshName = strName.hshCalcHash
            (
                kCIDTracer::c4HashModulus
            );

            tCIDLib::THashVal hshSpace = strNameSpace.hshCalcHash
            (
                kCIDTracer::c4HashModulus
            );

            // Run through the nodes and find the object
            while (pnodeCur)
            {
                // If the hash matches, check the name
                if ((pnodeCur->__hshName == hshName)
                &&  (pnodeCur->__hshNameSpace == hshSpace))
                {
                    if (pnodeCur->__strName.bSameText(strName)
                    &&  pnodeCur->__strNameSpace.bSameText(strNameSpace))
                    {
                        return pnodeCur->__pobjData;
                    }
                }

                // Go do the next node
                pnodeCur = pnodeCur->pnodeNext();
            }
            return 0;
        }

        T* pobjHead() const
        {
            return __pnodeHead->__pobjData;
        }

        T* pobjPop()
        {
            //
            //  We need to pull out the head node and return it to the
            //  caller. If the list is empty, just return 0.
            //
            if (!__pnodeHead)
                return 0;

            // Pull out the head by making its next node the head
            TRTListNode<T>* pnodeRet = __pnodeHead;
            __pnodeHead = __pnodeHead->__pnodeNext;

            // If this zeroed the head, then we are empty so zero the tail
            if (!__pnodeHead)
                __pnodeTail = 0;

            //
            //  Get the object out of the node which we chose. Then set the
            //  object pointer to 0 and delete the node. If we did not zero
            //  the node, it would delete the object.
            //
            T* pobjRet = pnodeRet->__pobjData;
            pnodeRet->__pobjData = 0;
            delete pnodeRet;

            // And now return the object pointer
            return pobjRet;
        }

        T* pobjTail() const
        {
            return __pnodeTail->__pobjData;
        }

        tCIDLib::TVoid Push(TRTListNode<T>* const pnodeToAdd)
        {
            Add(pnodeToAdd);
        }

        tCIDLib::TVoid Push(        T* const    pobjToAdd
                            , const TString&    strName = NUL_TString
                            , const TString&    strNameSpace = NUL_TString)
        {
            Add(pobjToAdd, strName, strNameSpace);
        }


    private :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4NodeCount
        //      The number of nodes currently in the list.
        //
        //  __eDelMode
        //      This is the default delete mode for the mode. It controls
        //      where the object in a node is deleted along with the node,
        //      if the caller does not indicate a specific preference.
        //
        //  __pnodeHead
        //  __pnodeTail
        //      A pointer to the first and last nodes.
        // --------------------------------------------------------------------
        tCIDLib::TCard4         __c4NodeCount;
        tCIDLib::EDeleteModes   __eDelMode;
        TRTListNode<T>*         __pnodeHead;
        TRTListNode<T>*         __pnodeTail;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TRTList<T>,TObject)
};

template <class T> const TClass TRTList<T>::clsThis;

#pragma pack(pop)



// -----------------------------------------------------------------------------
//  Typedef all of the needed versions of the node class and the list class.
//  We also export the instantiation of each list.
// -----------------------------------------------------------------------------
typedef TRTListNode<TFRGBClr>       TRTClrNode;
typedef TRTList<TFRGBClr>           TRTClrList;

typedef TRTListNode<T3DVector>      TVectorNode;
typedef TRTList<T3DVector>          TVectorList;
