//
// NAME: CIDTracer_Light.Hpp
//
// DESCRIPTION: 
//
//  This is the header file for the CIDTracer_Light.Cpp module. This module
//  module implements the TLightRay and TLightSource classes.
//
//  The TLightRay class provides all of the smarts to handle ray calculations.
//  It does not handle ray/object intersections though. Each object class
//  understands how to check its intersection with a ray (and to calculate
//  the surface normal and angle of reflection vectors thereof.)
//
//  The TLightSource class implements a light source. It is a very simple
//  derivative of the TTRTObject class, which is the base for all objects in
//  the scene. The light source just needs a vector to indicate its center
//  point, since light sources are considered to be perfect point sources.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 03/05/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: TLightRay
// PREFIX: lray
// -----------------------------------------------------------------------------
class CIDTRACEREXP TLightRay :

    public TObject, public MStreamable, public MDuplicable, public MFormattable
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and Destructors
        // ---------------------------------------------------------------------
        TLightRay();

        TLightRay
        (
            const   TLightRay&              lrayToCopy
        );

        TLightRay
        (
            const   T3DVector&              vecDir
            , const T3DVector&              vecOrg
        );

        ~TLightRay();


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TLightRay& operator=
        (
            const   TLightRay&              lrayToAssign
        );


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bCalcRefractedRay
        (
            const   T3DVector&              vecPoint
            , const T3DVector&              vecNormal
            , const tCIDLib::TFloat8        f8RefIndex1
            , const tCIDLib::TFloat8        f8RefIndex2
            ,       TLightRay&              lrayTarget
        )   const;

        tCIDLib::TCard4 c4EnteredTextures() const;

        tCIDLib::TVoid CalcRayPoint
        (
            const   tCIDLib::TFloat8        f8Mag
            ,       T3DVector&              vecTarget
        )   const;

        tCIDLib::TVoid CalcReflectedRay
        (
            const   T3DVector&              vecPoint
            , const T3DVector&              vecNormal
            ,       TLightRay&              lrayTarget
        )   const;

        tCIDLib::TVoid EnterTexture
        (
                    TRTTexture* const       ptxtrToEnter
        );

        tCIDLib::TVoid SetVectors
        (
            const   T3DVector&              vecOrg
            , const T3DVector&              vecDir
        );

        TRTTexture& txtrExitTexture();

        TRTTexture& txtrTop() const;

        const T3DVector& vecDir
        (
            const   T3DVector&              vecNew
        );

        const T3DVector& vecDir() const;

        const T3DVector& vecDir2() const;

        const T3DVector& vecOrg
        (
            const   T3DVector&              vecNew
        );

        const T3DVector& vecOrg() const;

        const T3DVector& vecOrg2() const;

        const T3DVector& vecOrgDir() const;

        const T3DVector& vecMixOrgDir() const;

        const T3DVector& vecMixDir() const;

        const T3DVector& vecMixOrg() const;


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private :
        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __prtlstTxStack
        //      This is a list of textures that the ray has refracted through.
        //      It is used in the stack scheme. It is required to track the
        //      refractive indexes of the entering and exiting textures. It
        //      just references its textures, so its a 'no delete' mode
        //      list.
        //
        //      This guy is only allocated it if used, since some uses of
        //      this class don't require it. So its a mutable member.
        //
        //  __vecOrg
        //  __vecOrg2
        //      The origin of the ray in 3D space, and its squared term
        //      sibling.
        //
        //  __vecDir
        //  __vecDir2
        //      The direction/magnitude of the ray, and its squared term
        //      sibling.
        // ---------------------------------------------------------------------
        mutable TRTTxtrList*    __prtlstTxStack;

        T3DVector               __vecOrg, __vecOrg2;
        T3DVector               __vecDir, __vecDir2;

        T3DVector               __vecOrgDir, __vecMixOrgDir;
        T3DVector               __vecMixOrg, __vecMixDir;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TLightRay,TObject)
        DefPolyDup(TLightRay)
};



// -----------------------------------------------------------------------------
//  CLASS: TLightSource
// PREFIX: lsrc
// -----------------------------------------------------------------------------
class CIDTRACEREXP TLightSource : public TRTObject
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and Destructors
        // ---------------------------------------------------------------------
        TLightSource();

        TLightSource
        (
            const   TLightSource&           lsrcSrc
        );

        TLightSource
        (
            const   T3DVector&              vecCenter
            , const TFRGBClr&               frgbColor
        );

        ~TLightSource();


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TLightSource& operator=
        (
            const   TLightSource&           lsrcSrc
        );


        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid AppendTexture
        (
                    TRTTexture* const       ptxtrAppend
            , const TString&                strName = NUL_TString
            , const TString&                strNameSpace = NUL_TString
        );

        tCIDLib::TVoid Invert();

        tCIDLib::TVoid ParseMethod
        (
                    TParseInfo&             prsiSource
        );

        tCIDLib::TVoid PrependTexture
        (
                    TRTTexture* const       ptxtrPrepend
            , const TString&                strName = NUL_TString
            , const TString&                strNameSpace = NUL_TString
        );

        tCIDLib::TVoid Transform
        (
            const   TTransform&             trnsApply
        );

        tCIDLib::TVoid TransformTextures
        (
            const   TTransform&             trnsApply
        );

        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TFloat8 f8MakeLightRay
        (
            const   T3DVector&              vecPoint
            ,       TLightRay&              lrayTarget
        )   const;

        const TFRGBClr& frgbClr() const;

        const TFRGBClr& frgbClr
        (
            const   TFRGBClr&               frgbNewClr
        );

        const T3DVector& vecCenter() const;

        const T3DVector& vecCenter
        (
            const   T3DVector&              vecNewSrc
        );


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _AllIntersects
        (
            const   TLightRay&              lrayTest
            ,       TIntersectArray&        intraIntersects
        )   const;

        tCIDLib::TBoolean _bPointInObj
        (
            const   T3DVector&              vecPoint
        )   const;

        tCIDLib::TBoolean _bTestIntersect
        (
            const   TLightRay&              lrayTest
            ,       tCIDLib::TFloat8&       f8Dist1
            ,       tCIDLib::TFloat8&       f8Dist2
        )   const;

        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _Rotate
        (
            const   T3DVector&              vecRotate
            , const TTransform&             trnsRotate
        );

        tCIDLib::TVoid _Scale
        (
            const   T3DVector&              vecScale
            , const TTransform&             trnsScale
        );

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;

        tCIDLib::TVoid _Translate
        (
            const   T3DVector&              vecTrans
            , const TTransform&             trnsTranslate
        );


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean operator==(const TLightSource&) const;

        tCIDLib::TBoolean operator!=(const TLightSource&) const;


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __frgbClr
        //      The color of the light.
        //
        //  __vecCenter
        //      The vector that represents the center point of the light
        //      source.
        // ---------------------------------------------------------------------
        TFRGBClr                __frgbClr;
        T3DVector                __vecCenter;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TLightSource,TRTObject)
        DefPolyDup(TLightSource)
};

#pragma pack(pop)


// -----------------------------------------------------------------------------
//  TLightRay: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline const T3DVector& TLightRay::vecDir() const
{
    return __vecDir;
}

inline const T3DVector& TLightRay::vecDir2() const
{
    return __vecDir2;
}

inline const T3DVector& TLightRay::vecOrg() const
{
    return __vecOrg;
}

inline const T3DVector& TLightRay::vecOrg2() const
{
    return __vecOrg2;
}

inline const T3DVector& TLightRay::vecOrgDir() const
{
    return __vecOrgDir;
}

inline const T3DVector& TLightRay::vecMixOrgDir()const
{
    return __vecMixOrgDir;
}

inline const T3DVector& TLightRay::vecMixDir() const
{
    return __vecMixDir;
}

inline const T3DVector& TLightRay::vecMixOrg() const
{
    return __vecMixOrg;
}




// -----------------------------------------------------------------------------
//  TLightSource: Constructors and destructors
// -----------------------------------------------------------------------------
inline TLightSource::~TLightSource()
{
}


// -----------------------------------------------------------------------------
//  TLightSource: Public, inherited methods
// -----------------------------------------------------------------------------
inline tCIDLib::TVoid TLightSource::Invert()
{}

inline tCIDLib::TVoid
TLightSource::TransformTextures(const TTransform& trnsApply)
{
}

// -----------------------------------------------------------------------------
//  TLightSource: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline const TFRGBClr& TLightSource::frgbClr() const
{
    return __frgbClr;
}

inline const
TFRGBClr& TLightSource::frgbClr(const TFRGBClr& frgbNewClr)
{
    __frgbClr = frgbNewClr;
    return __frgbClr;
}

inline const T3DVector& TLightSource::vecCenter() const
{
    return __vecCenter;
}

inline const T3DVector& TLightSource::vecCenter(const T3DVector& vecNewSrc)
{
    __vecCenter = vecNewSrc;
    return __vecCenter;
}


// -----------------------------------------------------------------------------
//  TLightSource: Protected, inherited methods
// -----------------------------------------------------------------------------
inline tCIDLib::TVoid
TLightSource::_AllIntersects(const TLightRay&, TIntersectArray&) const
{
}

inline tCIDLib::TBoolean TLightSource::_bPointInObj(const T3DVector&) const
{
    return kCIDLib::False;
}

inline tCIDLib::TBoolean
TLightSource::_bTestIntersect(  const   TLightRay&
                                ,       tCIDLib::TFloat8&
                                ,       tCIDLib::TFloat8&) const
{
    return kCIDLib::False;
}

inline tCIDLib::TVoid TLightSource::_Scale(const T3DVector&, const TTransform&)
{
}
