//
// NAME: CIDTracer_Intersect.Cpp
//
// DESCRIPTION: 
//
//  This module implements the basic intersection class, which describes
//  a light ray's intersection with an object, and the intersection array
//  class which is a specialized container for accumulating all of the
//  intersections made by a ray.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 10/31/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros for our classes
// -----------------------------------------------------------------------------
RTTIData(TIntersect,TObject)
RTTIData(TIntersectArray,TObject)


// -----------------------------------------------------------------------------
//  Local constant data
//
//  __c4ArrayChunk
//      This is the size of the chunks by which the intersect array is grown.
//      It is initially given this many elements, then grown by this many
//      when needed. Its basically 1K pointers, which is a page of memory.
//
//  __c4TotalCachePages
//      The total number of cache pages that the cache can hold. At max it
//      will use 64K of memory. If these are all used, then the image is
//      very complex or many threads are rendering, so the memory usage is
//      justified in rendering time.
// -----------------------------------------------------------------------------
const   tCIDLib::TCard4 __c4ArrayChunk          = 1024;
const   tCIDLib::TCard4 __c4TotalCachePages     = 16;


// -----------------------------------------------------------------------------
//  Local data
//
//  __apac1PageCache
//      To greatly speed up the rendering process, we cache a set of 'chunks'
//      here. The TIntersectArray class starts with a chunk of __c4ArrayChunk
//      pointers and then expands by that same size. These are used to provide
//      the initial chunk needed during the constructor. Since most scenes are
//      not complex enough to ever need more, this can have a significant
//      effect. There is no need to worry about the actual type of these
//      buffers so they are just raw byte arrays here.
//
//  __c4PageUsedFlags
//      This flag contains bits for each of the possible cached pages and
//      indicates whether it is in use or not.
//
//  __mtxSync
//      This is a semaphore that is used to control access to the cache list.
//      It is created during the module init.
// -----------------------------------------------------------------------------
static  tCIDLib::TCard1*    __apac1PageCache[__c4TotalCachePages];
static  tCIDLib::TCard4     __c4PageUsedFlags;
static  TMutex              __mtxSync;


// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __SortArray
//
// DESCRIPTION:
//
//  This is a recursive quick sort algorithm that is specialized for the
//  intersection array class.
// ---------------------------------------
//   INPUT: papintrArray is the pointer to the array of node pointers to sort.
//          i4Start, i4End are the 'divide and conquer' start points for this
//              level of recursion.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid __SortArray(         TIntersect*     papintrArray[]
                            , const tCIDLib::TInt4  i4Start
                            , const tCIDLib::TInt4  i4End)
{
    if (i4End <= i4Start)
        return;

    tCIDLib::TFloat8    f8Dist;
    tCIDLib::TInt4      i4I, i4J;
    TIntersect*         pintrTmp;

    // Get the distance we are comparing against
    f8Dist = papintrArray[i4End]->f8Distance;

    i4I = i4Start-1;
    i4J = i4End;
    while (1)
    {
        while (papintrArray[++i4I]->f8Distance < f8Dist);

        while (1)
        {
            i4J--;
            if (i4J < 0)
                break;

            if (papintrArray[i4J]->f8Distance <= f8Dist)
                break;
        }

        if (i4I >= i4J)
            break;

        pintrTmp = papintrArray[i4I];
        papintrArray[i4I] = papintrArray[i4J];
        papintrArray[i4J] = pintrTmp;
    }

    pintrTmp = papintrArray[i4I];
    papintrArray[i4I] = papintrArray[i4End];
    papintrArray[i4End] = pintrTmp;

    //
    //  Call ourself recursively for the sub-areas on either side of
    //  the split.
    //
    __SortArray(papintrArray, i4Start, i4I-1);
    __SortArray(papintrArray, i4I+1, i4End);
}



// -----------------------------------------------------------------------------
//  CLASS: TIntersect
// PREFIX: intr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TIntersect: Constructors and Destructors
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TIntersect
//
// DESCRIPTION:
//
//  The primary constructor is used to save the results of an object
//  intersection. The caller provides us with the ray that hit the object,
//  the object struck, and the distance calculated in the hit testing.
//  We save the distance and a pointer to the object struck, then calculate
//  the actual position vector for the intersection.
//
//  The copy constructor just copies over the source intersection's stuff.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
TIntersect::TIntersect( const   TLightRay&              lrayTest
                        , const TRTGeometricObj* const  prtobjIntersect
                        , const tCIDLib::TFloat8        f8Dist) :
    TObject()
    , f8Distance(f8Dist)
    , prtobjInter(prtobjIntersect)
    , vecIntersect(0, 0, 0)
{
    lrayTest.CalcRayPoint(f8Dist, vecIntersect);
}

TIntersect::TIntersect(const TIntersect& intrSrc) :

    TObject(intrSrc)
    , f8Distance(intrSrc.f8Distance)
    , prtobjInter(intrSrc.prtobjInter)
    , vecIntersect(intrSrc.vecIntersect)
{
}


// -----------------------------------------------------------------------------
//  TIntersect: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: operator=
//
// DESCRIPTION:
//
//  This method handles the assignment of one intersection to another.
// ---------------------------------------
//   INPUT: intrSrc is the source intersection object.
//
//  OUTPUT: None
//
//  RETURN: A reference to this object.
//
TIntersect& TIntersect::operator=(const TIntersect& intrSrc)
{
    if (this == &intrSrc)
        return *this;

    f8Distance      = intrSrc.f8Distance;
    prtobjInter     = intrSrc.prtobjInter;
    vecIntersect    = intrSrc.vecIntersect;

    return *this;
}


// -----------------------------------------------------------------------------
//  TIntersect: Protected, inherited methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _eSortComp
//
// DESCRIPTION:
//
//  This method will do a sort comparison of this object to the target
//  object. For this class, the sort field is the distance.
// ---------------------------------------
//   INPUT: objTarget is the object to compare against.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::ESortComps
TIntersect::_eSortComp(const TObject& objTarget) const
{
    TIntersect* const intrTmp = (TIntersect*)&objTarget;

    if (f8Distance > intrTmp->f8Distance)
        return tCIDLib::ESort_FirstGreater;
    else if (f8Distance < intrTmp->f8Distance)
        return tCIDLib::ESort_FirstLess;

    return tCIDLib::ESort_Equal;
}



// -----------------------------------------------------------------------------
//  CLASS: TIntersectArray
// PREFIX: intra
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TIntersectArray: Constructors and Destructors
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TIntersect
//                       ~TIntersect
//
// DESCRIPTION:
//
//  Just inits the data members. Starts off with 1024 elements, which is
//  probably fine for most of the time, but it will reallocated if needed. It
//  will take an initial page from the cache if one exists.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
TIntersectArray::TIntersectArray() :

    __c4CacheEntry(kCIDLib::c4MaxCard)
    , __c4CurHits(0)
    , __c4Elements(__c4ArrayChunk)
    , __papintrHits(0)
{
    //
    //  Get the initial chunk. This guy will either get it from the cache
    //  or allocate a new one if needed. If its a cached page, then the
    //  __c4CacheEntry field will be set to that page's cache slot.
    //
    __AllocArray();
}

TIntersectArray::~TIntersectArray()
{
    // First free all of the intersect objects
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4CurHits; c4Ind++)
        delete __papintrHits[c4Ind];

    //
    //  Free the array of pointers, literally or just back to the page
    //  cache as the case may be.
    //
    __FreeArray();
}


// -----------------------------------------------------------------------------
//  TIntersectArray: Public operators
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: operator[]
//
// DESCRIPTION:
//
//  Returns a pointer ot the intersect object at the passed index. Does index
//  checking if in debug mode.
// ---------------------------------------
//   INPUT: c4Index is the index of the element to return.
//
//  OUTPUT: None
//
//  RETURN: A pointer to the intersect object at the indicated index.
//
TIntersect* TIntersectArray::operator[](const tCIDLib::TCard4 c4Index)
{
    #if CID_DEBUG_ON
    if (c4Index >= __c4CurHits)
    {
        facCIDTracer.LogErr
        (
            __FILE__
            , __LINE__
            , kTracerErrs::errcIntr_IndexError
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TCardinal(c4Index)
            , TCardinal(__c4CurHits)
        );
    }
    #endif

    return __papintrHits[c4Index];
}



// -----------------------------------------------------------------------------
//  TIntersectArray: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: Add
//
// DESCRIPTION:
//
//  Adds the passed intersection object to the array at the end.
// ---------------------------------------
//   INPUT: pintrToAdd is the intersect object to add.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TIntersectArray::Add(TIntersect* const pintrToAdd)
{
    // See if we need to expand the array
    if (__c4CurHits == __c4Elements)
    {
        // Save the current array
        TIntersect** papintrTmp = __papintrHits;

        // Allocate a new array with an extra chunk
        __papintrHits = new TIntersect*[__c4Elements+__c4ArrayChunk];

        // Copy over the existing stuff
        TRawMem::CopyMemBuf
        (
            __papintrHits
            , papintrTmp
            , sizeof(tCIDLib::TVoid*) * __c4Elements
        );

        // Bump up the element count now
        __c4Elements += __c4ArrayChunk;
    }

    // And add the new element and bump up the hit count
    __papintrHits[__c4CurHits++] = pintrToAdd;
}


//
// FUNCTION/METHOD NAME: Flush
//
// DESCRIPTION:
//
//  Flushes the intersect objects out of the array and sets its current hit
//  count to 0;
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TIntersectArray::Flush()
{
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4CurHits; c4Ind++)
    {
        delete __papintrHits[c4Ind];
        __papintrHits[c4Ind] = 0;
    }

    __c4CurHits = 0;
}


//
// FUNCTION/METHOD NAME: Sort
//
// DESCRIPTION:
//
//  Does a local, specialized sort of the array of intersection pointers. It
//  is in ascending order by distance.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TIntersectArray::Sort()
{
    if (__c4CurHits <= 1)
        return;

    if (__c4CurHits == 2)
    {
        if (__papintrHits[0]->f8Distance > __papintrHits[1]->f8Distance)
        {
            TIntersect* pintrTmp = __papintrHits[0];
            __papintrHits[0] = __papintrHits[1];
            __papintrHits[1] = pintrTmp;
        }
        return;
    }

    __SortArray(__papintrHits, 0, __c4CurHits-1);
}


// -----------------------------------------------------------------------------
//  TIntersectArray: Private, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __AllocArray
//
// DESCRIPTION:
//
//  This guy will get an initial chunk of intersection pointers for this
//  object. It will either get it from the cached list of pages or allocate
//  a new one if needed.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TIntersectArray::__AllocArray()
{
    //
    //  Lock the sync semaphore and see if we can get a cached page for
    //  the initial set of pointers.
    //
    {
        TLocker lockAlloc(&__mtxSync, 1000);

        for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4TotalCachePages; c4Ind++)
        {
            //
            //  If we find one whose flag is not set and which has an
            //  allocated page available, take it.
            //
            if (!(__c4PageUsedFlags & (0x1UL << c4Ind))
            &&  __apac1PageCache[c4Ind])
            {
                break;
            }
        }

        if (c4Ind < __c4TotalCachePages)
        {
            // We got one, so take it and set its flag and save its index
            __papintrHits = (TIntersect**)__apac1PageCache[c4Ind];
            __c4PageUsedFlags |= (0x1UL << c4Ind);
            __c4CacheEntry = c4Ind;
        }
    }

    // If we did not get a cache hit, then make one
    if (!__papintrHits)
        __papintrHits = new TIntersect*[__c4ArrayChunk];
}


//
// FUNCTION/METHOD NAME: __FreeArray
//
// DESCRIPTION:
//
//  This method is called to free the array of intersection pointers. This
//  method decides if it is from the cache, can be added to the cache, or
//  just needs to be freed.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TIntersectArray::__FreeArray()
{
    // If its not chunk sized, then it not from the cache definitely
    if (__c4Elements != __c4ArrayChunk)
    {
        delete __papintrHits;
        __papintrHits = 0;
        __c4CacheEntry = kCIDLib::c4MaxCard;
        return;
    }

    // Lock the cache
    TLocker lockFree(&__mtxSync, 1000);

    //
    //  The array being freed was already from the cache. So we just need
    //  to mark this guy as free again and we are done.
    //
    if (__c4CacheEntry != kCIDLib::c4MaxCard)
    {
        __c4PageUsedFlags &= ~(0x1UL << __c4CacheEntry);
        __c4CacheEntry = kCIDLib::c4MaxCard;
        __papintrHits = 0;
        return;
    }

    //
    //  It is chunk sized and not in the cache currently, so find an empty
    //  slot (if there is one) to put it in.
    //
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4TotalCachePages; c4Ind++)
    {
        if (!__apac1PageCache[c4Ind])
            break;
    }

    //
    //  If we find a slot, then put this page into it, else we just have
    //  to delete the array because we have no where to put it.
    //
    if (c4Ind == __c4TotalCachePages)
    {
        delete __papintrHits;
    }
     else
    {
        __apac1PageCache[c4Ind] = (tCIDLib::TCard1*)__papintrHits;

        // Bump up the count of cache pages
        _pmtrgTracerCore->OffsetCard(tCIDTracer_::eCorEMetric_IntersectPages, 1);
    }

    //
    //  Zero out the pointer now because it either belongs to the cache or
    //  has been freed.
    //
    __papintrHits = 0;
}
