//
// NAME: CIDTracer_CSG.Cpp
//
// DESCRIPTION: 
//
//  This module implements the TRTSolidGeom class, which stands for
//  Constructive Solid Geometry. It is a very simple, but very powerful,
//  technique for creating complex shapes from multiple simpler objects.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/02/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData2(TRTSolidGeom,TRTMetaObj)



// -----------------------------------------------------------------------------
//  CLASS: TRTSolidGeom
// PREFIX: csg
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TRTSolidGeom: Constructors and Destructors
// -----------------------------------------------------------------------------

TRTSolidGeom::TRTSolidGeom() :

    TRTMetaObj()
    , __eOperation(tCIDTracer::ECSGOp_Union)
{
}

TRTSolidGeom::TRTSolidGeom(const TRTSolidGeom& csgSrc) :

    TRTMetaObj(csgSrc)
    , __eOperation(csgSrc.__eOperation)
{
}

TRTSolidGeom::~TRTSolidGeom()
{
}

// -----------------------------------------------------------------------------
//  TRTSolidGeom: Public operators
// -----------------------------------------------------------------------------

TRTSolidGeom&  TRTSolidGeom::operator=(const TRTSolidGeom& csgSrc)
{
    if (this == &csgSrc)
        return *this;

    // Copy our parent's members
    TParent::operator=(csgSrc);

    // Copy this class' members
    __eOperation = csgSrc.__eOperation;

    return *this;
}


// -----------------------------------------------------------------------------
//  TRTSolidGeom: Public, Inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid  TRTSolidGeom::Invert()
{
    // Call our parent
    TParent::Invert();

    if (__eOperation == tCIDTracer::ECSGOp_Intersection)
        __eOperation = tCIDTracer::ECSGOp_Union;
     else if (__eOperation == tCIDTracer::ECSGOp_Union)
        __eOperation = tCIDTracer::ECSGOp_Intersection;
}


tCIDLib::TVoid  TRTSolidGeom::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens     eToken;
    TString                 strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    if (eToken == TParseInfo::eToken_Object)
    {
        prsiSrc.UnGet(strToken);
        TParent::ParseMethod(prsiSrc);
        return;
    }

    // Must be followed by an open parens
    prsiSrc.CheckOpenParen();

    // Deal with the valid tokens at this level
    if (eToken == TParseInfo::eToken_CSGOperation)
    {
        TString   strName;

        // The next token should be the name of the operation
        prsiSrc.GetName(strName);

        if (strName == L"Intersect")
            eOperation(tCIDTracer::ECSGOp_Intersection);
         else if (strName == L"Difference")
            eOperation(tCIDTracer::ECSGOp_Difference);
         else if (strName == L"Union")
            eOperation(tCIDTracer::ECSGOp_Union);
         else
            prsiSrc.ThrowErr(kTracerErrs::errcCSG_BadOperation, strName);
    }
     else
    {
        // Unget the token and pass it to our parent
        prsiSrc.UnGet(L"(");
        prsiSrc.UnGet(strToken);
        TParent::ParseMethod(prsiSrc);
        return;
    }

    // Has to end with a close paren 
    prsiSrc.CheckCloseParen();
}


// -----------------------------------------------------------------------------
//  TRTSolidGeom: Protected, virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TRTSolidGeom::_AllIntersects(   const TLightRay&   lrayTest
                                , TIntersectArray& intraHits) const
{
    //
    //  According to the type of operation that this CSG object does, test
    //  for intersection with one of our children.
    //
    TRTObjNode* pnodeCur = rtlstChildren().pnodeHead();

    if (!pnodeCur)
        return;

    tCIDLib::TBoolean   bHit;
    TIntersect*         pintrCur;
    TIntersectArray     intraTmp;

    while (pnodeCur)
    {
        // Get a pointer to this object
        TRTObject* const prtobjTmp = pnodeCur->pobjData();

        //
        //  Get a list of the intersections underneath this object. If
        //  there are any intersections, then do the test of all the
        //  intersected objects according to the operation.
        //
        intraTmp.Flush();
        prtobjTmp->AllIntersects(lrayTest, intraTmp);

        tCIDLib::TCard4 c4ItemCnt = intraTmp.c4HitCount();
        if (!c4ItemCnt)
        {
            pnodeCur = pnodeCur->pnodeNext();
            continue;
        }

        // Sort the list by distance
        intraTmp.Sort();

        for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4ItemCnt; c4Ind++)
        {
            pintrCur = intraTmp[c4Ind];

            bHit = kCIDLib::True;
            if ((__eOperation == tCIDTracer::ECSGOp_Intersection)
            ||  (__eOperation == tCIDTracer::ECSGOp_Difference))
            {
                bHit = __bIntersectTest
                (
                    pintrCur->prtobjInter
                    , pintrCur->vecIntersect
                );
            }

            // If a hit, then copy the intersection to the target list
            if (bHit)
                intraHits.Add(::pDupObject<TIntersect>(*pintrCur));
        }

        // Move to the next node
        pnodeCur = pnodeCur->pnodeNext();
    }
}


tCIDLib::TBoolean 
TRTSolidGeom::_bPointInObj(const T3DVector& vecPoint) const
{
    tCIDLib::TBoolean  bHit, bRet;

    if ((__eOperation == tCIDTracer::ECSGOp_Intersection)
    ||  (__eOperation == tCIDTracer::ECSGOp_Difference))
    {
        bRet = kCIDLib::True;
    }
     else
    {
        bRet = kCIDLib::False;
    }

    TRTObjNode* pnodeCur = rtlstChildren().pnodeHead();
    while (pnodeCur)
    {
        // Get a pointer to the current object
        TRTObject* const prtobjCur = pnodeCur->pobjData();

        bHit = prtobjCur->bPointInObj(vecPoint);

        if ((__eOperation == tCIDTracer::ECSGOp_Intersection)
        ||  (__eOperation == tCIDTracer::ECSGOp_Difference))
        {
            if (!bHit)
            {
                bRet = kCIDLib::False;
                break;
            }
        }
         else if (__eOperation == tCIDTracer::ECSGOp_Union)
        {
            if (bHit)
                break;
        }

        // Move up to the next node
        pnodeCur = pnodeCur->pnodeNext();
    }
    return bRet;
}


tCIDLib::TBoolean  
TRTSolidGeom::_bTestIntersect(  const   TLightRay&          lrayTest
                                ,       tCIDLib::TFloat8&   f8Dist1
                                ,       tCIDLib::TFloat8&   f8Dist2) const
{
    TIntersectArray intraTmp;

    // See if any objects are intersected
    _AllIntersects(lrayTest, intraTmp);

    tCIDLib::TCard4  c4Cnt = intraTmp.c4HitCount();
    if (!c4Cnt)
        return kCIDLib::False;

    // Sort by the distance
    intraTmp.Sort();

    // Return the distance of the smallest distance
    TIntersect* pintrTmp = intraTmp[0];
    f8Dist1 = pintrTmp->f8Distance;

    if (f8Dist1 < kCIDTracer_::f8SmallTolerance)
        return kCIDLib::False;

    if (c4Cnt > 1)
    {
        // Do the tail as the exit point
        pintrTmp = intraTmp[c4Cnt-1];
        f8Dist2 = pintrTmp->f8Distance;

        if (f8Dist2 < kCIDTracer_::f8SmallTolerance)
            f8Dist2 = f8Dist1;
    }
     else
    {
        f8Dist2 = f8Dist1;
    }

    return kCIDLib::True;
}


tCIDLib::TVoid  TRTSolidGeom::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the attribute info into the string
    TParent::_FormatTo(strmToWriteTo);
    strmToWriteTo << L", Operation: " << __eOperation;
}

tCIDLib::TVoid  TRTSolidGeom::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Call our parent first
    TParent::_StreamFrom(strmToReadFrom);

    // Parse out our CSG operation
    strmToReadFrom >> __eOperation;
}

tCIDLib::TVoid 
TRTSolidGeom::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent first
    TParent::_StreamTo(strmToWriteTo);

    // Add our CSG operation
    strmToWriteTo << __eOperation;
}



// -----------------------------------------------------------------------------
//  TRTSolidGeom: Private, virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __NewObject
//
// DESCRIPTION:
//
//  We override this guy to process new child objects being added. For all
//  objects after the first one, we invert the object.
// ---------------------------------------
//   INPUT: prtobjNew is the new object being added
//          c4ObjCount is the count of objects before this one is added.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TRTSolidGeom::__NewObject(          TRTObject* const    prtobjNew
                            , const tCIDLib::TCard4     c4ObjCount)
{
    if (c4ObjCount && (__eOperation == tCIDTracer::ECSGOp_Difference))
        prtobjNew->Invert();
}


// -----------------------------------------------------------------------------
//  TRTSolidGeom: Private, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __bIntersectTest
//
// DESCRIPTION:
//
//  This method tests all of the child objects of this CSG object to see
//  if they are intersected. If all are not intersected, then we return
//  an eFalse, because intersections require all object to be intersected.
// ---------------------------------------
//   INPUT: prtobjSkip is the object to skip in the test
//          vecTest is the point to test
//
//  OUTPUT: None
//
//  RETURN: eTrue if it passed, else eFalse.
//
tCIDLib::TBoolean
TRTSolidGeom::__bIntersectTest( const   TRTGeometricObj* const  prtobjSkip
                                , const T3DVector&               vecTest) const
{
    TRTObjNode* pnodeCur = rtlstChildren().pnodeHead();

    while (pnodeCur)
    {
        TRTObject* const prtobjCur = pnodeCur->pobjData();
        if ((tCIDLib::TVoid*)prtobjCur != (tCIDLib::TVoid*)prtobjSkip)
        {
            // If the point is not in all objects, then return false
            if (!prtobjCur->bPointInObj(vecTest))
                return kCIDLib::False;
        }
        pnodeCur = pnodeCur->pnodeNext();
    }
    return kCIDLib::True;
}
