//
// NAME: CIDTracer_Box.Cpp
//
// DESCRIPTION:
//
//  This module implements the TRTBox class, which is a very specialized
//  CSG intersection of 6 planes. 6 raw normals and distances are maintained,
//  unlike a CSG box that manages 6 individual plane objects in a relatively
//  unoptimized way because it is a very general purpose class. Since boxes
//  make good bounding objects, this is well worth the effort.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 03/05/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  This class addes TBoxIntersect objects to the intersection lists.
//      It has to derive from TIntersect because it needs to have the
//      index of the intersected plane when CalcSurfaceNormal() is later
//      called.
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData(TBoxIntersect,TIntersect)
RTTIData2(TRTBox,TRTGeometricObj)



// -----------------------------------------------------------------------------
//  CLASS: TBoxIntersect
// PREFIX: intr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TBoxIntersect: Constructors and Destructors
// -----------------------------------------------------------------------------

TBoxIntersect::TBoxIntersect(   const   TLightRay&              lrayTest
                                , const TRTGeometricObj* const  prtobjIntersect
                                , const tCIDLib::TFloat8        f8Dist
                                , const tCIDLib::TCard4         c4PlaneIndex) :

    TIntersect(lrayTest, prtobjIntersect, f8Dist)
    , c4PlaneInd(c4PlaneIndex)
{
}

TBoxIntersect::TBoxIntersect(const TBoxIntersect& intrSrc) :

    TIntersect(intrSrc)
    , c4PlaneInd(intrSrc.c4PlaneInd)
{
}


// -----------------------------------------------------------------------------
//  TBoxIntersect: Public, non-virtual methods
// -----------------------------------------------------------------------------

TBoxIntersect& TBoxIntersect::operator=(const TBoxIntersect& intrSrc)
{
    // Check for self assignment
    if (this == &intrSrc)
        return *this;

    // Call our parent first
    TParent::operator=(intrSrc);

    // Then copy over our plane index
    c4PlaneInd = intrSrc.c4PlaneInd;

    return *this;
}



// -----------------------------------------------------------------------------
//  CLASS: TRTBox
// PREFIX: box
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TRTBox: Constructors and Destructors
// -----------------------------------------------------------------------------

TRTBox::TRTBox()
{
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
        __af8Distances[c4Ind] = -0.5;

    __avecNormals[0] = T3DVector(-1.0,  0.0,  0.0);
    __avecNormals[1] = T3DVector( 1.0,  0.0,  0.0);
    __avecNormals[2] = T3DVector( 0.0, -1.0,  0.0);
    __avecNormals[3] = T3DVector( 0.0,  1.0,  0.0);
    __avecNormals[4] = T3DVector( 0.0,  0.0, -1.0);
    __avecNormals[5] = T3DVector( 0.0,  0.0,  1.0);
}

TRTBox::TRTBox(const TRTBox& boxSrc) :

    TRTGeometricObj(boxSrc)
{
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
    {
        __avecNormals[c4Ind] = boxSrc.__avecNormals[c4Ind];
        __af8Distances[c4Ind] = boxSrc.__af8Distances[c4Ind];
    }
}


// -----------------------------------------------------------------------------
//  TRTBox: Public operators
// -----------------------------------------------------------------------------

TRTBox&  TRTBox::operator=(const TRTBox& boxSrc)
{
    // Check for self assignment
    if (this == &boxSrc)
        return *this;

    // Do our parent first
    TParent::operator=(boxSrc);

    // Then copy over the info for each plane
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
    {
        __avecNormals[c4Ind] = boxSrc.__avecNormals[c4Ind];
        __af8Distances[c4Ind] = boxSrc.__af8Distances[c4Ind];
    }
    return *this;
}


// -----------------------------------------------------------------------------
//  TRTBox: Public, Inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TRTBox::CalcSurfaceNormal(  const   TIntersect* const   pintrSrc
                            ,       T3DVector&          vecNormal) const
{
    #if CID_DEBUG_ON
    if (!pintrSrc->bIsA(L"TBoxIntersect"))
    {
        facCIDTracer.LogMsg
        (
            __FILE__
            , __LINE__
            , L"Got an intersection of type %(1)"
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , pintrSrc->clsIsA()
        );
    }
    #endif

    // Cast to a box intersect
    TBoxIntersect* const pintrBox = (TBoxIntersect*)pintrSrc;

    vecNormal = __avecNormals[pintrBox->c4PlaneInd];
}

tCIDLib::TVoid  TRTBox::Invert()
{
    TParent::Invert();

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
    {
        __avecNormals[c4Ind].Negate();
        __af8Distances[c4Ind] *= -1.0;
    }
}

tCIDLib::TVoid  TRTBox::Transform(const TTransform& trnsApply)
{
    // Call the parent version
    TParent::Transform(trnsApply);

    // Transform our normals
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
    {
        trnsApply.mtrxTrans.Transform(__avecNormals[c4Ind]);

        //
        //  Make a vector from the distance and transform it.
        //
        T3DVector vecTmp
        (
            __af8Distances[c4Ind]
            , __af8Distances[c4Ind]
            , __af8Distances[c4Ind]
        );
        trnsApply.mtrxTrans.Transform(vecTmp);
        __af8Distances[c4Ind] = vecTmp.f8Magnitude();
    }
}


// -----------------------------------------------------------------------------
//  TRTBox: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TRTBox::_AllIntersects( const   TLightRay&  lrayTest
                        , TIntersectArray&  intraHits) const
{
    tCIDLib::TBoolean   bFound;
    tCIDLib::TFloat8    f8Dist, f8Tmp;
    T3DVector           vecTmp;

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
    {
        if (__bTestIntersect(lrayTest, c4Ind, f8Dist))
        {
            // Calculate the intersection point
            lrayTest.CalcRayPoint(f8Dist, vecTmp);

            bFound = kCIDLib::True;
            for (tCIDLib::TCard4 c4Inner = 0; c4Inner < c4PlaneCount; c4Inner++)
            {
                // Don't check with the plane that was hit originally
                if (c4Inner == c4Ind)
                    continue;

                f8Tmp = vecTmp.f8Dot(__avecNormals[c4Inner]);

                if ((f8Tmp + __af8Distances[c4Inner]) > kCIDTracer_::f8SmallTolerance)
                {
                    bFound = kCIDLib::False;
                    break;
                }
            }

            if (bFound)
            {
                intraHits.Add
                (
                    new TBoxIntersect
                    (
                        lrayTest
                        , this
                        , f8Dist
                        , c4Ind
                    )
                );
            }
        }
    }
}


tCIDLib::TBoolean TRTBox::_bPointInObj(const T3DVector& vecPoint) const
{
    tCIDLib::TFloat8 f8Tmp;

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
    {
        f8Tmp = vecPoint.f8Dot(__avecNormals[c4Ind]);

        if ((f8Tmp + __af8Distances[c4Ind]) > kCIDTracer_::f8SmallTolerance)
            return kCIDLib::False;
    }
    return kCIDLib::True;
}


tCIDLib::TBoolean 
TRTBox::_bTestIntersect(const   TLightRay&          lrayTest
                        ,       tCIDLib::TFloat8&   f8Dist1
                        ,       tCIDLib::TFloat8&   f8Dist2) const
{
    TIntersectArray intraHits;

    // Get all intersections to temp list
    _AllIntersects(lrayTest, intraHits);

    tCIDLib::TCard4  c4Cnt = intraHits.c4HitCount();
    if (!c4Cnt)
        return kCIDLib::False;

    // Sort it by distance
    intraHits.Sort();

    // Make the head the first distance
    TIntersect* pintrCur = intraHits[0];
    f8Dist1 = pintrCur->f8Distance;

    // If the tail is different, then make it distance 2
    f8Dist2 = f8Dist1;
    if (c4Cnt > 1)
    {
        pintrCur = intraHits[c4Cnt-1];

        if (f8Dist1 != pintrCur->f8Distance)
            f8Dist2 = pintrCur->f8Distance;
    }
    return kCIDLib::True;
}


tCIDLib::TVoid  TRTBox::_FormatTo(TTextStream& strmToWriteTo) const
{
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
    {
        strmToWriteTo   << L"Dist: " << __af8Distances[c4Ind]
                        << L", Normal: " << __avecNormals[c4Ind];
    }
}


tCIDLib::TVoid 
TRTBox::_Rotate(const T3DVector&  vecRotate, const TTransform& trnsRotate)
{
    // Call the parent version
    TParent::_Rotate(vecRotate, trnsRotate);

    // Apply it to the plane normals
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
        trnsRotate.mtrxTrans.Transform(__avecNormals[c4Ind]);
}


tCIDLib::TVoid 
TRTBox::_Scale(const T3DVector& vecScale, const TTransform& trnsScale)
{
    // Call the parent version
    TParent::_Scale(vecScale, trnsScale);

    tCIDLib::TFloat8 f8Len;

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
    {
        __avecNormals[c4Ind] /= vecScale;
        f8Len = __avecNormals[c4Ind].f8Magnitude();
        __avecNormals[c4Ind] *= (1.0 / f8Len);
        __af8Distances[c4Ind] /= f8Len;
    }
}


tCIDLib::TVoid  TRTBox::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Call our parent's version
    TParent::_StreamFrom(strmToReadFrom);

    // Get the center point vector out
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
        strmToReadFrom >> __avecNormals[c4Ind];

    // Get the distance out
    strmToReadFrom.ReadFloat8Array(__af8Distances, c4PlaneCount);
}

tCIDLib::TVoid  TRTBox::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent's version
    TParent::_StreamTo(strmToWriteTo);

    // Write out the center point vector
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
        strmToWriteTo << __avecNormals[c4Ind];

    // Write out the distance
    strmToWriteTo.WriteFloat8Array(__af8Distances, c4PlaneCount);
}


tCIDLib::TVoid 
TRTBox::_Translate(const T3DVector& vecTrans, const TTransform& trnsTranslate)
{
    // Call the parent version
    TParent::_Translate(vecTrans, trnsTranslate);

    // Get a vector that will provide the translation
    T3DVector    vecTmp;
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < c4PlaneCount; c4Ind++)
    {
        vecTmp = vecTrans;
        vecTmp *= __avecNormals[c4Ind];

        // And subtract the sum of the magnitudes to get the distance
        __af8Distances[c4Ind] -= vecTmp.f8MagSum();
    }
}


// -----------------------------------------------------------------------------
//  TRTBox: Protected, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __bTestIntersect
//
// DESCRIPTION:
//
//  This method will test the passed ray to see if it intersects the c4Ind'th
//  plane object.
// ---------------------------------------
//   INPUT: lrayTest is the ray to test
//          c4Ind is the index of the plane to test. It is assumed that the
//              caller won't pass anything out of range.
//
//  OUTPUT: f8Dist1 is set to the distance if an intersect
//
//  RETURN: eTrue if an intersect, else eFalse.
//
tCIDLib::TBoolean 
TRTBox::__bTestIntersect(   const   TLightRay&          lrayTest
                            , const tCIDLib::TCard4     c4Ind
                            ,       tCIDLib::TFloat8&   f8Dist1) const
{
    tCIDLib::TFloat8 f8Dir, f8Org;

    // Calc the dot product of the plane and ray origin
    f8Org = __avecNormals[c4Ind].f8Dot(lrayTest.vecOrg());
    f8Org += __af8Distances[c4Ind];
    f8Org *= -1.0;

    f8Dir = __avecNormals[c4Ind].f8Dot(lrayTest.vecDir());
    if ((f8Dir < kCIDTracer_::f8SmallTolerance)
    &&  (f8Dir > - kCIDTracer_::f8SmallTolerance))
    {
        return kCIDLib::False;
    }

    //
    //  Calculate the distance. If less than the small tolerance, then the
    //  object is behind the ray origin.
    //
    f8Dist1 = f8Org / f8Dir;

    if ((f8Dist1 >= kCIDTracer_::f8SmallTolerance)
    &&  (f8Dist1 <= kCIDTracer_::f8MaxDistance))
        return kCIDLib::True;

    return kCIDLib::False;
}
