//
// NAME: CIDTracer_BaseTextures.Cpp
//
// DESCRIPTION:
//
//  This module implements a couple of the most fundamental textures, which
//  are derived from TRTTexture. These are all procedural and don't require
//  much extra data or code beyond the parent.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 05/08/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Local macros
// -----------------------------------------------------------------------------
#define f8CalcFloor(x) ((x) >= 0.0 ? TMathLib::f8Floor(x) : (0.0-TMathLib::f8Floor(0.0-(x))-1.0))


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI stuff for our classes
// -----------------------------------------------------------------------------
RTTIData(TTxtrChecker,TRTTexture)
RTTIData(TTxtrStripe,TRTTexture)




// -----------------------------------------------------------------------------
//  CLASS: TTxtrChecker
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrChecker: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrChecker::TTxtrChecker(const TFRGBClr& frgbClr1, const TFRGBClr& frgbClr2) :

    TRTTexture()
    , __frgbClr2(frgbClr2)
{
    frgbClr(frgbClr1);
}

TTxtrChecker::TTxtrChecker() :

    TRTTexture()
    , __frgbClr2(TFRGBClr(0,0,0))
{
    // Set the primary color
    frgbClr(TFRGBClr(1,1,1));
}

TTxtrChecker::TTxtrChecker(const TTxtrChecker& txtrCopy) :

    TRTTexture(txtrCopy)
{
    // Copy our stuff
    __frgbClr2 = txtrCopy.__frgbClr2;
}


// -----------------------------------------------------------------------------
//  TTxtrChecker: Public, virtual methods
// -----------------------------------------------------------------------------

TTxtrChecker& TTxtrChecker::operator=(const TTxtrChecker& txtrToAssign)
{
    if (this == &txtrToAssign)
        return *this;

    // Copy our parents stuff
    TParent::operator=(txtrToAssign);

    // Copy our stuff
    __frgbClr2 = txtrToAssign.__frgbClr2;

    return *this;
}


// -----------------------------------------------------------------------------
//  TTxtrChecker: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TTxtrChecker::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens     eToken;
    TString                 strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    // Must be an open parens
    prsiSrc.CheckOpenParen();

    // Deal with the valid methods at this level
    if (eToken == TParseInfo::eToken_AltColor)
    {
        prsiSrc.ParseColor(__frgbClr2);
    }
     else
    {
        // Unget the token and pass it to parent
        prsiSrc.UnGet("(");
        prsiSrc.UnGet(strToken);
        TParent::ParseMethod(prsiSrc);
        return;
    }

    // Has to end with a close paren and semicolon
    prsiSrc.CheckCloseParen();
}


tCIDLib::TVoid 
TTxtrChecker::QueryColorAt( const   T3DVector&   vecPoint
                            ,       TFRGBClr&   frgbColor) const
{
    T3DVector vecTmp(vecPoint);

    // Translate the point via the inverse transform
    trnsTexture().mtrxInvTrans.Transform(vecTmp);

    tCIDLib::TInt4 i4Pat =  tCIDLib::TInt4(f8CalcFloor(vecTmp.f8X()))
                            + tCIDLib::TInt4(f8CalcFloor(vecTmp.f8Z()));
    if (i4Pat & 1)
        frgbColor = frgbClr();
    else
        frgbColor = __frgbClr2;
}


// -----------------------------------------------------------------------------
//  TTxtrChecker: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TTxtrChecker::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the attribute info into the string
    TParent::_FormatTo(strmToWriteTo);
    strmToWriteTo << L", Clr2=" << __frgbClr2;
}

tCIDLib::TVoid TTxtrChecker::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Call our parent's version
    TParent::_StreamFrom(strmToReadFrom);

    // Get the object data members out
    strmToReadFrom >> __frgbClr2;
}

tCIDLib::TVoid TTxtrChecker::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent's version
    TParent::_StreamTo(strmToWriteTo);

    // Write out the object data members
    strmToWriteTo << __frgbClr2;
}




// -----------------------------------------------------------------------------
//  CLASS: TTxtrStripe
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrStripe: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrStripe::TTxtrStripe(const TFRGBClr& frgbClr1, const TFRGBClr& frgbClr2) :

    __frgbClr2(frgbClr2)
{
    // Set the primary color
    frgbClr(frgbClr1);
}

TTxtrStripe::TTxtrStripe() :

    __frgbClr2(TFRGBClr(1,1,1))
{
    // Set the primary color
    frgbClr(TFRGBClr(0,0,0));
}

TTxtrStripe::TTxtrStripe(const TTxtrStripe& txtrToCopy) :

    TRTTexture(txtrToCopy)
    , __frgbClr2(txtrToCopy.__frgbClr2)
{
}


// -----------------------------------------------------------------------------
//  TTxtrStripe: Public operators
// -----------------------------------------------------------------------------

TTxtrStripe& TTxtrStripe::operator=(const TTxtrStripe& txtrToAssign)
{
    if (this == &txtrToAssign)
        return *this;

    // Copy our parents stuff
    TParent::operator=(txtrToAssign);

    // Copy our stuff
    __frgbClr2 = txtrToAssign.__frgbClr2;

    return *this;
}


// -----------------------------------------------------------------------------
//  TTxtrStripe: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TTxtrStripe::QueryColorAt(  const   T3DVector&   vecPoint
                            ,       TFRGBClr&   frgbColor) const
{
    T3DVector vecTmp(vecPoint);

    // Translate the point via the inverse transform
    trnsTexture().mtrxInvTrans.Transform(vecTmp);

    if (tCIDLib::TInt4(f8CalcFloor(vecTmp.f8X())) & 1)
        frgbColor = frgbClr();
    else
        frgbColor = __frgbClr2;
}


tCIDLib::TVoid TTxtrStripe::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens     eToken;
    TString                 strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    // Must be an open parens
    prsiSrc.CheckOpenParen();

    // Deal with the valid methods at this level
    if (eToken == TParseInfo::eToken_AltColor)
    {
        prsiSrc.ParseColor(__frgbClr2);
    }
     else
    {
        // Unget the token and pass it to parent
        prsiSrc.UnGet(L"(");
        prsiSrc.UnGet(strToken);
        TParent::ParseMethod(prsiSrc);
        return;
    }

    // Has to end with a close paren 
    prsiSrc.CheckCloseParen();
}


// -----------------------------------------------------------------------------
//  TTxtrStripe: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TTxtrStripe::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the attribute info into the string
    strmToWriteTo << L"Clr1:" << frgbClr() << L", Clr2:" << __frgbClr2;
}


tCIDLib::TVoid TTxtrStripe::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Call our parent's version
    TParent::_StreamFrom(strmToReadFrom);

    // Get the object data members out
    strmToReadFrom >> __frgbClr2;
}

tCIDLib::TVoid TTxtrStripe::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent's version
    TParent::_StreamTo(strmToWriteTo);

    // Write out the object data members
    strmToWriteTo << __frgbClr2;
}
