//
// NAME: CIDTracer_BandedTextures.Cpp
//
// DESCRIPTION:
//
//  This file implements a set of base texture classes that work on the
//  principle of a set of bands of color that are interpolated in various
//  ways.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 06/30/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData2(TTxtrAgate,TTxtrBanded)
RTTIData2(TTxtrBanded,TTxtrNoise)
RTTIData2(TTxtrGranite,TTxtrNoise)
RTTIData2(TTxtrMarble,TTxtrBanded)
RTTIData2(TTxtrSpots,TTxtrBanded)
RTTIData2(TTxtrWood,TTxtrBanded)



// -----------------------------------------------------------------------------
//  CLASS: TTxtrBanded
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrBanded: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrBanded::TTxtrBanded() :

    TTxtrNoise()
    , __prtlstBands(0)
{
    __prtlstBands = new TRTBandList;
}

 TTxtrBanded::TTxtrBanded(const TTxtrBanded& txtrToCopy) :

    TTxtrNoise(txtrToCopy)
    , __prtlstBands(0)
{
    // Duplicate the band list
    __prtlstBands = new TRTBandList(*txtrToCopy.__prtlstBands);
}


 TTxtrBanded::~TTxtrBanded()
{
    __prtlstBands->Flush();
    delete __prtlstBands;
}


// -----------------------------------------------------------------------------
//  TTxtrBanded: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TTxtrBanded::QueryColorAt(const T3DVector& vecPoint, TFRGBClr& frgbClr) const
{
    T3DVector vecTmp(vecPoint);

    // Translate the point via the inverse transform
    trnsTexture().mtrxInvTrans.Transform(vecTmp);

    //
    //  Get the noise value. We call a virtual protected method here to
    //  let clients adjust the noise as needed.
    //
    tCIDLib::TFloat8 f8NoiseVal = _f8CalcNoise(vecTmp);

    // Just in case, clip the value to the valid range
    if (f8NoiseVal < 0.0)
        f8NoiseVal = 0.0;
    else if (f8NoiseVal > 1.0)
        f8NoiseVal = 1.0;

    // If we did not find one, just call the default handler
    TRTBandNode* pnodeCur = __prtlstBands->pnodeHead();

    if (!pnodeCur)
    {
        frgbClr = this->frgbClr();
        _CalcDefColor(f8NoiseVal, frgbClr);
        return;
    }

    //
    //  Enumerate the bands and look for the first one that encompases
    //  the noise value.
    //
    while (pnodeCur)
    {
        // Get a pointer to the band item
        TBandItem* const pbandCur = pnodeCur->pobjData();

        if ((f8NoiseVal >= pbandCur->f8StartVal)
        &&  (f8NoiseVal <= pbandCur->f8EndVal))
        {
            //
            //  Call the protected virtual method that lets the derivative
            //  look at the colors and decide what to do. The default
            //  provided by this class interpolates the color.
            //
            TFRGBClr frgbNewClr;
            _CalcBandColor
            (
                f8NoiseVal
                , pbandCur->f8StartVal
                , pbandCur->f8EndVal
                , pbandCur->frgbStartClr
                , pbandCur->frgbEndClr
                , frgbNewClr
            );
            frgbClr += frgbNewClr;
            return;
        }

        // Move to the next one
        pnodeCur = pnodeCur->pnodeNext();
    }

    // Wwe did not find a band, so set it to black
    frgbClr.FadeToBlack();
}

tCIDLib::TVoid  TTxtrBanded::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens     eToken;
    TString                 strName(kCIDLib::pszEmptyZStr, 64);
    TString                 strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    // Must be followed by an open parens
    prsiSrc.CheckOpenParen();

    // Deal with the valid methods at this level
    if ((eToken == TParseInfo::eToken_AppendBand)
    ||  (eToken == TParseInfo::eToken_PrependBand))
    {
        tCIDLib::TFloat8    f8Start, f8End;
        TFRGBClr            frgbStart, frgbEnd;

        prsiSrc.ParseFloat(f8Start);
        prsiSrc.CheckComma();

        prsiSrc.ParseFloat(f8End);
        prsiSrc.CheckComma();

        prsiSrc.ParseColor(frgbStart);
        prsiSrc.CheckComma();

        prsiSrc.ParseColor(frgbEnd);

        // Add the new band to the list
        if (eToken == TParseInfo::eToken_AppendBand)
            AppendBand(f8Start, f8End, frgbStart, frgbEnd);
         else
            PrependBand(f8Start, f8End, frgbStart, frgbEnd);
    }
     else
    {
        // Unget the token and pass it to parent
        prsiSrc.UnGet(L"(");
        prsiSrc.UnGet(strToken);
        TParent::ParseMethod(prsiSrc);
        return;
    }

    // Gotta be a close paren
    prsiSrc.CheckCloseParen();
}


// -----------------------------------------------------------------------------
//  TTxtrBanded: Public operators
// -----------------------------------------------------------------------------

TTxtrBanded&  TTxtrBanded::operator=(const TTxtrBanded& txtrSrc)
{
    // Check for assignment to self
    if (this == &txtrSrc)
        return *this;

    // Handle our parent's fields
    TParent::operator=(txtrSrc);

    *__prtlstBands = *txtrSrc.__prtlstBands;
    return *this;
}


// -----------------------------------------------------------------------------
//  TTxtrBanded: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TTxtrBanded::AppendBand(const  tCIDLib::TFloat8    f8StartVal
                        , const tCIDLib::TFloat8    f8EndVal
                        , const TFRGBClr&           frgbStartClr
                        , const TFRGBClr&           frgbEndClr)
{
    __prtlstBands->Append
    (
        new TBandItem
        (
            f8EndVal
            , f8StartVal
            , frgbEndClr
            , frgbStartClr
        )
    );
}

tCIDLib::TVoid 
TTxtrBanded::PrependBand(   const   tCIDLib::TFloat8    f8StartVal
                            , const tCIDLib::TFloat8    f8EndVal
                            , const TFRGBClr&           frgbStartClr
                            , const TFRGBClr&           frgbEndClr)
{
    __prtlstBands->Add
    (
        new TBandItem
        (
            f8EndVal
            , f8StartVal
            , frgbEndClr
            , frgbStartClr
        )
    );
}


// -----------------------------------------------------------------------------
//  TTxtrBanded: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid  TTxtrBanded::_FormatTo(TTextStream& strmToWriteTo) const
{
    TParent::_FormatTo(strmToWriteTo);
}


tCIDLib::TVoid  TTxtrBanded::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Flush any existing bands
    __prtlstBands->Flush();

    // Call our parent first
    TParent::_StreamFrom(strmToReadFrom);

    // Get out the count of bands
    tCIDLib::TCard4 c4Count;
    strmToReadFrom >> c4Count;

    // Now stream back in our band items
    TBandItem* pbandNew;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
    {
        ::PolymorphicRead(pbandNew, strmToReadFrom);
        __prtlstBands->Append(pbandNew);
    }
}

tCIDLib::TVoid 
TTxtrBanded::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent first
    TParent::_StreamTo(strmToWriteTo);

    // Add the number of bands we have
    strmToWriteTo << __prtlstBands->c4NodeCount();

    // Now stream out our band items
    TRTBandNode* pnodeCur = __prtlstBands->pnodeHead();
    while (pnodeCur)
    {
        ::PolymorphicWrite(pnodeCur->pobjData(), strmToWriteTo);
        pnodeCur = pnodeCur->pnodeNext();
    }
}


// -----------------------------------------------------------------------------
//  TTxtrBanded: Protected, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TTxtrBanded::_CalcBandColor(const   tCIDLib::TFloat8    f8NoiseVal
                            , const tCIDLib::TFloat8    f8StartVal
                            , const tCIDLib::TFloat8    f8EndVal
                            , const TFRGBClr&           frgbStartClr
                            , const TFRGBClr&           frgbEndClr
                            ,       TFRGBClr&           frgbResult) const
{
    tCIDLib::TFloat8 f8Frac = (f8NoiseVal - f8StartVal)
                                            / (f8EndVal - f8StartVal);

    // Tell the color to interpolate itself
    frgbResult.Interpolate(f8Frac, frgbStartClr, frgbEndClr);
}



// -----------------------------------------------------------------------------
//  CLASS: TTxtrAgate
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrAgate: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrAgate::TTxtrAgate() :

    TTxtrBanded()
{
}

 TTxtrAgate::TTxtrAgate(const TTxtrAgate& txtrToCopy) :

    TTxtrBanded(txtrToCopy)
{
}


// -----------------------------------------------------------------------------
//  TTxtrAgate: Public operators
// -----------------------------------------------------------------------------

TTxtrAgate&  TTxtrAgate::operator=(const TTxtrAgate& txtrSrc)
{
    // Check for assignment to self
    if (this == &txtrSrc)
        return *this;

    // Handle our parent's fields
    TParent::operator=(txtrSrc);

    return *this;
}


// -----------------------------------------------------------------------------
//  TTxtrAgate: Protected, virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TFloat8 TTxtrAgate::_f8CalcNoise(const T3DVector& vecPoint) const
{
    tCIDLib::TFloat8    f8Noise;
   
    f8Noise = _f8Cycloidal(1.3 * _f8Turbulence(vecPoint) + 1.1 * vecPoint.f8Z())
                + 1;
    f8Noise *= 0.5;
    return TMathLib::f8Power(f8Noise, 0.77);
}


tCIDLib::TVoid 
TTxtrAgate::_CalcDefColor(const tCIDLib::TFloat8 f8Noise, TFRGBClr& frgbClr) const
{
    tCIDLib::TFloat8  f8Hue = 1.0 - f8Noise;

    if (f8Noise < 0.5)
        frgbClr.Adjust(1.0 - (f8Noise / 10), 1.0 - (f8Noise / 5), f8Hue, 0.0);
     else if (f8Noise < 0.6)
        frgbClr.Adjust(0.9, 0.7, f8Hue, 0.0);
     else
        frgbClr.Adjust(0.6 + f8Hue, 0.3 + f8Hue, f8Hue, 0.0);
}




// -----------------------------------------------------------------------------
//  CLASS: TTxtrGranite
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrGranite: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrGranite::TTxtrGranite() :

    TTxtrBanded()
{
}

TTxtrGranite::TTxtrGranite(const TTxtrGranite& txtrToCopy) :

    TTxtrBanded(txtrToCopy)
{
}

// -----------------------------------------------------------------------------
//  TTxtrGranite: Public operators
// -----------------------------------------------------------------------------

TTxtrGranite& TTxtrGranite::operator=(const TTxtrGranite& txtrToAssign)
{
    if (this == &txtrToAssign)
        return *this;

    // Copy over our parent's stuff
    TParent::operator=(txtrToAssign);

    return *this;
}


// -----------------------------------------------------------------------------
//  TTxtrAgate: Protected, virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TFloat8  TTxtrGranite::_f8CalcNoise(const T3DVector& vecPoint) const
{
    tCIDLib::TFloat8 f8Freq  = 1.0;
    tCIDLib::TFloat8 f8Noise = 0.0;
    tCIDLib::TFloat8 f8Tmp;
    tCIDLib::TFloat8 f8X = vecPoint.f8X();
    tCIDLib::TFloat8 f8Y = vecPoint.f8Y();
    tCIDLib::TFloat8 f8Z = vecPoint.f8Z();

    for (tCIDLib::TInt4 i4Cnt = 0; i4Cnt < 6; f8Freq *= 2.0, i4Cnt++)
    {
        f8Tmp = 0.5 - _f8Noise
        (
            f8X * 4 * f8Freq
            , f8Y * 4 * f8Freq
            , f8Z * 4 * f8Freq
        );

        f8Tmp = TMathLib::f8Abs(f8Tmp);
        f8Noise += (f8Tmp / f8Freq);
    }

    return f8Noise;
}


//
// FUNCTION/METHOD NAME: _CalcDefColor
//
// DESCRIPTION:
//
//  We override this method to handle the situation where the noise value
//  does not fall into a band.
// ---------------------------------------
//   INPUT: f8Noise is the noise level
//
//  OUTPUT: frgbClr is the color to adjust with the noise function.
//
//  RETURN: None
//
tCIDLib::TVoid 
TTxtrGranite::_CalcDefColor(const   tCIDLib::TFloat8    f8Noise
                            ,       TFRGBClr&           frgbClr) const
{
    frgbClr += f8Noise;
}




// -----------------------------------------------------------------------------
//  CLASS: TTxtrMarble
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrMarble: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrMarble::TTxtrMarble()
{
}

 TTxtrMarble::TTxtrMarble(const TTxtrMarble& txtrToCopy) :

    TTxtrBanded(txtrToCopy)
{
}


// -----------------------------------------------------------------------------
//  TTxtrMarble: Public operators
// -----------------------------------------------------------------------------

TTxtrMarble&  TTxtrMarble::operator=(const TTxtrMarble& txtrToAssign)
{
    // Check for assignment to self
    if (this == &txtrToAssign)
        return *this;

    // Handle our parent's fields
    TParent::operator=(txtrToAssign);

    return *this;
}


// -----------------------------------------------------------------------------
//  TTxtrMarble: Protected, virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TFloat8 
TTxtrMarble::_f8CalcNoise(const T3DVector& vecPoint) const
{
    return _f8TriangleWave
    (
        vecPoint.f8X()
        + _f8Turbulence(vecPoint)
        * f8Turbulence()
    );
}


tCIDLib::TVoid 
TTxtrMarble::_CalcDefColor( const   tCIDLib::TFloat8    f8Noise
                            ,       TFRGBClr&           frgbClr) const
{
    if (f8Noise < 0.0)
    {
        frgbClr.Adjust(0.9, 0.8, 0.8, 0.0);
    }
     else if (f8Noise < 0.9)
    {
        tCIDLib::TFloat8 f8Hue = 0.8 - f8Noise * 0.8;
        frgbClr.Adjust(0.9, f8Hue, f8Hue, 0.0);
    }
}



// -----------------------------------------------------------------------------
//  CLASS: TTxtrSpots
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrSpots: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrSpots::TTxtrSpots() :

    TTxtrBanded()
{
}

 TTxtrSpots::TTxtrSpots(const TTxtrSpots& txtrSpots) :

    TTxtrBanded(txtrSpots)
{
}

// -----------------------------------------------------------------------------
//  TTxtrSpots: Public operators
// -----------------------------------------------------------------------------

TTxtrSpots&  TTxtrSpots::operator=(const TTxtrSpots& txtrSrc)
{
    // Check for assignment to self
    if (this == &txtrSrc)
        return *this;

    // Handle our parent's fields
    TParent::operator=(txtrSrc);

    return *this;
}




// -----------------------------------------------------------------------------
//  CLASS: TTxtrWood
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTxtrWood: Constructors and Destructors
// -----------------------------------------------------------------------------

TTxtrWood::TTxtrWood() :

    TTxtrBanded()
{
}

 TTxtrWood::TTxtrWood(const TTxtrWood& txtrToCopy) :

    TTxtrBanded(txtrToCopy)
{
}


// -----------------------------------------------------------------------------
//  TTxtrWood: Public operators
// -----------------------------------------------------------------------------

TTxtrWood&  TTxtrWood::operator=(const TTxtrWood& txtrToAssign)
{
    // Check for assignment to self
    if (this == &txtrToAssign)
        return *this;

    // Handle our parent's fields
    TParent::operator=(txtrToAssign);

    return *this;
}


// -----------------------------------------------------------------------------
//  TTxtrWood: Protected, virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TFloat8 
TTxtrWood::_f8CalcNoise(const T3DVector& vecPoint) const
{
    tCIDLib::TFloat8 f8Length;
    tCIDLib::TFloat8 f8X     = vecPoint.f8X();
    tCIDLib::TFloat8 f8Y     = vecPoint.f8Y();
    tCIDLib::TFloat8 f8Z     = vecPoint.f8Z();
    T3DVector         vecWood;

    _VecTurbulence(vecWood, f8X, f8Y, f8Z);

    T3DVector vecTmp
    (
        _f8Cycloidal((f8X + vecWood.f8X()) * f8Turbulence())
        , _f8Cycloidal((f8Y + vecWood.f8Y()) * f8Turbulence())
        , 0.0
    );

    vecTmp.Adjust(f8X, f8Y, f8Z);
    f8Length = vecTmp.f8Magnitude();

    return _f8TriangleWave(f8Length);
}


tCIDLib::TVoid 
TTxtrWood::_CalcDefColor(const tCIDLib::TFloat8 f8Noise, TFRGBClr& frgbClr) const
{
    if (f8Noise > 0.6)
        frgbClr.Adjust(0.4, 0.133, 0.066, 0.0);
     else
        frgbClr.Adjust(0.666, 0.312, 0.2, 0.0);
}
