//
// NAME: CIDTracer_BandedItem.Cpp
//
// DESCRIPTION:
//
//  This file implements a helper class used by the banded textures.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 06/30/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData2(TBandItem,TObject)



// -----------------------------------------------------------------------------
//   CLASS: TBandItem
//  PREFIX: band
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TBandItem: Constructors and Destructors
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TBandItem
//
// DESCRIPTION:
//
//  There is a copy constructor and one to pass in the parameters.
// ---------------------------------------
//   INPUT: f8End, f8Start are the limit values for the band, they must
//              be from 0.0 to 1.0
//          frgbEnd, frgbStart are the ending and starting colors for the
//              band. They are interpolated between according to the actual
//              value's relation to the end/start values.
//
//  OUTPUT: None
//
//  RETURN: None
//
TBandItem::TBandItem(   const   tCIDLib::TFloat8    f8End
                        , const tCIDLib::TFloat8    f8Start
                        , const TFRGBClr&           frgbEnd
                        , const TFRGBClr&           frgbStart) :
    f8EndVal(f8End)
    , f8StartVal(f8Start)
    , frgbEndClr(frgbEnd)
    , frgbStartClr(frgbStart)
{
    tCIDLib::TFloat8 f8Tmp;
    if (f8EndVal < f8StartVal)
    {
        f8Tmp = f8EndVal;
        f8EndVal = f8StartVal;
        f8StartVal = f8Tmp;
    }
}

TBandItem::TBandItem() :

    f8EndVal(1.0)
    , f8StartVal(0.0)
    , frgbEndClr(TFacCIDTracer::frgbBlack)
    , frgbStartClr(TFacCIDTracer::frgbWhite)
{
}

TBandItem::TBandItem(const TBandItem& bandSrc) :

    f8EndVal(bandSrc.f8EndVal)
    , f8StartVal(bandSrc.f8StartVal)
    , frgbEndClr(bandSrc.frgbEndClr)
    , frgbStartClr(bandSrc.frgbStartClr)
{
}

// -----------------------------------------------------------------------------
//  TBandItem: Public operators
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: operator==
//
// DESCRIPTION:
//
//  Compares this band item to the passed one
// ---------------------------------------
//   INPUT: bandToTest is the band item to test
//
//  OUTPUT: None
//
//  RETURN: eTrue if equal, else eFalse.
//
tCIDLib::TBoolean TBandItem::operator==(const TBandItem& bandToTest) const
{
    if (f8EndVal != bandToTest.f8EndVal)
        return kCIDLib::False;

    if (f8StartVal != bandToTest.f8StartVal)
        return kCIDLib::False;

    if (frgbEndClr != bandToTest.frgbEndClr)
        return kCIDLib::False;

    if (frgbStartClr != bandToTest.frgbStartClr)
        return kCIDLib::False;

    return kCIDLib::True;
}


// -----------------------------------------------------------------------------
//  TBandItem: Protected, inherited methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _FormatTo
//
// DESCRIPTION:
//
//  This method will format the object's attributes to the text stream.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: strmToWriteTo is the stream to format ourself to.
//
//  RETURN: None
//
tCIDLib::TVoid  TBandItem::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo   << L"Start: " << f8StartVal
                    << L", End: " << f8EndVal
                    << L", Start Clr: " << frgbEndClr
                    << L", End Clr: " << frgbStartClr;
}


//
// FUNCTION/METHOD NAME: _StreamFrom
//                       _StreamTo
//
// DESCRIPTION:
//
//  These methods handle streaming in and out band item objects. We just
//  stream out the two floating point start/end members then ask the two
//  related color objects to stream themselves.
// ---------------------------------------
//   INPUT: strmToReadFrom is the stream to parse from.
//          strmToWriteTo is the stream to write to.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid  TBandItem::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Stream in the two floats first
    strmToReadFrom >> f8EndVal;
    strmToReadFrom << f8StartVal;

    // Now ask our two object members to stream themselves in
    strmToReadFrom >> frgbEndClr;
    strmToReadFrom >> frgbStartClr;
}

tCIDLib::TVoid  TBandItem::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Stream out the two floats first
    strmToWriteTo << f8EndVal;
    strmToWriteTo << f8StartVal;

    // Now ask our two object members to stream themselves
    strmToWriteTo << frgbEndClr;
    strmToWriteTo << frgbStartClr;
}
