//
//  FILE NAME: CIDTracer.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/23/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module of the facility. This guy provides the DLL
//  init function in order to handle initialization of any modules that
//  require it.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"
#include    "CIDLib_ModuleInit_.Hpp"



// ----------------------------------------------------------------------------
//  Global data
//
//  facCIDTracer
//      This is the facility object for CIDTracer.Dll. Its exported via the
//      main header.
// ----------------------------------------------------------------------------
TFacCIDTracer   facCIDTracer;


// ----------------------------------------------------------------------------
//  Intrafacility data
//
//  _pmtrgTracerCore
//      The core metrics for the ray tracer. Its exported for intrafacility
//      use via the intrafacility header.
// ----------------------------------------------------------------------------
TMetricGroup*   _pmtrgTracerCore;


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
static tCIDLib::TVoid __InitCIDTracer
(
    const   tCIDLib::EInitTerm      eInitTerm
    , const tCIDLib::EGlobalStates  eGlobals
    , const TModule&                modInit
    , const tCIDLib::TCard4         c4MaxChars
    ,       tCIDLib::Tch* const     pszFailReason
);


// ----------------------------------------------------------------------------
//  Data for local use
//
//  __afnInitTerm
//      This is the list of init/term functions for this facility.
// ----------------------------------------------------------------------------
static tCIDLib::TModuleInitTermRec  __afnInitTerm[] =
{
      { __InitCIDTracer         , L"Base Init" }
    , { _InitTermNoiseTextures  , L"Noise Textures" }
    , { _InitTermParser         , L"Scene Parser" }
};
static tCIDLib::TCard4      __c4InitCount = c4ArrayElems(__afnInitTerm);



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

static tCIDLib::TVoid
__InitCIDTracer(const   tCIDLib::EInitTerm      eInitTerm
                , const tCIDLib::EGlobalStates  eGlobals
                , const TModule&                modInit
                , const tCIDLib::TCard4         c4MaxChars
                ,       tCIDLib::Tch* const     pszFailReason)
{
    tCIDLib::Tch* pszReason = L"Uknown";
    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            //
            //  Load up a couple of primal strings that are used in emergency
            //  popups. These have default english values which will be there
            //  if these fail, but this will set up the translated versions.
            //
            //  We have to cast off the const'ness of the pointers in order
            //  to update them.
            //
            const tCIDLib::Tch* pszTmp;
            const tCIDLib::Tch* pszNC;

            pszReason = L"Loading title messages";
            pszTmp = modInit.pszLoadCIDMsg(kTracerMsgs::midFac_Title1);
            if (pszTmp)
            {
                pszNC = const_cast<tCIDLib::Tch*>(kCIDTracer_::pszTitle1);
                pszNC = pszTmp;
            }

            pszTmp = modInit.pszLoadCIDMsg(kTracerMsgs::midFac_Title2);
            if (pszTmp)
            {
                pszNC = const_cast<tCIDLib::Tch*>(kCIDTracer_::pszTitle2);
                pszNC = pszTmp;
            }

            //
            //  Set up the core ray tracer metrics
            //
            pszReason = L"Setting up fractal metrics";
            TMetricSystem::TMetricInfo amtriList[tCIDTracer_::eCorEMetric_Count];
            TMetricSystem::TMetricInfo* pmtriTmp;

            pmtriTmp = &amtriList[tCIDTracer_::eCorEMetric_ActiveScenes];
            pmtriTmp->eType          = tCIDLib::EMetric_Cardinal;
            pmtriTmp->strMetricName  = L"Active Scenes";
            pmtriTmp->strMetricHelp  = L"The count of scenes currently being traced";

            pmtriTmp++;
            pmtriTmp->eType          = tCIDLib::EMetric_Cardinal;
            pmtriTmp->strMetricName  = L"Intersect Array Cache Pages";
            pmtriTmp->strMetricHelp  = L"The # of cache pages in use by TIntersectArray";

            //
            //  Create a metric system object, which we need to register a
            //  metrics group. Then register our group.
            //
            _pmtrgTracerCore = TMetricSystem::pmtrgRegisterGroup
            (
                L"CIDRay Core Metrics"
                , L"Fundamental CIDLib Ray Tracing Metrics"
                , amtriList
                , tCIDTracer_::eCorEMetric_Count
            );
        }
    }

    catch(...)
    {
        TRawStr::CopyCatStr
        (
            pszFailReason
            , c4MaxChars
            , pszReason
            , L" failed"
        );
        throw;
    }
}

// ----------------------------------------------------------------------------
//  DLL Initialization and program entry point
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: CIDTracerInit
//
// DESCRIPTION:
//
//  This is the entry point for the facility. It is called from the host OS
//  DLL init. It just passes on the call to the standard module init/term
//  function which does all the work.
// -------------------------------------
//   INPUT: hmodThis is the module handle of this DLL.
//          eInitTerm is the reason we are getting called.
//          pReserved is not used of course.
//
//  OUTPUT: None
//
//  RETURN: 1 if successful, else 0. If a 0 is returned, then the load of
//              the client app will fail.
//
tCIDLib::TCard4 DLLINITAPI
CIDTracerInit(  tCIDLib::TModHandle     hmodThis
                , tCIDLib::EInitTerm    eInitTerm
                , tCIDLib::TVoid*       pDummy)
{
    // Now do the module inits
    if (!bCIDLib_DoModuleInitTerm
    (
        eInitTerm
        , hmodThis
        , tCIDLib::EModType_Dll
        , __c4InitCount
        , __afnInitTerm
        , kCIDTracer_::pszTitle1
        , kCIDTracer_::pszTitle2))
    {
        return 0;
    }
    return 1;
}
