//
// NAME: CIDLib_ValueMap2.Cpp
//
// DESCRIPTION:
//
//  This module implements a couple of simple derivatives of the abstract
//  TValueMap class. TLogMap does a logarithmic mapping. TModuloMap does a
//  modulo division mapping.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 09/08/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData(TLogMap,TValueMap)
RTTIData(TModMap,TValueMap)


// -----------------------------------------------------------------------------
//  TLogMap: Constructors and Destructors
// -----------------------------------------------------------------------------

TLogMap::TLogMap() :

    TValueMap()
    , __ac1LogTable(0)
    , __c4ElemSize(1)
    , __eMode(tCIDLib::ELogMapMode_Spread)
{
    __AllocTable();
    __InitTable();
}

TLogMap::TLogMap(   const   tCIDLib::TCard4         c4MaxIndex1
                    , const tCIDLib::TCard4         c4MaxIndex2
                    , const tCIDLib::ELogMapModes   eMode) :

    TValueMap(c4MaxIndex1, c4MaxIndex2)
    , __ac1LogTable(0)
    , __c4ElemSize(0)
    , __eMode(eMode)
{
    if (c4MaxIndex1 <= 0xFF)
        __c4ElemSize = 1;
    else if (c4MaxIndex1 <= 0xFFFF)
        __c4ElemSize = 2;
    else
        __c4ElemSize = 4;

    __AllocTable();
    __InitTable();
}

TLogMap::TLogMap(const TLogMap& vmapToCopy) :

    TValueMap(vmapToCopy)
    , __ac1LogTable(0)
    , __c4ElemSize(vmapToCopy.__c4ElemSize)
    , __eMode(vmapToCopy.__eMode)
{
    __AllocTable();
    __InitTable();
}

TLogMap::~TLogMap()
{
    delete [] __ac1LogTable;
}


// -----------------------------------------------------------------------------
//  TLogMap: Public operators
// -----------------------------------------------------------------------------

TLogMap& TLogMap::operator=(const TLogMap& vmapToAssign)
{
    // Remember the current max indexes
    tCIDLib::TCard4 c4Max1 = c4MaxIndex1();
    tCIDLib::TCard4 c4Max2 = c4MaxIndex2();

    // Call our parent, which will update the new maximums
    TParent::operator=(vmapToAssign);

    // Store our map mode and element size
    __c4ElemSize    = vmapToAssign.__c4ElemSize;
    __eMode         = vmapToAssign.__eMode;

    // If the indexes changed, then redo the look up table
    if ((c4Max1 != c4MaxIndex1()) || (c4Max2 != c4MaxIndex2()))
    {
        delete [] __ac1LogTable;
        __ac1LogTable = 0;
        __AllocTable();
        __InitTable();
    }
    return *this;
}


tCIDLib::TBoolean TLogMap::operator==(const TLogMap& vmapToCompare) const
{
    if (this == &vmapToCompare)
        return kCIDLib::True;

    // Compare our parent's stuff first
    if (!TParent::operator==(vmapToCompare))
        return kCIDLib::False;

    // Compare our members
    if (__c4ElemSize != vmapToCompare.__c4ElemSize)
        return kCIDLib::False;

    if (__eMode != vmapToCompare.__eMode)
        return kCIDLib::False;

    return kCIDLib::True;
}


// -----------------------------------------------------------------------------
//  TLogMap: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TCard4
TLogMap::c4MapIndex(const tCIDLib::TCard4 c4Index1) const
{
    // Sanity check the index
    if (c4Index1 > c4MaxIndex1())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcVMap_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TCardinal(c4Index1)
            , TString(L"first")
            , TCardinal(c4MaxIndex1())
        );
    }

    //
    //  Return the element at that index. This method handles the different
    //  element sizes we can have.
    //
    return __c4ElemAt(c4Index1);
}


tCIDLib::TVoid
TLogMap::SetNewRanges(  const   tCIDLib::TCard4 c4NewMax1
                        , const tCIDLib::TCard4 c4NewMax2)
{
    // If they changed, then we need to reallocate
    tCIDLib::TCard4 c4Max1 = c4MaxIndex1();
    tCIDLib::TCard4 c4Max2 = c4MaxIndex2();

    // Call our parent
    TParent::SetNewRanges(c4NewMax1, c4NewMax2);

    // Redo the table if we need to
    if ((c4NewMax1 != c4Max1) || (c4NewMax2 != c4Max2))
    {
        delete [] __ac1LogTable;
        __ac1LogTable = 0;
        __AllocTable();
        __InitTable();
    }
}

// -----------------------------------------------------------------------------
//  TLogMap: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TLogMap::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Save the old versions
    tCIDLib::TCard4 c4Max1 = c4MaxIndex1();
    tCIDLib::TCard4 c4Max2 = c4MaxIndex2();

    // Call our parent
    TParent::_StreamFrom(strmToReadFrom);

    // Pull in the map mode
    strmToReadFrom >> __eMode;

    // If the indices changed, then reinitialize the table
    if ((c4MaxIndex1() != c4Max1) || (c4MaxIndex2() != c4Max2))
    {
        if (c4MaxIndex1() < 0xFF)
            __c4ElemSize = 1;
        else if (c4MaxIndex1() < 0xFFFF)
            __c4ElemSize = 2;
        else
            __c4ElemSize = 4;

        delete [] __ac1LogTable;
        __ac1LogTable = 0;
        __AllocTable();
    }
    __InitTable();
}

tCIDLib::TVoid TLogMap::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent first
    TParent::_StreamTo(strmToWriteTo);

    // And write out our map mode
    strmToWriteTo << __eMode;
}


// -----------------------------------------------------------------------------
//  TLogMap: Private, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __AllocTable
//
// DESCRIPTION:
//
//  This method allocates the table the correct size according to the first
//  max index and the element size.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TLogMap::__AllocTable()
{
    __ac1LogTable = new tCIDLib::TCard1[(c4MaxIndex1()+1) * __c4ElemSize];
}

//
// FUNCTION/METHOD NAME: __c4ElemAt
//
// DESCRIPTION:
//
//  This method will return the indicated element, taking into account the
//  element size.
// ---------------------------------------
//   INPUT: c4Index is the index of the element to return. Its assumed that
//              the caller checked it.
//
//  OUTPUT: None
//
//  RETURN: The value at that index, cast to a TCard4.
//
tCIDLib::TCard4 TLogMap::__c4ElemAt(const tCIDLib::TCard4 c4Index) const
{
    #if CID_DEBUG_ON
    if (c4Index > c4MaxIndex1())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcVMap_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TCardinal(c4Index)
            , TString(L"first")
            , TCardinal(c4MaxIndex1())
        );
    }
    #endif

    tCIDLib::TCard4 c4Ret = 0;
    if (__c4ElemSize == 1)
    {
        c4Ret = tCIDLib::TCard4(__ac1LogTable[c4Index]);
    }
     else if (__c4ElemSize == 2)
    {
        c4Ret = *((tCIDLib::TCard2*)(__ac1LogTable + (c4Index*2)));
    }
     else if (__c4ElemSize == 4)
    {
        c4Ret = *((tCIDLib::TCard4*)(__ac1LogTable + (c4Index*4)));
    }
     else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcVMap_BadElemSize
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TCardinal(__c4ElemSize)
        );
    }
    return c4Ret;
}


//
// FUNCTION/METHOD NAME: __InitTable
//
// DESCRIPTION:
//
//  Initializes the table, given the current state. It assumes that the
//  other members are set and the table is allocated.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TLogMap::__InitTable()
{
    tCIDLib::TFloat8    f8Spread;
    tCIDLib::TCard4     c4Index;
    tCIDLib::TCard4     c4Limit;
    tCIDLib::TCard4     c4Prev;

    // For speed and convenience, get the max indexes out
    tCIDLib::TCard4     c4Max1 = c4MaxIndex1();
    tCIDLib::TCard4     c4Max2 = c4MaxIndex2();

    tCIDLib::TCard4     c4Mode = 0;
    if (__eMode)
        c4Mode++;

    if (c4Mode >= c4Max1)
        c4Mode = c4Max1 - 1;

    f8Spread = TMathLib::f8Log(tCIDLib::TFloat8(c4Max1-c4Mode));
    f8Spread /= tCIDLib::TFloat8(c4Max2 - (c4Mode ? 2.0 : 1.0));
    for (c4Prev = 1; c4Prev <= c4Mode; c4Prev++)
        __PutAt(c4Prev, 1);

    for (c4Index = (c4Mode ? 2 : 1); c4Index < c4Max2; c4Index++)
    {
        c4Limit = tCIDLib::TCard4
        (
            TMathLib::f8Exponent(tCIDLib::TFloat8(c4Index) * f8Spread)
        );

        c4Limit += c4Mode;
        if ((c4Limit > c4Max1) || (c4Index == c4Max2-1))
            c4Limit = c4Max1-1;

        while (c4Prev <= c4Limit)
        {
            __PutAt(c4Prev, c4Index);
            c4Prev++;
        }
    }

    __PutAt(0, 0);
    for (c4Index = 1; c4Index < c4Max1; c4Index++)
    {
        if (__c4ElemAt(c4Index) > __c4ElemAt(c4Index-1))
            __PutAt(c4Index, __c4ElemAt(c4Index-1)+1);
    }
}


//
// FUNCTION/METHOD NAME: __PutAt
//
// DESCRIPTION:
//
//  This method will put a new value into the table. It will cut down the
//  value to fit into the element size. It is assumed that our internal
//  callers will not call us with a value too big. If they do, then it will
//  get sliced.
// ---------------------------------------
//   INPUT: c4Index is the index to put the value at.
//          c4Val is the value to put in.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TLogMap::__PutAt(const   tCIDLib::TCard4 c4Index
                                , const tCIDLib::TCard4 c4Val)
{
    #if CID_DEBUG_ON
    if (c4Index > c4MaxIndex1())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcVMap_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TCardinal(c4Index)
            , TString(L"first")
            , TCardinal(c4MaxIndex1())
        );
    }
    #endif

    tCIDLib::TCard4 c4Ret = 0;
    if (__c4ElemSize == 1)
    {
        __ac1LogTable[c4Index] = tCIDLib::TCard1(c4Val);
    }
     else if (__c4ElemSize == 2)
    {
        *((tCIDLib::TCard2*)(__ac1LogTable + (c4Index*2))) = tCIDLib::TCard2(c4Val);
    }
     else if (__c4ElemSize == 4)
    {
        *((tCIDLib::TCard4*)(__ac1LogTable + (c4Index*4))) = c4Val;
    }
     else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcVMap_BadElemSize
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TCardinal(__c4ElemSize)
        );
    }
}
