//
// NAME: CIDLib_UniqueName.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_UniqueName.Cpp module. This module
//  implements the TUniqueName class, which is used to generate
//  sequentially numbered names.
//
//  This class uses the replacement token system. The text of the object
//  must contain a replacement parameter for the character '1' (i.e. %(1))
//  which will be replaced by a sequentially incremented number. The token
//  can be any legal token replacement so the format of the number can be
//  controlled.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/07/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  We don't have a Cpp file, so our RTTI macros are done in CIDLib.Cpp
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TUniqueName
//  PREFIX: unam
// ----------------------------------------------------------------------------
class CIDLIBEXP TUniqueName : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Constuctors and Destructors
        // --------------------------------------------------------------------
        TUniqueName
        (
            const   TString&                strText
            , const tCIDLib::TInt4          i4InitialValue = 0
        );

        TUniqueName
        (
            const   tCIDLib::Tch* const     pszText
            , const tCIDLib::TInt4          i4InitialValue = 0
        );

        ~TUniqueName();


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid QueryNewName
        (
                    TString&                strToFill
        );

        TString strQueryNewName();


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TUniqueName();

        TUniqueName(const TUniqueName&);

        tCIDLib::TVoid operator=(const TUniqueName&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __scntNumber
        //      This is the thread safe counter that is used to generate the
        //      unique numbers.
        //
        //  __strText
        //      The text that serves as the template for each name. It must
        //      have the token %(1) in it somewhere.
        // --------------------------------------------------------------------
        TSafeCounter    __scntNumber;
        TString         __strText;


        // --------------------------------------------------------------------
        //  Do any needed macros
        // --------------------------------------------------------------------
        RTTIMacros(TUniqueName,TObject)
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  Constructors and destructors
// ----------------------------------------------------------------------------
inline TUniqueName::TUniqueName(    const   TString&        strText
                                    , const tCIDLib::TInt4  i4Value) :
    __scntNumber(i4Value)
    , __strText(strText)
{
}

inline TUniqueName::TUniqueName(const   tCIDLib::Tch* const pszText
                                , const tCIDLib::TInt4      i4Value) :
    __scntNumber(i4Value)
    , __strText(pszText)
{
}

inline TUniqueName::~TUniqueName()
{
}

// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TVoid TUniqueName::QueryNewName(TString& strToFill)
{
    // Get the current value out and add 1 for next time
    tCIDLib::TInt4 i4Value = __scntNumber.i4AddTo(1);

    // Get a copy of the text and replace the parameter
    strToFill = __strText;
    strToFill.ReplaceToken(TInteger(i4Value), L'1');
}

inline TString TUniqueName::strQueryNewName()
{
    // Get the current value and add 1 for the next time
    tCIDLib::TInt4 i4Value = __scntNumber.i4AddTo(1);

    // Get a copy of the text and replace the parameter
    TString strRet = __strText;
    strRet.ReplaceToken(TInteger(i4Value), L'1');

    return strRet;
}
