//
// NAME: CIDLib_TypedMBuf.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_TypedMBuf.Cpp module. This module provides
//  the TTypedMBuf class. This class provides a very convenient way to manage
//  a TMemBuf derived class whose memory buffer is a structure. This is often
//  required to interact with third party code or to shared memory between
//  processes.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/01/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  The default constructor creates a THeapBuf object. The other version
//      will create a system memory buffer, and the third version will create
//      a named shared buffer.
//
//  2)  They are all fully allocated immediately. I.e. their initial alloc size
//      is set to the maximum alloc size, which is the size of the structure.
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TTypedMBuf
//  PREFIX: tmbuf
// ----------------------------------------------------------------------------
template <class T> class TTypedMBuf :

    public TObject, public MDuplicable, public MStreamable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TTypedMBuf();

        TTypedMBuf
        (
            const   tCIDLib::EMemAccFlags   eAccess
        );

        TTypedMBuf
        (
            const   TResourceName&          rsnToUse
            , const tCIDLib::EMemAccFlags   eAccess = tCIDLib::EMemAcc_ReadWrite
            , const tCIDLib::ECreateActions eCreateFlags = tCIDLib::ECreateAct_OpenOrCreate
        );

        TTypedMBuf
        (
            const   TTypedMBuf<T>&          tmbufToCopy
        );
        
        ~TTypedMBuf();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        T* operator->();

        const T* operator->() const;

        const T& operator()() const;


    protected :
        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        tCIDLib::TVoid operator=(const TTypedMBuf<T>&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __pmbufData
        //      This is the pointer to the memory buffer object that we are
        //      wrapping. It some derivative of TMemBuf.
        // --------------------------------------------------------------------
        TMemBuf*        __pmbufData;
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  Constructors and destructors
// ----------------------------------------------------------------------------
template <class T> TTypedMBuf<T>::TTypedMBuf() :

    __pmbufData(0)
{
}

template <class T>
TTypedMBuf<T>::TTypedMBuf(const tCIDLib::EMemAccFlags eAccess) :

    __pmbufData(0)
{
    __pmbufData = new TSysBuf(sizeof(T), eAccess);
}


template <class T>
TTypedMBuf<T>::TTypedMBuf(  const   TResourceName&          rsnToUse
                            , const tCIDLib::EMemAccFlags   eAccess
                            , const tCIDLib::ECreateActions eCreateFlags) :
    __pmbufData(0)
{
    __pmbufData = new TSharedMemBuf
    (
        sizeof(T)
        , rsnToUse
        , eAccess
        , eCreateFlags
    );
}

template <class T> TTypedMBuf<T>::TTypedMBuf(const TTypedMBuf<T>& tmbufToCopy) :

    __pmbufData(0)
{
    // Duplicate the memory buffer object
    __pmbufData = ::pDupObject<TMemBuf>(tmbufToCopy.__pmbufData);
}

template <class T> TTypedMBuf<T>::~TTypedMBuf()
{
    delete __pmbufData;
}


// ----------------------------------------------------------------------------
//  Public operators
// ----------------------------------------------------------------------------
template <class T> T* TTypedMBuf<T>::operator->()
{
    return (T*)__pmbufData->pData();
}

template <class T> const T* TTypedMBuf<T>::operator->() const
{
    return (const T*)__pmbufData->pData();
}

template <class T> const T& TTypedMBuf<T>::operator()() const
{
    return *(const T*)__pmbufData->pData();
}


// ----------------------------------------------------------------------------
//  Protected, inherited methods
// ----------------------------------------------------------------------------

template <class T> tCIDLib::TVoid
TTypedMBuf<T>::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Destroy the existing buffer and read in the new one
    delete __pmbufData;
    __pmbufData = 0;

    ::PolymorphicRead(__pmbufData, strmToReadFrom);
}

template <class T> tCIDLib::TVoid
TTypedMBuf<T>::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    ::PolymorphicWrite(__pmbufData, strmToWriteTo);
}
