//
// NAME: CIDLib_TypeRegistry.Cpp
//
// DESCRIPTION: 
//
//  This module implements a couple of the static methods of the facility
//  class, TFacCIDLib. These are the ones related to the dynamic type
//  registration system. This module maintains an internal hash table (which
//  is just a simple and raw implementation because lots of low level stuff
//  depend upon it.) It is used to store registered TClass objects and their
//  factory functions.
//
//  A simple class is used to store the class objects and factory function
//  in the hash table. It is internal only.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/21/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//
//  1)  Note that we HAVE to make sure everything here gets created during
//      the 'pre constructor' module init because TClass objects all over
//      the place are going to call into here to register classes. Some
//      are almost certain to construct before this module does. So use
//      pointers and allocate during init where needed.
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
class   TRegNode;



// ----------------------------------------------------------------------------
//  Local, static data
//
//  __apnodeTable
//      This is the hash table for type names. It is an array of pointers
//      to nodes. Each pointer is a slot in the hash table and all of the
//      entries in that slot's linked list has the same hash value. The
//      hash table's size is driven by the modulus value used by the TClass
//      class' internally.
//
//  __pcrsLock
//      This is a critical section for controlling access to the registration
//      list. Since the locks are very short and are always within this process,
//      this should hopefully gain us some performance over a mutex.
// ----------------------------------------------------------------------------
static  TRegNode*           __apnodeTable[kCIDLib::c4ClassModulus];
static  TCriticalSection*   __pcrsLock;



// ----------------------------------------------------------------------------
//   CLASS: TRegNode
//  PREFIX: node
// ----------------------------------------------------------------------------
class TRegNode
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TRegNode
        (
            const   TString&                strToReg
            ,       tCIDLib::TObjFactory    pFactoryFunc
            ,       TRegNode* const         pNext   = 0
        );

        ~TRegNode() {}


        // --------------------------------------------------------------------
        //  Public data member
        //
        //  pszClassName
        //      This is the registered class name.
        //
        //  hshClass
        //      This is a fast access pointer to the internal hash of the
        //      class. This is gotten from the class object itself.
        //
        //  pFactory
        //      This is the factory function for this class. It will create
        //      a new object of this type. Its set from the one passed to
        //      the constructor.
        //
        //  pnodeNext
        //      This is a pointer to the next node in this slot. 0 if the
        //      end.
        // --------------------------------------------------------------------
        tCIDLib::Tch*           pszClassName;
        tCIDLib::THashVal       hshClass;
        tCIDLib::TObjFactory    pFactory;
        TRegNode*               pnodeNext;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TRegNode();

        TRegNode
        (
            const   TRegNode&               regToCopy
        );

        tCIDLib::TVoid operator=
        (
            const   TRegNode&               regToAssign
        );
};

TRegNode::TRegNode( const   TString&                strToReg
                    ,       tCIDLib::TObjFactory    pFactoryFunc
                    ,       TRegNode* const         pNext) :
    hshClass(0)
    , pFactory(pFactoryFunc)
    , pnodeNext(pNext)
    , pszClassName(0)
{
    // Replicate the passed name and calc and store its hash
    pszClassName = TRawStr::pszReplicate(strToReg.pszData());
    hshClass = TRawStr::hshHashStr(pszClassName, kCIDLib::c4ClassModulus);
}


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __pnodeFindByName
//
// DESCRIPTION:
//
//  This method will find the node that represents the passed name. If not
//  found it will log an exception, giving the passed method name and line
//  number.
// ---------------------------------------
//   INPUT: pszTypeName is the name to look up
//          pszMethod is the method to use if we log an exception
//          c4Line is the line number to use if the log an exception
//
//  OUTPUT: None
//
//  RETURN: A pointer to the node for the passed type
//
static TRegNode* __pnodeFindByName( const   tCIDLib::Tch* const pszTypeName
                                    ,       tCIDLib::TCard4     c4Line)
{
    //
    //  Lets find the type in the hash table. So we do a hash on the
    //  string, using the modulus of the TCIDLib class.
    //
    tCIDLib::THashVal hshFind = TRawStr::hshHashStr
    (
        pszTypeName
        , kCIDLib::c4ClassModulus
    );

    //
    //  If there is no list head at this slot, then of couse it does
    //  not exist.
    //
    if (__apnodeTable[hshFind])
    {
        TRegNode* pnodeCur = __apnodeTable[hshFind];
        while (pnodeCur)
        {
            if (pnodeCur->hshClass == hshFind)
            {
                if (!TRawStr::eCompareStr(pszTypeName, pnodeCur->pszClassName))
                    return pnodeCur;
            }
            pnodeCur = pnodeCur->pnodeNext;
        }
    }

    // We did not find it, so log and error
    facCIDLib.LogErr
    (
        __FILE__
        , c4Line
        , kCIDErrs::errcTReg_NotFound
        , tCIDLib::ESev_ProcessFatal
        , tCIDLib::EClass_AppError
        , TString(pszTypeName)
    );

    // Make compiler happy
    return 0;
}



// ----------------------------------------------------------------------------
//  Intrafacility functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _InitTermTypeRegistry
//
// DESCRIPTION:
//
//	This function is called from the CIDLib.Dll's initialization. It handles
//  any module init/term we need
// ---------------------------------------
//   INPUT: eInitTerm indicates what initialization phase we are in.
//          eGlobals indicates whether this is before constructors or
//              after destructors for globals.
//          modInit is a temp module object for this module.
//          c4MaxChars is the max chars that the failure reason buffer
//              can hold.
//
//  OUTPUT: pszFailReason is filled with the reason for a failure.
//
//  RETURN: None
//
tCIDLib::TVoid
_InitTermTypeRegistry(  const   tCIDLib::EInitTerm      eInitTerm
                        , const tCIDLib::EGlobalStates  eGlobals
                        , const TModule&                modInit
                        , const tCIDLib::TCard4         c4MaxChars
                        ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midGen_Unknown);
    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            //  Init the hash table entries
            tCIDLib::TCard4 c4Index;
            for (c4Index = 0; c4Index < kCIDLib::c4ClassModulus; c4Index++)
                __apnodeTable[c4Index] = 0;

            // Create the access critical section
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midTReg_CreateCritSec);
            __pcrsLock = new TCriticalSection;
        }
         else if ((eInitTerm == tCIDLib::EInitTerm_Terminate)
              &&  (eGlobals == tCIDLib::EGlobalState_After))
        {
            if (__pcrsLock)
            {
                pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midTReg_DeleteCritSec);
                delete __pcrsLock;
                __pcrsLock = 0;
            }
        }
    }

    catch(...)
    {
        TRawStr::CopyStr(pszFailReason, pszPhase, c4MaxChars);
        throw;
    }
}




// ----------------------------------------------------------------------------
//  TClass: Public, static methods
// ----------------------------------------------------------------------------

TClass TClass::clsForType(const TString& strTypeName)
{
    TCritSecLock crslAccess(__pcrsLock);

    // Find the node for the passed type name
    TRegNode* pnodeNew = __pnodeFindByName(strTypeName.pszData(), __LINE__);
    return TClass(pnodeNew->pszClassName);
}


TObject* TClass::pobjMakeNewOfType(const tCIDLib::Tch* const pszTypeName)
{
    TCritSecLock crslAccess(__pcrsLock);

    // Find the node for the passed type name
    TRegNode* pnodeNew = __pnodeFindByName(pszTypeName, __LINE__);
    return pnodeNew->pFactory();
}

TObject* TClass::pobjMakeNewOfType(const TString& strTypeName)
{
    TCritSecLock crslAccess(__pcrsLock);

    // Find the node for the passed type name
    TRegNode* pnodeNew = __pnodeFindByName(strTypeName.pszData(), __LINE__);
    return pnodeNew->pFactory();
}


tCIDLib::TVoid
TClass::RegisterClass(  const   TString&                strToRegister
                        ,       tCIDLib::TObjFactory    pFactoryFunc)
{
    TCritSecLock crslAccess(__pcrsLock);

    // Make sure that the class name is not empty
    if (strToRegister.bEmpty())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcTReg_InvalidName
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , strToRegister
        );
    }

    // Get the hash of the passed class name
    tCIDLib::THashVal hshOfClass = TRawStr::hshHashStr
    (
        strToRegister.pszData()
        , kCIDLib::c4ClassModulus
    );

    // Should not happen but check during development
    #if CID_DEBUG_ON
    if (hshOfClass >= kCIDLib::c4ClassModulus)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcTReg_InvalidHash
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , TCardinal(hshOfClass)
        );
    }
    #endif

    //
    //  Add this new class to the registry. First we use the hash value to
    //  find the slot in the hash table. Then search that slot's list for
    //  a match.
    //
    //  If the slot is open, then this is the new head of that slot. Else
    //  insert at the start of the list.
    //
    TRegNode* pHead = __apnodeTable[hshOfClass];
    if (!pHead)
        __apnodeTable[hshOfClass] = new TRegNode(strToRegister, pFactoryFunc);
     else
        __apnodeTable[hshOfClass] = new TRegNode(strToRegister, pFactoryFunc, pHead);
}
