//
// NAME: CIDLib_Type.Cpp
//
// DESCRIPTION: 
//
//  This module provides formatting for the facility specific enums from
//  CIDLib_Type.Hpp. Because CIDLib is special (i.e. the core platform
//  independent facility), its types and constants namespaces are actually
//  down in the kernel facility.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 03/27/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//
//  1)  Some enums are very commonly understood and therefore will be used
//      in very non-software senses, such as boolean values. Therefore, the
//      text for those types are loaded dyanmically in order to support
//      translation for them. Others are very tied to the source code and
//      anyone expected to understand the code (which is in english) would
//      understand them in their english form.
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Local static data
//
//  These are all data structures that indicate how to load text for
//  some enums that we want to be translatable.
// ----------------------------------------------------------------------------
static tCIDLib::TEnumTextMap<tCIDLib::TBoolean>  __aetmBoolean[] =
{
      { kCIDLib::False              , kCIDMsgs::midGen_False    , L"False" }
    , { kCIDLib::True               , kCIDMsgs::midGen_True     , L"True" }
};

static tCIDLib::TEnumTextMap<tCIDLib::EWeekDays>  __aetmDays[] =
{
      { tCIDLib::EWeekDay_Sunday    , kCIDMsgs::midDay_Sunday   , L"Sunday" }
    , { tCIDLib::EWeekDay_Monday    , kCIDMsgs::midDay_Monday   , L"Monday" }
    , { tCIDLib::EWeekDay_Tuesday   , kCIDMsgs::midDay_Tuesday  , L"Tuesday" }
    , { tCIDLib::EWeekDay_Wednesday , kCIDMsgs::midDay_Wednesday, L"Wednesday" }
    , { tCIDLib::EWeekDay_Thursday  , kCIDMsgs::midDay_Thursday , L"Thursday" }
    , { tCIDLib::EWeekDay_Friday    , kCIDMsgs::midDay_Friday   , L"Friday" }
    , { tCIDLib::EWeekDay_Saturday  , kCIDMsgs::midDay_Saturday , L"Saturday" }
};

static tCIDLib::TEnumTextMap<tCIDLib::EMonths>  __aetmMonths[] =
{
      { tCIDLib::EMonth_January     , kCIDMsgs::midMonth_Jan    , L"January" }
    , { tCIDLib::EMonth_February    , kCIDMsgs::midMonth_Feb    , L"February" }
    , { tCIDLib::EMonth_March       , kCIDMsgs::midMonth_Mar    , L"March" }
    , { tCIDLib::EMonth_April       , kCIDMsgs::midMonth_Apr    , L"April" }
    , { tCIDLib::EMonth_May         , kCIDMsgs::midMonth_May    , L"May" }
    , { tCIDLib::EMonth_June        , kCIDMsgs::midMonth_Jun    , L"June" }
    , { tCIDLib::EMonth_July        , kCIDMsgs::midMonth_Jul    , L"July" }
    , { tCIDLib::EMonth_August      , kCIDMsgs::midMonth_Aug    , L"August" }
    , { tCIDLib::EMonth_September   , kCIDMsgs::midMonth_Sep    , L"September" }
    , { tCIDLib::EMonth_October     , kCIDMsgs::midMonth_Oct    , L"October" }
    , { tCIDLib::EMonth_November    , kCIDMsgs::midMonth_Nov    , L"November" }
    , { tCIDLib::EMonth_December    , kCIDMsgs::midMonth_Dec    , L"December" }
};



// ----------------------------------------------------------------------------
//  Intra-facility methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
_InitTermThisType(  const   tCIDLib::EInitTerm      eInitTerm
                    , const tCIDLib::EGlobalStates  eGlobals
                    , const TModule&                modInit
                    , const tCIDLib::TCard4         c4MaxChars
                    ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midGen_Unknown);
    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            //
            //  Load up all of our strings. We just set up const pointers
            //  to the text, since they are already mapped into the address
            //  space anyway.
            //
            tCIDLib::TCard4         c4Index;
            const tCIDLib::Tch*     pszTmp;

            // Load the boolean values
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midType_LoadStrings);
            for (c4Index = 0; c4Index < c4ArrayElems(__aetmBoolean); c4Index++)
            {
                pszTmp = modInit.pszLoadCIDMsg(__aetmBoolean[c4Index].midText);
                if (pszTmp)
                    __aetmBoolean[c4Index].pszText = pszTmp;
            }

            // Load up the day values
            pszPhase = L"Loading text for day names";
            for (c4Index = 0; c4Index < c4ArrayElems(__aetmDays); c4Index++)
            {
                pszTmp = modInit.pszLoadCIDMsg(__aetmDays[c4Index].midText);
                if (pszTmp)
                    __aetmDays[c4Index].pszText = pszTmp;
            }

            // Load up the month values
            pszPhase = L"Loading text for month names";
            for (c4Index = 0; c4Index < c4ArrayElems(__aetmMonths); c4Index++)
            {
                pszTmp = modInit.pszLoadCIDMsg(__aetmMonths[c4Index].midText);
                if (pszTmp)
                    __aetmMonths[c4Index].pszText = pszTmp;
            }
        }
    }

    catch(...)
    {
        TRawStr::CopyStr(pszFailReason, pszPhase, c4MaxChars);
        throw;
    }
}


// ----------------------------------------------------------------------------
//  Global functions
// ----------------------------------------------------------------------------

TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EAngleTypes eAngle)
{
    if (eAngle == tCIDLib::EAngle_Radians)
        strmToWriteTo << L"Radians";
    else if (eAngle == tCIDLib::EAngle_Degrees)
        strmToWriteTo << L"Degrees";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eAngle)
            , TString(L"angle units")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EAudioCues eCue)
{
    if (eCue == tCIDLib::EAudioCue_Alert)
        strmToWriteTo << L"Alert";
    else if (eCue == tCIDLib::EAudioCue_Bell)
        strmToWriteTo << L"Bell";
    else if (eCue == tCIDLib::EAudioCue_Click)
        strmToWriteTo << L"Click";
    else if (eCue == tCIDLib::EAudioCue_Chime)
        strmToWriteTo << L"Chime";
    else if (eCue == tCIDLib::EAudioCue_Error)
        strmToWriteTo << L"Error";
    else if (eCue == tCIDLib::EAudioCue_HiPulse)
        strmToWriteTo << L"HiPulse";
    else if (eCue == tCIDLib::EAudioCue_Information)
        strmToWriteTo << L"Info";
    else if (eCue == tCIDLib::EAudioCue_LowPulse)
        strmToWriteTo << L"LoPulse";
    else if (eCue == tCIDLib::EAudioCue_MedPulse)
        strmToWriteTo << L"MedPulse";
    else if (eCue == tCIDLib::EAudioCue_Question)
        strmToWriteTo << L"Question";
    else if (eCue == tCIDLib::EAudioCue_Rising)
        strmToWriteTo << L"Rising";
    else if (eCue == tCIDLib::EAudioCue_Sinking)
        strmToWriteTo << L"Sinking";
    else if (eCue == tCIDLib::EAudioCue_Warning)
        strmToWriteTo << L"Warning";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eCue)
            , TString(L"EAudioCues")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::TBoolean bBool)
{
    if (bBool == kCIDLib::True)
        strmToWriteTo << __aetmBoolean[1].pszText;
    else if (bBool == kCIDLib::False)
        strmToWriteTo << __aetmBoolean[0].pszText;
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(bBool)
            , TString(L"TBoolean")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::ECPUTypes eCPUType)
{
    if (eCPUType == tCIDLib::ECPUType_386DX)
        strmToWriteTo << L"386DX";
    else if (eCPUType == tCIDLib::ECPUType_486DX)
        strmToWriteTo << L"486DX";
    else if (eCPUType == tCIDLib::ECPUType_Pentium)
        strmToWriteTo << L"Pentium";
    else if (eCPUType == tCIDLib::ECPUType_R4000)
        strmToWriteTo << L"R4000";
    else if (eCPUType == tCIDLib::ECPUType_Alpha)
        strmToWriteTo << L"Alpha";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eCPUType)
            , TString(L"ECPUTypes")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EDeleteModes eDelMode)
{
    if (eDelMode == tCIDLib::EDel_NoDelete)
        strmToWriteTo << L"No Delete";
    else if (eDelMode == tCIDLib::EDel_Default)
        strmToWriteTo << L"Default";
    else if (eDelMode == tCIDLib::EDel_Delete)
        strmToWriteTo << L"Delete";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eDelMode)
            , TString(L"eDelModes")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EEnds eAnEnd)
{
    if (eAnEnd == tCIDLib::EEnd_First)
        strmToWriteTo << L"First";
    else if (eAnEnd == tCIDLib::EEnd_Last)
        strmToWriteTo << L"Last";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eAnEnd)
            , TString(L"EEnds")
        );
    }
   
    return strmToWriteTo;
}


TTextStream &
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EErrClasses eClass)
{
    if (eClass == tCIDLib::EClass_None)
        strmToWriteTo << L"None";
    else if (eClass == tCIDLib::EClass_OutResource)
        strmToWriteTo << L"Out Resource";
    else if (eClass == tCIDLib::EClass_TempSituation)
        strmToWriteTo << L"Temp Situation";
    else if (eClass == tCIDLib::EClass_Authority)
        strmToWriteTo << L"Authority";
    else if (eClass == tCIDLib::EClass_Internal)
        strmToWriteTo << L"Internal";
    else if (eClass == tCIDLib::EClass_HardwareFailure)
        strmToWriteTo << L"Hardware Error";
    else if (eClass == tCIDLib::EClass_SystemFailure)
        strmToWriteTo << L"System Error";
    else if (eClass == tCIDLib::EClass_AppError)
        strmToWriteTo << L"App Error";
    else if (eClass == tCIDLib::EClass_NotFound)
        strmToWriteTo << L"Not Found";
    else if (eClass == tCIDLib::EClass_BadParms)
        strmToWriteTo << L"Bad Parms";
    else if (eClass == tCIDLib::EClass_Locked)
        strmToWriteTo << L"Locked";
    else if (eClass == tCIDLib::EClass_Media)
        strmToWriteTo << L"Media";
    else if (eClass == tCIDLib::EClass_Already)
        strmToWriteTo << L"Already";
    else if (eClass == tCIDLib::EClass_Unknown)
        strmToWriteTo << L"Unknown";
    else if (eClass == tCIDLib::EClass_CantDo)
        strmToWriteTo << L"Cant Do";
    else if (eClass == tCIDLib::EClass_Timeout)
        strmToWriteTo << L"Time Out";
    else if (eClass == tCIDLib::EClass_DynamicType)
        strmToWriteTo << L"Dynamic Type";
    else if (eClass == tCIDLib::EClass_TypeMatch)
        strmToWriteTo << L"Type Match";
    else if (eClass == tCIDLib::EClass_StreamFormat)
        strmToWriteTo << L"Stream Format";
    else if (eClass == tCIDLib::EClass_UserInput)
        strmToWriteTo << L"User Input";
    else if (eClass == tCIDLib::EClass_NotReady)
        strmToWriteTo << L"Not Ready";
    else if (eClass == tCIDLib::EClass_AppStatus)
        strmToWriteTo << L"App Status";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eClass)
            , TString(L"EErrClasses")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EExitCodes eCode)
{
    //
    //  If its one of the named values, then show the text. Else just
    //  display the numeric value, since this type is often just used to
    //  cast random values.
    //
    if ((eCode >= tCIDLib::EExitCodes_FirstNamed)
    &&  (eCode <= tCIDLib::EExitCodes_LastNamed))
    {
        if (eCode == tCIDLib::EExit_AdminRestart)
            strmToWriteTo << L"Admin Restart";
        else if (eCode == tCIDLib::EExit_FatalError)
            strmToWriteTo << L"Fatal Error";
        else if (eCode == tCIDLib::EExit_RuntimeError)
            strmToWriteTo << L"Runtime Error";
        else if (eCode == tCIDLib::EExit_SystemException)
            strmToWriteTo << L"System Exception";
        else if (eCode == tCIDLib::EExit_BadParameters)
            strmToWriteTo << L"Bad Parameters";
        else if (eCode == tCIDLib::EExit_ResourceAccess)
            strmToWriteTo << L"Resource Access";
        else if (eCode == tCIDLib::EExit_PermissionLevel)
            strmToWriteTo << L"Permission Level";
        else if (eCode == tCIDLib::EExit_NotFound)
            strmToWriteTo << L"Not Found";
        else if (eCode == tCIDLib::EExit_InitFailed)
            strmToWriteTo << L"Init Failed";
    }
     else
    {
        strmToWriteTo << TInteger(eCode);
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EFileAttrs eAttrs)
{
    TString   strAttr(kCIDLib::pszEmptyZStr, 16);

    if (eAttrs & tCIDLib::EFileAttr_Directory)
    {
        strAttr = "<DIR>";
    }
     else
    {
      strAttr = ".... ";

      if (eAttrs & tCIDLib::EFileAttr_ReadOnly)
          strAttr[0] = 'R';

      if (eAttrs & tCIDLib::EFileAttr_Hidden)
          strAttr[1] = 'H';

      if (eAttrs & tCIDLib::EFileAttr_System)
          strAttr[2] = 'S';

      if (eAttrs & tCIDLib::EFileAttr_Archive)
          strAttr[3] = 'A';
    }
    strmToWriteTo << strAttr.pszData();
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EHJustify eJustify)
{
    if (eJustify == tCIDLib::EHJustify_Left)
        strmToWriteTo << L"Left";
    else if (eJustify == tCIDLib::EHJustify_Right)
        strmToWriteTo << L"Right";
    else if (eJustify == tCIDLib::EHJustify_Center)
        strmToWriteTo << L"Center";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eJustify)
            , TString(L"EHJustify")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::ELockStates eState)
{
    if (eState == tCIDLib::ELockState_Unlocked)
        strmToWriteTo << L"Unlocked";
    else if (eState == tCIDLib::ELockState_Locked)
        strmToWriteTo << L"Locked";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eState)
            , TString(L"ELockStates")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::ELogMapModes eMode)
{
    if (eMode == tCIDLib::ELogMapMode_Regular)
        strmToWriteTo << L"Regular";
    else if (eMode == tCIDLib::ELogMapMode_Spread)
        strmToWriteTo << L"Spread";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eMode)
            , TString(L"ELogMapModes")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EMatrixElems eElem)
{
    if (eElem == tCIDLib::EMatrix_11)
        strmToWriteTo << L"1-1";
    else if (eElem == tCIDLib::EMatrix_12)
        strmToWriteTo << L"1-2";
    else if (eElem == tCIDLib::EMatrix_13)
        strmToWriteTo << L"1-3";
    else if (eElem == tCIDLib::EMatrix_14)
        strmToWriteTo << L"1-4";
    else if (eElem == tCIDLib::EMatrix_21)
        strmToWriteTo << L"2-1";
    else if (eElem == tCIDLib::EMatrix_22)
        strmToWriteTo << L"2-2";
    else if (eElem == tCIDLib::EMatrix_23)
        strmToWriteTo << L"2-3";
    else if (eElem == tCIDLib::EMatrix_24)
        strmToWriteTo << L"2-4";
    else if (eElem == tCIDLib::EMatrix_31)
        strmToWriteTo << L"3-1";
    else if (eElem == tCIDLib::EMatrix_32)
        strmToWriteTo << L"3-2";
    else if (eElem == tCIDLib::EMatrix_33)
        strmToWriteTo << L"3-3";
    else if (eElem == tCIDLib::EMatrix_34)
        strmToWriteTo << L"3-4";
    else if (eElem == tCIDLib::EMatrix_Xt)
        strmToWriteTo << L"Xt";
    else if (eElem == tCIDLib::EMatrix_Yt)
        strmToWriteTo << L"Yt";
    else if (eElem == tCIDLib::EMatrix_Zt)
        strmToWriteTo << L"Zt";
    else if (eElem == tCIDLib::EMatrix_44)
        strmToWriteTo << L"4-4";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eElem)
            , TString(L"EMatrixElems")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EMetricTypes eMetricType)
{
    if (eMetricType == tCIDLib::EMetric_Accum)
        strmToWriteTo << L"Accumulator";
     else if (eMetricType == tCIDLib::EMetric_Cardinal)
        strmToWriteTo << L"Card Value";
     else if (eMetricType == tCIDLib::EMetric_FlagSet)
        strmToWriteTo << L"Flag Set";
     else if (eMetricType == tCIDLib::EMetric_Integer)
        strmToWriteTo << L"Int Value";
     else if (eMetricType == tCIDLib::EMetric_TickCounter)
        strmToWriteTo << L"Tick Counter";
     else if (eMetricType == tCIDLib::EMetric_Toggle)
        strmToWriteTo << L"Toggle";
     else if (eMetricType == tCIDLib::EMetric_PeakMeter)
        strmToWriteTo << L"Peak Meter";
     else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eMetricType)
            , TString(L"EMetricTypes")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EMonths eMonth)
{
    if ((eMonth >= tCIDLib::EMonth_January)
    &&  (eMonth <= tCIDLib::EMonth_December))
    {
        strmToWriteTo << __aetmMonths[eMonth].pszText;
    }
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eMonth)
            , TString(L"EMonths")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EProcStates eProcState)
{
    if (eProcState == tCIDLib::EProcState_Initializing)
        strmToWriteTo << L"Initializing";
    else if (eProcState == tCIDLib::EProcState_EnteredMainThread)
        strmToWriteTo << L"Main Thread";
    else if (eProcState == tCIDLib::EProcState_Ready)
        strmToWriteTo << L"Running";
    else if (eProcState == tCIDLib::EProcState_Terminating)
        strmToWriteTo << L"Terminating";
    else if (eProcState == tCIDLib::EProcState_Dead)
        strmToWriteTo << L"Dead";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eProcState)
            , TString(L"EProcStates")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EPrioClasses ePrioClass)
{
    if (ePrioClass == tCIDLib::EPrioClass_IdleTime)
        strmToWriteTo << L"Idle";
    else if (ePrioClass == tCIDLib::EPrioClass_Normal)
        strmToWriteTo << L"Regular";
    else if (ePrioClass == tCIDLib::EPrioClass_High)
        strmToWriteTo << L"Critical";
    else if (ePrioClass == tCIDLib::EPrioClass_RealTime)
        strmToWriteTo << L"Fgn Server";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(ePrioClass)
            , TString(L"EPrioClasses")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EPrioLevels ePrioLevel)
{
    if (ePrioLevel == tCIDLib::EPrioLevel_Lowest)
        strmToWriteTo << L"Lowest";
    else if (ePrioLevel == tCIDLib::EPrioLevel_BelowNormal)
        strmToWriteTo << L"Below Normal";
    else if (ePrioLevel == tCIDLib::EPrioLevel_Normal)
        strmToWriteTo << L"Normal";
    else if (ePrioLevel == tCIDLib::EPrioLevel_AboveNormal)
        strmToWriteTo << L"Above Normal";
    else if (ePrioLevel == tCIDLib::EPrioLevel_Highest)
        strmToWriteTo << L"Highest";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(ePrioLevel)
            , TString(L"EPrioLevels")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::ERadices eRadix)
{
    if (eRadix == tCIDLib::ERadix_Bin)
        strmToWriteTo << L"Binary";
    else if (eRadix == tCIDLib::ERadix_Oct)
        strmToWriteTo << L"Octal";
    else if (eRadix == tCIDLib::ERadix_Dec)
        strmToWriteTo << L"Decimal";
    else if (eRadix == tCIDLib::ERadix_Hex)
        strmToWriteTo << L"Hex";
    else
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eRadix)
            , TString(L"ERadices")
        );
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::ERscTypes eType)
{
    if (eType == tCIDLib::ERscType_None)
        strmToWriteTo << L"None";
    else if (eType == tCIDLib::ERscType_Pointer)
        strmToWriteTo << L"Pointer";
    else if (eType == tCIDLib::ERscType_Bitmap)
        strmToWriteTo << L"Bitmap";
    else if (eType == tCIDLib::ERscType_Icon)
        strmToWriteTo << L"Icon";
    else if (eType == tCIDLib::ERscType_Menu)
        strmToWriteTo << L"Menu";
    else if (eType == tCIDLib::ERscType_String)
        strmToWriteTo << L"String";
    else if (eType == tCIDLib::ERscType_FontDir)
        strmToWriteTo << L"Font Dir";
    else if (eType == tCIDLib::ERscType_Font)
        strmToWriteTo << L"Font";
    else if (eType == tCIDLib::ERscType_AccelTable)
        strmToWriteTo << L"Accel Table";
    else if (eType == tCIDLib::ERscType_RcData)
        strmToWriteTo << L"Rc Data";
    else if (eType == tCIDLib::ERscType_Message)
        strmToWriteTo << L"Message";
    else if (eType == tCIDLib::ERscType_GroupCursor)
        strmToWriteTo << L"Group Cursor";
    else if (eType == tCIDLib::ERscType_GroupIcon)
        strmToWriteTo << L"Group Icon";
    else if (eType == tCIDLib::ERscType_Version)
        strmToWriteTo << L"Group Version";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eType)
            , TString(L"ERscTypes")
        );
    }
    return strmToWriteTo;
}



TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::ESeverities eSeverity)
{
    if (eSeverity == tCIDLib::ESev_Information)
        strmToWriteTo << L"Info";
    else if (eSeverity == tCIDLib::ESev_Warning)
        strmToWriteTo << L"Warning";
    else if (eSeverity == tCIDLib::ESev_APIFailed)
        strmToWriteTo << L"API Failed";
    else if (eSeverity == tCIDLib::ESev_ProcessFatal)
        strmToWriteTo << L"Proc Fatal";
    else if (eSeverity == tCIDLib::ESev_SystemFatal)
        strmToWriteTo << L"Sys Fatal";
    else if (eSeverity == tCIDLib::ESev_Status)
        strmToWriteTo << L"Status";
     else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eSeverity)
            , TString(L"eErrSeverities")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EShareStates eState)
{
    if (eState == tCIDLib::EShareState_Unshared)
        strmToWriteTo << L"Unshared";
    else if (eState == tCIDLib::EShareState_Shared)
        strmToWriteTo << L"Shared";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eState)
            , TString(L"EShareStates")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::ESymmetries eSymmetry)
{
    if (eSymmetry == tCIDLib::ESymmetry_None)
        strmToWriteTo << L"None";
    else if (eSymmetry == tCIDLib::ESymmetry_VerticalFlip)
        strmToWriteTo << L"Vertical";
    else if (eSymmetry == tCIDLib::ESymmetry_HorizontalFlip)
        strmToWriteTo << L"Horizontal";
    else if (eSymmetry == tCIDLib::ESymmetry_QuadrantFlip)
        strmToWriteTo << L"Quadrant";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eSymmetry)
            , TString(L"symmetry")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::ESysExcepts eExcept)
{
    if (eExcept == tCIDLib::ESysExcept_Unknown)
        strmToWriteTo << L"Unknown";
    else if (eExcept == tCIDLib::ESysExcept_BreakPoint)
        strmToWriteTo << L"Break Point";
    else if (eExcept == tCIDLib::ESysExcept_DataAlignment)
        strmToWriteTo << L"Data Alignment";
    else if (eExcept == tCIDLib::ESysExcept_Float_Denormal)
        strmToWriteTo << L"Denormal Float";
    else if (eExcept == tCIDLib::ESysExcept_Float_DivByZero)
        strmToWriteTo << L"Float Div By Zero";
    else if (eExcept == tCIDLib::ESysExcept_Float_InexactResult)
        strmToWriteTo << L"Inexact Result";
    else if (eExcept == tCIDLib::ESysExcept_Float_InvalidOp)
        strmToWriteTo << L"Float Invalid Op";
    else if (eExcept == tCIDLib::ESysExcept_Float_Overflow)
        strmToWriteTo << L"Float Overflow";
    else if (eExcept == tCIDLib::ESysExcept_Float_Stack)
        strmToWriteTo << L"Float Stack";
    else if (eExcept == tCIDLib::ESysExcept_Float_Underflow)
        strmToWriteTo << L"Float Underflow";
    else if (eExcept == tCIDLib::ESysExcept_Int_DivByZero)
        strmToWriteTo << L"Int Div By Zero";
    else if (eExcept == tCIDLib::ESysExcept_Int_Overflow)
        strmToWriteTo << L"Int Overflow";
    else if (eExcept == tCIDLib::ESysExcept_PageLoadFailed)
        strmToWriteTo << L"Page Load Failed";
    else if (eExcept == tCIDLib::ESysExcept_ReadAccess)
        strmToWriteTo << L"Read Access";
    else if (eExcept == tCIDLib::ESysExcept_StackOverflow)
        strmToWriteTo << L"Stack Overflow";
    else if (eExcept == tCIDLib::ESysExcept_WriteAccess)
        strmToWriteTo << L"Write Access";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eExcept)
            , TString(L"system exception")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::ETrailFormats eTrail)
{
    if (eTrail == tCIDLib::ETrail_Zeroes)
        strmToWriteTo << L"Zeros";
    else if (eTrail == tCIDLib::ETrail_Spaces)
        strmToWriteTo << L"Spaces";
    else if (eTrail == tCIDLib::ETrail_Ignore)
        strmToWriteTo << L"Ignore";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eTrail)
            , TString(L"ETrailFormats")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::ETriStates eState)
{
    if (eState == tCIDLib::ETriState_Off)
        strmToWriteTo << L"Off";
    else if (eState == tCIDLib::ETriState_On)
        strmToWriteTo << L"On";
    else if (eState == tCIDLib::ETriState_Kinda)
        strmToWriteTo << L"Kinda";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eState)
            , TString(L"ETriStates")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EVJustify eJustify)
{
    if (eJustify == tCIDLib::EVJustify_Bottom)
        strmToWriteTo << L"Bottom";
    else if (eJustify == tCIDLib::EVJustify_Top)
        strmToWriteTo << L"Top";
    else if (eJustify == tCIDLib::EVJustify_Center)
        strmToWriteTo << L"Center";
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eJustify)
            , TString(L"EVJustify")
        );
    }
    return strmToWriteTo;
}


TTextStream&
operator<<(TTextStream& strmToWriteTo, const tCIDLib::EWeekDays eWeekDay)
{
    if ((eWeekDay >= tCIDLib::EWeekDay_Sunday)
    &&  (eWeekDay <= tCIDLib::EWeekDay_Saturday))
    {
        strmToWriteTo << __aetmDays[eWeekDay].pszText;
    }
    else
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEnum_BadValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eWeekDay)
            , TString(L"EWeekDays")
        );
    }
    return strmToWriteTo;
}
