//
// NAME: CIDLib_Time.Hpp
//
// DESCRIPTION:
//
//  This module implements the TTime class. TTime represents a time stamp
//  that extends from 01/01/1601. Its raw form, gotten via i8Time(), is the
//  number of 100 nanosecond intervals since that base time.
//
//  A very simple derivative is also provided, TCurrentTime, that just sets
//  itself to the current time. It adds no data or methods to the base
//  class. Its just very convenient and self documenting to have.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 07/05/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  Time values (in the 100 nanosecond relative stamp form) cannot be
//      negative ever. If you want to find the difference between two
//      times and they might be negative, us the i8Difference() method
//      to get the difference.
//



#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TTime
//  PREFIX: tm
// ----------------------------------------------------------------------------
class CIDLIBEXP TTime :

    public TObject, public MDuplicable, public MFormattable, public MStreamable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TTime();

        TTime
        (
            const   TTime&                  tmToCopy
        );

        TTime
        (
            const   tCIDLib::TInt8          i8Time
        );

        TTime
        (
            const   tCIDLib::TCard4         c4Year
            , const tCIDLib::EMonths        eMonth
            , const tCIDLib::TCard4         c4Day
            , const tCIDLib::TCard4         c4Hours = 0UL
            , const tCIDLib::TCard4         c4Minutes = 0UL
            , const tCIDLib::TCard4         c4Seconds = 0UL
            , const tCIDLib::TCard4         c4Hundredths = 0UL
        );

        ~TTime();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        tCIDLib::TBoolean operator==
        (
            const   TTime&                  tmToCompare
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TTime&                  tmToCompare
        )   const;

        TTime& operator=
        (
            const   TTime&                  tmToAssign
        );

        tCIDLib::TVoid operator+=
        (
            const   TTime&                  tmToAdd
        );

        tCIDLib::TVoid operator+=
        (
            const   tCIDLib::TInt8&         i8ToAdd
        );

        tCIDLib::TVoid operator-=
        (
            const   TTime&                  tmToSubtract
        );

        tCIDLib::TVoid operator-=
        (
            const   tCIDLib::TInt8&         i8ToSubtract
        );

        friend TTime CIDLIBEXP operator+
        (
            const   TTime&                  tmLHS
            , const TTime&                  tmRHS
        );

        friend TTime CIDLIBEXP operator-
        (
            const   TTime&                  tmLHS
            , const TTime&                  tmRHS
        );

        friend tCIDLib::TBoolean CIDLIBEXP operator<
        (
            const   TTime&                  tmLHS
            , const TTime&                  tmRHS
        );

        friend tCIDLib::TBoolean CIDLIBEXP operator<=
        (
            const   TTime&                  tmLHS
            , const TTime&                  tmRHS
        );

        friend tCIDLib::TBoolean CIDLIBEXP operator>
        (
            const   TTime&                  tmLHS
            , const TTime&                  tmRHS
        );

        friend tCIDLib::TBoolean CIDLIBEXP operator>=
        (
            const   TTime&                  tmLHS
            , const TTime&                  tmRHS
        );


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TCard4 c4AsTimeInfo
        (
                    tCIDLib::TCard4&        c4Hour
            ,       tCIDLib::TCard4&        c4Minute
            ,       tCIDLib::TCard4&        c4Second    = NUL_TCard4
        )   const;

        tCIDLib::TCard4 c4JulianDate() const;

        tCIDLib::TInt8 i8Difference
        (
            const   TTime&                  tmToSubtract
        )   const;

        tCIDLib::EWeekDays eAsDateInfo
        (
                    tCIDLib::TCard4&        c4Year
            ,       tCIDLib::EMonths&       eMonth
            ,       tCIDLib::TCard4&        c4Day
        )   const;

        tCIDLib::EWeekDays eDayOfWeek() const;

        tCIDLib::TVoid FromDetails
        (
            const   tCIDLib::TCard4         c4Year
            , const tCIDLib::EMonths        eMonth
            , const tCIDLib::TCard4         c4Day
            , const tCIDLib::TCard4         c4Hour      = 0
            , const tCIDLib::TCard4         c4Minute    = 0
            , const tCIDLib::TCard4         c4Second    = 0
            , const tCIDLib::TCard4         c4MilliSecs = 0
        );

        tCIDLib::TVoid FormatToStr
        (
                    TString&                strToWriteTo
            , const TString&                strFormat
        )   const;

        tCIDLib::TInt8 i8MilliSeconds() const;

        tCIDLib::TInt8 i8Time() const;

        tCIDLib::TVoid SetToNow();

        tCIDLib::TVoid SetToMidnight();

        const TString& strDefaultFormat() const;

        const TString& strDefaultFormat
        (
            const   TString&                strNewFormat
        );


    protected   :
        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __ktmsThis
        //      This is the platform time stamp wrapper object that contains
        //      the actual time stamp info.
        //
        //  __strDefFormat
        //      This is the formatting option to use when formating the
        //      object via the standard formatting methods (which don't
        //      allow any class specific parameters.) Defaulted in the
        //      constructor.
        // --------------------------------------------------------------------
        TKrnlTimeStamp          __ktmsThis;
        TString                 __strDefFormat;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TTime,TObject)
        DefPolyDup(TTime)
};


// ----------------------------------------------------------------------------
//   CLASS: TCurrentTime
//  PREFIX: tm
// ----------------------------------------------------------------------------
class CIDLIBEXP TCurrentTime : public TTime
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TCurrentTime();

        TCurrentTime
        (
            const   TCurrentTime&           tmToCopy
        );

        ~TCurrentTime();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TCurrentTime& operator=
        (
            const   TCurrentTime&           tmToAssign
        );

        tCIDLib::TBoolean operator==
        (
            const   TCurrentTime&           tmToComprare
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TCurrentTime&           tmToCompare
        )   const;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TCurrentTime,TTime)
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  TTime: Constructors and destructors
// ----------------------------------------------------------------------------
inline TTime::~TTime()
{
}


// ----------------------------------------------------------------------------
//  TTime: Public operators
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TTime::operator!=(const TTime& tmToTest) const
{
    return !operator==(tmToTest);
}

inline tCIDLib::TBoolean operator<(const TTime& tmLHS, const TTime& tmRHS)
{
    return tmLHS.__ktmsThis.i8Time() < tmRHS.__ktmsThis.i8Time();
}

inline tCIDLib::TBoolean operator<=(const TTime& tmLHS, const TTime& tmRHS)
{
    return tmLHS.__ktmsThis.i8Time() <= tmRHS.__ktmsThis.i8Time();
}

inline tCIDLib::TBoolean operator>(const TTime& tmLHS, const TTime& tmRHS)
{
    return tmLHS.__ktmsThis.i8Time() > tmRHS.__ktmsThis.i8Time();
}

inline tCIDLib::TBoolean operator>=(const TTime& tmLHS, const TTime& tmRHS)
{
    return tmLHS.__ktmsThis.i8Time() >= tmRHS.__ktmsThis.i8Time();
}


// ----------------------------------------------------------------------------
//  TTime: Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TCard4 TTime::c4JulianDate() const
{
    return __ktmsThis.c4JulianDate();
}

inline tCIDLib::TInt8 TTime::i8Difference(const TTime& tmToSubtract) const
{
    return __ktmsThis.i8Difference(tmToSubtract.__ktmsThis);
}

inline tCIDLib::EWeekDays TTime::eDayOfWeek() const
{
    return __ktmsThis.eDayOfWeek();
}

inline tCIDLib::TInt8 TTime::i8MilliSeconds() const
{
    return __ktmsThis.i8MilliSeconds();
}

inline tCIDLib::TInt8 TTime::i8Time() const
{
    return __ktmsThis.i8Time();
}

inline tCIDLib::TVoid TTime::SetToNow()
{
    __ktmsThis.SetToNow();
}

inline tCIDLib::TVoid TTime::SetToMidnight()
{
    __ktmsThis.SetToMidnight();
}

inline const TString& TTime::strDefaultFormat() const
{
    return __strDefFormat;
}

inline const TString& TTime::strDefaultFormat(const TString& strNewDefault)
{
    __strDefFormat = strNewDefault;
    return __strDefFormat;
}



// ----------------------------------------------------------------------------
//  TCurrentTime: Constructors and Destructors
// ----------------------------------------------------------------------------
inline TCurrentTime::TCurrentTime()
{
    SetToNow();
}

inline TCurrentTime::TCurrentTime(const TCurrentTime& tmToCopy) :

    TTime(tmToCopy)
{
}

inline TCurrentTime::~TCurrentTime()
{
}


// ----------------------------------------------------------------------------
//  TCurrentTime: Public operators
// ----------------------------------------------------------------------------

inline TCurrentTime& TCurrentTime::operator=(const TCurrentTime& tmToAssign)
{
    TParent::operator=(tmToAssign);
    return *this;
}

inline tCIDLib::TBoolean
TCurrentTime::operator==(const TCurrentTime& tmToCompare) const
{
    return TParent::operator==(tmToCompare);
}

inline tCIDLib::TBoolean
TCurrentTime::operator!=(const TCurrentTime& tmToCompare) const
{
    return !TParent::operator==(tmToCompare);
}
