//
// NAME: CIDLib_Time.Cpp
//
// DESCRIPTION:
//
//  This module implements the basic time stamp class, TTime, and a simple
//  derivative TCurrentTime.
//
//
// AUTHOR: Dean Roddey
//
// CREATE TDate: 07/05/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData2(TTime,TObject)
RTTIData2(TCurrentTime,TTime)


// -----------------------------------------------------------------------------
//  Local, static data
//
//  __pszDefFormat
//      This is the default format string used in the constructors.
// -----------------------------------------------------------------------------
static const tCIDLib::Tch* const __pszDefFormat = L"%(M,2,0)/%(D,2,0)/%(Y)";



// -----------------------------------------------------------------------------
//   CLASS: TTime
//  PREFIX: tm
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTime: Public, virtual methods
// -----------------------------------------------------------------------------

TTime::TTime() :

    __strDefFormat(__pszDefFormat)
{
}

TTime::TTime(const TTime& tmToCopy) :

    __ktmsThis(tmToCopy.__ktmsThis)
    , __strDefFormat(__pszDefFormat)
{
}

TTime::TTime(   const   tCIDLib::TCard4     c4Year
                , const tCIDLib::EMonths    eMonth
                , const tCIDLib::TCard4     c4Day
                , const tCIDLib::TCard4     c4Hours
                , const tCIDLib::TCard4     c4Minutes
                , const tCIDLib::TCard4     c4Seconds
                , const tCIDLib::TCard4     c4Hundredths) :

    __strDefFormat(__pszDefFormat)
{
    try
    {
        __ktmsThis.FromDetails
        (
            c4Year
            , eMonth
            , c4Day
            , c4Hours
            , c4Minutes
            , c4Seconds
            , c4Hundredths
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcTime_BadDetails
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }
}

TTime::TTime(const tCIDLib::TInt8 i8Time) :

    __ktmsThis(i8Time)
    , __strDefFormat(__pszDefFormat)
{
}



// -----------------------------------------------------------------------------
//  TTime: Public operators
// -----------------------------------------------------------------------------

tCIDLib::TBoolean TTime::operator==(const TTime& tmToTest) const
{
    if (__ktmsThis.i8Time() != tmToTest.__ktmsThis.i8Time())
        return kCIDLib::False;

    if (__strDefFormat != tmToTest.__strDefFormat)
        return kCIDLib::False;

    // Everything matched, so return true
    return kCIDLib::True;
}


TTime& TTime::operator=(const TTime& tmToAssign)
{
    if (&tmToAssign == this)
        return *this;

    __strDefFormat = tmToAssign.__strDefFormat;
    __ktmsThis   = tmToAssign.__ktmsThis;

    return *this;
}


tCIDLib::TVoid TTime::operator+=(const TTime& tmToAdd)
{
    tCIDLib::TInt8 i8Res =  __ktmsThis.i8Time()
                            + tmToAdd.__ktmsThis.i8Time();
    if (i8Res < 0)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcTime_NegativeTime
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }
    __ktmsThis = i8Res;
}

tCIDLib::TVoid TTime::operator+=(const tCIDLib::TInt8& i8ToAdd)
{
    tCIDLib::TInt8 i8Res = __ktmsThis.i8Time() + i8ToAdd;

    if (i8Res < 0)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcTime_NegativeTime
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }
    __ktmsThis = i8Res;
}


tCIDLib::TVoid TTime::operator-=(const TTime& tmToSubtract)
{
    tCIDLib::TInt8 i8Res =  __ktmsThis.i8Time()
                            - tmToSubtract.__ktmsThis.i8Time();
    if (i8Res < 0)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcTime_NegativeTime
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }
    __ktmsThis = i8Res;
}

tCIDLib::TVoid TTime::operator-=(const tCIDLib::TInt8& i8ToAdd)
{
    tCIDLib::TInt8 i8Res = __ktmsThis.i8Time() - i8ToAdd;

    if (i8Res < 0)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcTime_NegativeTime
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }
    __ktmsThis = i8Res;
}


TTime operator+(const TTime& tmLHS, const TTime& tmRHS)
{
    tCIDLib::TInt8 i8Res =  tmLHS.__ktmsThis.i8Time()
                            + tmRHS.__ktmsThis.i8Time();
    if (i8Res < 0)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcTime_NegativeTime
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }
    return TTime(i8Res);
}

TTime operator-(const TTime& tmLHS, const TTime& tmRHS)
{
    tCIDLib::TInt8 i8Res =  tmLHS.__ktmsThis.i8Time()
                            - tmRHS.__ktmsThis.i8Time();
    if (i8Res < 0)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcTime_NegativeTime
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }
    return TTime(i8Res);
}


// -----------------------------------------------------------------------------
//  TTime: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TCard4
TTime::c4AsTimeInfo(tCIDLib::TCard4&    c4Hour
                    , tCIDLib::TCard4&  c4Minute
                    , tCIDLib::TCard4&  c4Second) const
{
    try
    {
        return __ktmsThis.c4AsTimeInfo(c4Hour, c4Minute, c4Second);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcTime_AsTime
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}


tCIDLib::EWeekDays
TTime::eAsDateInfo( tCIDLib::TCard4&    c4Year
                    , tCIDLib::EMonths& eMonth
                    , tCIDLib::TCard4&  c4Day) const
{
    try
    {
        return __ktmsThis.eAsDateInfo(c4Year, eMonth, c4Day);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcTime_AsDate
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}


tCIDLib::TVoid
TTime::FromDetails( const   tCIDLib::TCard4     c4Year
                    , const tCIDLib::EMonths    eMonth
                    , const tCIDLib::TCard4     c4Day
                    , const tCIDLib::TCard4     c4Hour
                    , const tCIDLib::TCard4     c4Minute
                    , const tCIDLib::TCard4     c4Second
                    , const tCIDLib::TCard4     c4MilliSecs)
{
    try
    {
        __ktmsThis.FromDetails
        (
            c4Year
            , eMonth
            , c4Day
            , c4Hour
            , c4Minute
            , c4Second
            , c4MilliSecs
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcTime_BadDetails
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }
}


tCIDLib::TVoid 
TTime::FormatToStr(TString& strToWriteTo, const TString& strFormat) const
{
    // Start by moving the format string to the target
    strToWriteTo = strFormat;

    // Convert the time to the detail info
    tCIDLib::TCard4     c4Year;
    tCIDLib::EMonths    eMonth;
    tCIDLib::TCard4     c4Day;
    tCIDLib::EWeekDays  eWeekDay;
    tCIDLib::TCard4     c4Hour;
    tCIDLib::TCard4     c4Minute;
    tCIDLib::TCard4     c4Second;
    tCIDLib::TCard4     c4MilliSecs;

    tCIDLib::TErrCode   errcRet;
    try
    {
        errcRet = kCIDErrs::errcTime_AsDate;
        eWeekDay = __ktmsThis.eAsDateInfo(c4Year, eMonth, c4Day);

        errcRet = kCIDErrs::errcTime_AsTime;
        c4MilliSecs = __ktmsThis.c4AsTimeInfo(c4Hour, c4Minute, c4Second);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , errcRet
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    //
    //  Now search for each token type and replace it with the correct
    //  value, if found.
    //
    TStringStream strmTmp(64);
    if (strToWriteTo.bTokenExists(L'M'))
        strToWriteTo.ReplaceToken(tCIDLib::TInt1(eMonth+1), L'M');

    if (strToWriteTo.bTokenExists(L'm'))
    {
        strmTmp.Reset();
        strmTmp << eMonth;
        strToWriteTo.ReplaceToken(strmTmp.strData(), L'm');
    }

    if (strToWriteTo.bTokenExists(L'D'))
        strToWriteTo.ReplaceToken(c4Day, L'D');

    if (strToWriteTo.bTokenExists(L'd'))
    {
        strmTmp.Reset();
        strmTmp << eWeekDay;
        strToWriteTo.ReplaceToken(strmTmp.strData(), L'd');
    }

    if (strToWriteTo.bTokenExists(L'Y'))
        strToWriteTo.ReplaceToken(c4Year, L'Y');

    if (strToWriteTo.bTokenExists(L'y'))
    {
        tCIDLib::TCard4 c4Tmp = c4Year;
        if (c4Year > 1999)
            c4Tmp -= 2000;
        else
            c4Tmp -= 1900;
        strToWriteTo.ReplaceToken(c4Tmp, L'y');
    }

    if (strToWriteTo.bTokenExists(L'R'))
        strToWriteTo.ReplaceToken(c4Hour, L'R');

    if (strToWriteTo.bTokenExists(L'r'))
    {
        tCIDLib::TCard4 c4Tmp = c4Hour;
        if (c4Tmp >= 12)
            c4Tmp -= 12;
        strToWriteTo.ReplaceToken(c4Tmp, L'r');
    }

    if (strToWriteTo.bTokenExists(L'h'))
        strToWriteTo.ReplaceToken(c4Minute, L'h');

    if (strToWriteTo.bTokenExists(L's'))
        strToWriteTo.ReplaceToken(c4Second, L's');

    if (strToWriteTo.bTokenExists(L'a'))
    {
        const tCIDLib::Tch* pszTmp = L"am";
        if (!c4Hour || (c4Hour > 12))
            pszTmp = L"pm";
        strToWriteTo.ReplaceToken(pszTmp, L'a');
    }
}


// -----------------------------------------------------------------------------
//  TTime: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TTime::_FormatTo(TTextStream& strmToWriteTo) const
{
    TString strTmp(kCIDLib::pszEmptyZStr, 128, 128);

    FormatToStr(strTmp, __strDefFormat);
    strmToWriteTo << strTmp;
}


tCIDLib::TVoid TTime::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    tCIDLib::TInt8 i8Tmp;
    strmToReadFrom >> i8Tmp;
    __ktmsThis = i8Tmp;
    strmToReadFrom >> __strDefFormat;
}

tCIDLib::TVoid TTime::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __ktmsThis.i8Time();
    strmToWriteTo << __strDefFormat;
}
