//
// NAME: CIDLib_Thread.Hpp
//
// DESCRIPTION:
//
//  This module provides the thread control class, TThread. Threads are named
//  for easy identification in log file messages and such.
//
//  Also implemented here is a simple thread priority janitor class,
//  TThreadPrioJan, which makes it safe to apply new priorities to threads in an
//  exception based system.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/16/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// GOTCHAS/CAVEATS:
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//   CLASS: TThread
//  PREFIX: thr
// -----------------------------------------------------------------------------
class CIDLIBEXP TThread : public TObject, public MFormattable
{
    public  :
        // ---------------------------------------------------------------------
        //  Contructors and Destructors
        // ---------------------------------------------------------------------
        TThread
        (
            const   TString&                strName
            ,       tCIDLib::TThreadFuncPtr pfnThread = 0
            , const tCIDLib::TCard4         c4StackSz = 0
            , const tCIDLib::TBoolean       bSelfPrio = kCIDLib::False
        );

        ~TThread();


        // ---------------------------------------------------------------------
        //  Static methods
        // ---------------------------------------------------------------------
        static TThread* pthrCaller();

        static tCIDLib::TVoid RegisterPerThreadObj
        (
            const   TString&                strName
            , const TClass&                 clsDataType
        );

        static tCIDLib::TVoid Sleep
        (
            const   tCIDLib::TCard4         c4MilliSecs
        );


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bCheckShutdownRequest() const;

        tCIDLib::TBoolean bIsRunning() const;

        tCIDLib::EExitCodes eTermCode() const;

        tCIDLib::EPrioLevels ePriority() const;

        tCIDLib::TThreadHandle hthrThis() const;

        const TKrnlThread& kthrThis() const;

        tCIDLib::TExitFuncPtr pfnSetOnExit
        (
            const   tCIDLib::TExitFuncPtr   pfnNew
        );

        TObject* pobjPerThread
        (
            const   TString&                strName
        );

        tCIDLib::TVoid Release();

        tCIDLib::TVoid RequestShutdown
        (
            const   tCIDLib::TCard4         c4MilliSecs = kCIDLib::c4MaxWait
        );

        tCIDLib::TVoid SetPriority
        (
            const   tCIDLib::EPrioLevels    ePrioLev
        );

        tCIDLib::TVoid Start
        (
                    tCIDLib::TVoid* const   pData = 0
            , const tCIDLib::EThreadFlags   eFlags = tCIDLib::EThreadFlag_None
        );

        const TString& strName() const;

        tCIDLib::TVoid Sync();

        tCIDLib::TThreadId tidThis() const;

        tCIDLib::TVoid WaitSync
        (
            const   tCIDLib::TCard4         c4MilliSecs = kCIDLib::c4MaxWait
        );

        tCIDLib::EExitCodes eWaitForDeath
        (
            const   tCIDLib::TCard4         c4MilliSecs = kCIDLib::c4MaxWait
        )   const;


    protected   :
        // ---------------------------------------------------------------------
        //  Declare friends
        //
        //  This class has many more that usual because of its unusual need
        //  for maximum performance and very primal nature.
        // ---------------------------------------------------------------------
        friend tCIDLib::TVoid _InitTermThread
        (
            const   tCIDLib::EInitTerm      eInitTerm
            , const tCIDLib::EGlobalStates  eGlobals
            , const TModule&                modInit
            , const tCIDLib::TCard4         c4MaxChars
            ,       tCIDLib::Tch* const     pszFailReason
        );

        friend tCIDLib::TVoid __AddThread
        (
                    TThread* const          pthrNew
        );

        friend tCIDLib::TVoid __AddThreadAt
        (
                    TThread* const          pthrNew
            , const tCIDLib::TCard4         c4Index
        );

        friend tCIDLib::EExitCodes THREADCALL __eThreadStart
        (
                    tCIDLib::TVoid*         pData
        );

        friend tCIDLib::EExitCodes __eThreadStart2
        (
                    TThread* const          pthrToStart
            ,       tCIDLib::TVoid* const   pUserData
        );


        // ---------------------------------------------------------------------
        //  Protected, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TVoid _Init();

        virtual tCIDLib::TVoid _Terminate();

        virtual tCIDLib::EExitCodes _eProcess();


        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;


        // ---------------------------------------------------------------------
        //  Protected, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _CheckCallerIsSelf
        (
            const   tCIDLib::TCard4         c4Line
            , const tCIDLib::Tch* const     pszAuxText = 0
        )   const;

        tCIDLib::TVoid _Exiting();

        tCIDLib::TVoid _ShutdownProcessing();


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors/operators
        // ---------------------------------------------------------------------
        TThread();

        TThread(const TThread&);

        tCIDLib::TVoid operator=(const TThread&);


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __bSelfPrio
        //      This flag (when set) prevents any other thread from setting
        //      this thread's priority.
        //
        //  __bShutdownRequest
        //      This is set to indicate that another is making a request to
        //      us to shutdown. The usual sync actions also take place, but
        //      this flag is set for use by the bCheckShutdownRequest()
        //      method.
        //
        //  __bSyncRequest
        //      This is set to indicate that another thread has called
        //      WaitSync(). It is also used to make sure that two threads
        //      don't try to sync at the same time. We store the tid of the
        //      syncing thread to be able to insure that only he releases
        //      the sync (in __tidSyncRequester.)
        //
        //  __c4ListInd
        //      This is our object's index into the internal list of threads.
        //      This is an optimization. It is set during construction and
        //      left alone.
        //
        //  __c4StackSz
        //      The size of the stack to give to the thread. Passed in by the
        //      creator.
        //
        //  __hevSync
        //      This event semaphore is used by the Sync()/WaitSync() methods
        //      to allow the thread to sync with another thread. It is also
        //      used during thread startup to sync.
        //
        //  __hevResponse
        //      This event semaphore is used by the Sync()/WaitSync() methods
        //      to allow the thread to sync with another thread.
        //
        //  __kthrThis
        //      The kernel thread object for this thread. This gives us the
        //      basic control and info that we need for threads. Its set to
        //      0 when this thread is not running.
        //
        //  __pfnFunc
        //      This is the address of the function that will be the entry
        //      point of the thread. It is called from the .Start() method.
        //      When it returns, the destructor is called.
        //
        //  __pfnOnExit
        //      This is the address of an optional on-exit function that
        //      may be provided by the client code. It will be called when
        //      the thread is shutting down. If the client does not set the
        //      function, it is 0.
        //
        //  __tidSyncReq
        //      When a sync request is made, we store the id of the thread
        //      that is syncing. This allows us to make sure that only that
        //      thread releases the sync.
        //
        //  __szName
        //      The name of the thread. This was passed in by the caller in
        //      the constructor.
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean       __bSelfPrio;
        tCIDLib::TBoolean       __bShutdownRequest;
        tCIDLib::TBoolean       __bSyncRequest;
        tCIDLib::TCard4         __c4ListInd;
        tCIDLib::TCard4         __c4StackSz;
        TKrnlEvent              __kevSync;
        TKrnlEvent              __kevResponse;
        TKrnlThread             __kthrThis;
        tCIDLib::TThreadFuncPtr __pfnFunc;
        tCIDLib::TExitFuncPtr   __pfnOnExit;
        tCIDLib::TThreadId      __tidSyncReq;
        TString                 __strName;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TThread,TObject)
};


// -----------------------------------------------------------------------------
//   CLASS: TPrioJanitor
//  PREFIX: jan
// -----------------------------------------------------------------------------
class CIDLIBEXP TPrioJanitor : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        TPrioJanitor
        (
            const   tCIDLib::EPrioLevels    eLevel
        );

        ~TPrioJanitor();


    protected   :
        // --------------------------------------------------------------------
        //  Hidden constructors and operators
        // --------------------------------------------------------------------
        TPrioJanitor();

        TPrioJanitor(const TPrioJanitor&);

        tCIDLib::TVoid operator=(const TPrioJanitor&);


    private :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __eLevel
        //      This is used to store the priority of the thread when the
        //      janitor is created.
        //
        //  __pthrTarget
        //      This is the thread we were created under. It is used to make
        //      sure we get back to the original thread and not some other
        //      one.
        // --------------------------------------------------------------------
        tCIDLib::EPrioLevels    __eLevel;
        TThread*                __pthrTarget;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TPrioJanitor,TObject)
};

#pragma pack(pop)


// -----------------------------------------------------------------------------
//  TThread: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean TThread::bCheckShutdownRequest() const
{
    return __bShutdownRequest;
}

inline tCIDLib::TThreadHandle TThread::hthrThis() const
{
    return __kthrThis.hthrThis();
}

inline const TKrnlThread& TThread::kthrThis() const
{
    return __kthrThis;
}

inline const TString& TThread::strName() const
{
    return __strName;
}

inline tCIDLib::TThreadId TThread::tidThis() const
{
    return __kthrThis.tidThis();
}
