//
// NAME: CIDLib_Thread.Cpp
//
// DESCRIPTION:
//
//  This module implements the TThread class, which provides process
//  control for all CIDLib based applications. It maintains a list of all
//  currently existing thread objects. It also provides the real thread
//  startup function which just calls the client's function. This way we
//  can install exception handlers for unhandled exceptions.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/16/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  The _InitTermProcess() method, that is called during facility init,
//      will create an after the fact thread object for the main thread.
//
//  2)  The local thread list and per-thread object manipulation functions
//      assume that the publically visible methods calling them have locked
//      the access mutex if that is required.
//
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"
#include    "CIDLib_PerThreadData_.Hpp"


// ----------------------------------------------------------------------------
//  Because of a side effect of how structured exceptions are used, we have
//  to include a system header here.
// ----------------------------------------------------------------------------
#include    <windows.h>


// ----------------------------------------------------------------------------
//  Define the per-thread attribute we use below
// ----------------------------------------------------------------------------
#define     THREADDATA      __declspec(thread)


// ----------------------------------------------------------------------------
//  Do our standard members macros
// ----------------------------------------------------------------------------
RTTIData(TThread,TObject)
RTTIData(TPerThreadData,TObject)
RTTIData(TPrioJanitor,TObject)


// ----------------------------------------------------------------------------
//  Constant data for local use
//
//  __c4MaxThreads
//      The maximum number of threads supported per-process.
//
//  pszThread1Name
//      This is the name given to thread1, the thread object for which is
//      faked in because the thread is already running.
// ----------------------------------------------------------------------------
static const tCIDLib::TCard4    __c4MaxThreads   = 1024;
static const tCIDLib::Tch*      __pszThread1Name = L"_Thread1_";


// ----------------------------------------------------------------------------
//  Local data types
//
//  TStartData
//      This structure is used to pass startup data to the __ThreadStart()
//      function. This function is used to start all threads. It does any
//      needed setup (which must be done in the context of the new thread)
//      and then does a regular call to the actual thread function.
//
//  TThreadList
//      An array of these is use to keep track of all of the running threads.
//      It is a small associative array so the .tidThread field is used to
//      find the right entry, and the other field is the thread object. This
//      prevents having to dereference each object to get to the id during the
//      search. If pthrThis is 0, then that entry is not used. Otherwise, the
//      tid is kCIDLib::tidInvalid if the thread is inactive, and some other
//      number if it is active.
// ----------------------------------------------------------------------------
struct  TStartData
{
    TThread*                pThread;
    tCIDLib::TVoid*         pData;
};

struct  TThreadList
{
    TThread*                pthrThis;
    tCIDLib::TThreadId      tidThread;
};


// ----------------------------------------------------------------------------
//  Variables for local use
//
//  __apptdList
//      This is an array of per-thread data description objects. This list
//      maintains the process wide list of objects that have been added
//      (because they must all be applied to all threads, potentially.)
//
//  __aThreads
//      A list of the thread objects that currently exist. We can use the
//      tidThread member to know whether the thread is currently running or
//      not (without have to dereference the thread object and access its
//      internal thread id member.)
//
//  __bInited
//      This is set when the _bInitProcess method has been called. So, if an
//      error occurs during startup, we won't flip out by trying to access
//      bad data.
//
//  __c4PTDCount
//      The count of items in the per-thread list, which holds entries for the
//      registered per-thread objects.
//
//  __c4ThreadCount
//      The count of thread objects currently defined. They are not all
//      necessarily running.
//
//  __pcrsListAccess
//      A critical section to coordinate access to the __aThreads array and
//      the per-thread object list, to insure that multiple threads don't step
//      on each other's toes.
//
//  __pthrPrimary
//      This is set to the first thread started by the user's Exe. When
//      this thread is seen exiting, then the program is ended.
// ----------------------------------------------------------------------------
static TPerThreadData*      __apptdList[kCIDLib::c4MaxPerThreadObjs];
static TThreadList          __aThreads[__c4MaxThreads];
static tCIDLib::TBoolean    __bInited;
static tCIDLib::TCard4      __c4PTDCount;
static tCIDLib::TCard4      __c4ThreadCount;
static TCriticalSection*    __pcrsListAccess;
static TThread*             __pthrPrimary;


// ----------------------------------------------------------------------------
//  Per-Thread data. This is a special one and is used to track the per
//  thread data objects for each thread.
// ----------------------------------------------------------------------------
static THREADDATA TObject* __apobjPerThreadList[kCIDLib::c4MaxPerThreadObjs];


// ----------------------------------------------------------------------------
//  Functions for local use only
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __AddThreadAt
//
// DESCRIPTION:
//
//  This method allows a thread to be added at a particular slot and with
//  an existing thread id (which is usually just __c4tidInactive.) The
//  index must be available. __AddThread() above selects an empty slot and
//  then calls this version.
// ---------------------------------------
//   INPUT: pthrNew is the new thread object to add.
//          tidThread is the thread id to initially set.
//          c4Index is the slot to use.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid __AddThreadAt(       TThread* const      pthrNew
                            , const tCIDLib::TCard4     c4Index)
{
    #if CID_DEBUG_ON
    if (__aThreads[c4Index].tidThread != kCIDLib::tidInvalid)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_CannotAddAt
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TCardinal(c4Index)
        );
    }
    #endif

    // Store the thread, set the tid, and bump up the count
    __aThreads[c4Index].pthrThis    = pthrNew;
    __aThreads[c4Index].tidThread   = pthrNew->tidThis();
    __c4ThreadCount++;

    // Store the thread id and index in the thread object
    __aThreads[c4Index].pthrThis->__c4ListInd = c4Index;
}


//
// FUNCTION/METHOD NAME: __AddThread
//
// DESCRIPTION:
//
//  This method is called when a new thread is to be added to the list. It
//  will abort if the thread count exceeds the maximum.
// ---------------------------------------
//   INPUT: pthrNew is the new thread object to add.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid __AddThread(TThread* const pthrNew)
{
    if (__c4ThreadCount == __c4MaxThreads)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_TooManyThreads
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_OutResource
            , TCardinal(__c4MaxThreads)
        );
    }

    //
    //  Scan for an empty entry in the list. If we don't find one, then
    //  there is a major screw up because the thread count is wrong. Also,
    //  count how many are taken while we look. If it exceeds
    //  __c4ThreadCount, then that is an error too.
    //
    tCIDLib::TCard4  c4Ind, c4Count = 0;
    for (c4Ind = 0; c4Ind < __c4MaxThreads; c4Ind++)
    {
        if (!__aThreads[c4Ind].pthrThis)
            break;
        c4Count++;
    }

    #if CID_DEBUG_ON
    if ((c4Count > __c4ThreadCount) || (c4Ind == __c4MaxThreads))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_ThreadCount
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TCardinal(__c4ThreadCount)
            , TCardinal(c4Count)
        );
    }
    #endif

    // And add it at the free slot
    __AddThreadAt(pthrNew, c4Ind);
}


//
// FUNCTION/METHOD NAME: __bRemoveId
//                       __bRemoveName
//
// DESCRIPTION:
//
//  This method will find the thread in the list with the passed id or
//  name and then remove the entry, setting it to 0 and bumping down
//  the thread count.
// ---------------------------------------
//   INPUT: tidTarget is the thread id to look for
//          strName is the thread name to look for
//
//  OUTPUT: None
//
//  RETURN: eTrue if the id was found, else eFalse.
//
static tCIDLib::TBoolean __bRemoveId(const tCIDLib::TThreadId tidTarget)
{
    #if CID_DEBUG_ON
    if (!__c4ThreadCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_ThreadCount_0
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
        );
    }

    //
    //  If the passed thread id is invalid, then freak out cause its
    //  already dead.
    //
    if (tidTarget == kCIDLib::tidInvalid)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_NotRunning
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TString(L"????")
        );
    }
    #endif

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4MaxThreads; c4Ind++)
    {
        if (__aThreads[c4Ind].tidThread == tidTarget)
        {
            __c4ThreadCount--;
            __aThreads[c4Ind].pthrThis = 0;
            __aThreads[c4Ind].tidThread = kCIDLib::tidInvalid;
            return kCIDLib::True;
        }
    }
    return kCIDLib::False;
}

static tCIDLib::TBoolean __bRemoveName(const TString& strName)
{
    if (!__c4ThreadCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_ThreadCount_0
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
        );
    }

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4MaxThreads; c4Ind++)
    {
        if (__aThreads[c4Ind].pthrThis)
        {
            if (__aThreads[c4Ind].pthrThis->strName() == strName)
            {
                __c4ThreadCount--;
                __aThreads[c4Ind].pthrThis = 0;
                __aThreads[c4Ind].tidThread = kCIDLib::tidInvalid;
                return kCIDLib::True;
            }
        }
    }
    return kCIDLib::False;
}


//
// FUNCTION/METHOD NAME: __DestroyPerThreadData
//
// DESCRIPTION:
//
//  This method will run the linked list of per thread data objects and
//  destroys the data for this thread. Although it only accesses this
//  thread's data, it also uses __c4PTDCount, so we have to protect ourself
//  anyway.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __DestroyPerThreadData()
{
    TCritSecLock crslAccess(__pcrsListAccess);
    for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4PTDCount; c4Index++)
    {
        delete __apobjPerThreadList[c4Index];
        __apobjPerThreadList[c4Index] = 0;
    }
}


//
// FUNCTION/METHOD NAME: __i4ExceptFilter
//
// DESCRIPTION:
//
//  This method is the exception filter for the exception handler placed
//  on every new thread. It logs the exception information via a call
//  to the kernel, to which it passes the exception information.
// ---------------------------------------
//   INPUT: pExceptInfo is the exception information that the logger
//              needs to log.
//
//  OUTPUT: None
//
//  RETURN: 0 to let the exception continue after we log it.
//
static tCIDLib::TInt4 __i4ExceptFilter(EXCEPTION_POINTERS* pExceptInfo)
{
    //
    //  Log the exception info. In order to avoid making the system
    //  specific struture public, this guy really takes just a void
    //  pointer and we are responsible for passing the right thing.
    //
    const TThread* const pthrTmp = TThread::pthrCaller();
    TKrnlThread::DumpException
    (
        pthrTmp->kthrThis()
        , pthrTmp->strName().pszData()
        , pExceptInfo
    );

    // Let the exception continue
    return EXCEPTION_CONTINUE_SEARCH;
}


//
// FUNCTION/METHOD NAME: __pthrFindId
//
// DESCRIPTION:
//
//  This method will find the thread in the list with the passed id.
// ---------------------------------------
//   INPUT: tidThread is the thread id to look for
//
//  OUTPUT: None
//
//  RETURN: A pointer to the thread or 0 if not found
//
static TThread* __pthrFindId(const tCIDLib::TThreadId tidThread)
{
    if (tidThread == kCIDLib::tidInvalid)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_NotRunning
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TString(L"????")
        );
    }

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4MaxThreads; c4Ind++)
    {
        if (__aThreads[c4Ind].tidThread == tidThread)
                return __aThreads[c4Ind].pthrThis;
    }
    return (TThread*)0;
}


//
// FUNCTION/METHOD NAME: __pptdFind
//
// DESCRIPTION:
//
//  This method will look up a per thread data element with the passed name.
//  If one is registered, then a pointer to the node is returned and the
//  index of that one in the list is returned.
// ---------------------------------------
//   INPUT: strName is the name of the element to find.
//
//  OUTPUT: c4Index is the index of the element. If not found, then the
//              value is not changed.
//
//  RETURN: None
//
static TPerThreadData*
__pptdFind(const TString& strName, tCIDLib::TCard4& c4Index)
{
    TCritSecLock crslAccess(__pcrsListAccess);

    // Calculate the passed string's hash
    tCIDLib::THashVal hshToFind = strName.hshCalcHash(kCIDLib::c4ClassModulus);

    // Now find the named element
    for (tCIDLib::TCard4 c4FindIndex = 0; c4FindIndex < __c4PTDCount; c4FindIndex++)
    {
        if (__apptdList[c4FindIndex]->hshName() == hshToFind)
        {
            if (strName == __apptdList[c4FindIndex]->strName())
            {
                c4Index = c4FindIndex;
                return __apptdList[c4FindIndex];
            }
        }
    }
    return 0;
}


//
// FUNCTION/METHOD NAME: __pthrFindName
//
// DESCRIPTION:
//
//  This method will find the named thread in the list.
// ---------------------------------------
//   INPUT: strName is the name of the thread to look for
//
//  OUTPUT: None
//
//  RETURN: A pointer to the thread or 0 if not found
//
static TThread* __pthrFindName(const TString& strName)
{
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4MaxThreads; c4Ind++)
    {
        if (__aThreads[c4Ind].pthrThis)
        {
            if (__aThreads[c4Ind].pthrThis->strName() == strName)
                return __aThreads[c4Ind].pthrThis;
        }
    }
    return 0;
}


//
// FUNCTION/METHOD NAME: __eThreadStart
//                       __eThreadStart2
//
// DESCRIPTION:
//
//  __eThreadStart() is where all threads actually get started up at. The
//  __eThreadStart() method is actually started first. It gets a structure
//  with the user data and the thread object pointer. It installs a
//  structured exception handler, then calls __eThreadStart2().
//
//  __eThreadStart2() installs a C++ exception handler and then calls
//  the user thread code, passing on the user data.
//
//  Both of these functions catch any unhandled exceptions and log them,
//  and handle cleanup of the thread. The finally clause of the structured
//  exception is used to do the thread shutdown processing.
// ---------------------------------------
//   INPUT: pData is a pointer to a TStartData data structure. It cannot be
//              const because it must fit the prototype of the RTL's view of
//              a thread function.
//
//  OUTPUT: None
//
//  RETURN: The thread's exit return value
//
tCIDLib::EExitCodes
__eThreadStart2(TThread* const pThread, tCIDLib::TVoid* const pUserData)
{
    //
    //  Call the thread function, or the _Process() method if they did
    //  not provide a function.
    //
    tCIDLib::EExitCodes eRet = tCIDLib::EExit_Normal;
    try
    {
        if (pThread->__pfnFunc)
            eRet = pThread->__pfnFunc(*pThread, pUserData);
         else
            eRet = pThread->_eProcess();
    }

    catch (const TError& errToCatch)
    {
        // Set the exit code to indicate a runtime error
        eRet = tCIDLib::EExit_RuntimeError;

        //
        //  Log the error. BE SURE to use Status so that it always goes to
        //  the log but does not throw another exception.
        //
        const tCIDLib::Tch* pszTmp;
        pszTmp = facCIDLib.pszLoadCIDMsg(kCIDMsgs::midThrd_UnhandledRTE);
        TString strTmp(L"Unhandled runtime error.", 128);
        if (pszTmp)
            strTmp = pszTmp;

        //
        //  If it would have caused a runtime error to be logged, then just
        //  add a reminder to check the file. Otherwise, log it here.
        //
        if ((errToCatch.eSeverity() == tCIDLib::ESev_ProcessFatal)
        ||  (errToCatch.eSeverity() == tCIDLib::ESev_SystemFatal))
        {
            pszTmp = facCIDLib.pszLoadCIDMsg(kCIDMsgs::midThrd_SeeErrorInf);
            if (pszTmp)
                strTmp.Append(pszTmp);
            else
                strTmp.Append(L" See ErrorInf.CIDLib");
        }
         else
        {
            //
            //  Make sure we don't get into some kind of freak out. We just
            //  want to log it, but we don't know what its severity it, so
            //  it could cause another throw. So we just catch it
            //  unconditionally.
            try
            {
                const TThread* const pthrTmp = TThread::pthrCaller();
                TKrnlThread::DumpRuntimeError
                (
                    pthrTmp->kthrThis()
                    , pthrTmp->strName().pszData()
                    , errToCatch.strModName().pszData()
                    , errToCatch.strErrText().pszData()
                    , errToCatch.strAuxText().pszData()
                    , errToCatch.errcId()
                    , errToCatch.errcKrnlId()
                    , errToCatch.errcHostId()
                    , errToCatch.strFileName().pszData()
                    , errToCatch.c4LineNum()
                );
                facCIDLib.LogErrObj(errToCatch);
            }

            catch(...)
            {
            }
        }

        try
        {
            facCIDLib.LogMsg
            (
                __FILE__
                , __LINE__
                , strTmp
                , tCIDLib::ESev_Status
            );
        }

        catch(...)
        {
            _ERRPOPUP_
            (
                facCIDLib.pszLoadCIDMsg(kCIDMsgs::midThrd_ExcpWhileLogging)
                , errToCatch.errcId()
            )
        }

        //
        //  Rethrow it so the program will terminate. But only if not
        //  the primary thread. If its the primary thread, then just
        //  return.
        //
        if (pThread != __pthrPrimary)
            throw;
    }
    return eRet;
}

tCIDLib::EExitCodes THREADCALL __eThreadStart(tCIDLib::TVoid* pData)
{
    TThread*            pThread;
    tCIDLib::TVoid*     pUserData;

    // Look at the passed pointer as a start data structure
    TStartData* const pStart  = (TStartData*)(pData);

    //
    //  Get the thread pointer and data pointer out of the passed data buffer
    //  then delete it. It had to be allocated because of the async nature
    //  of the thread startup.
    //
    pThread     = pStart->pThread;
    pUserData   = pStart->pData;

    // Delete the start data buffer
    delete pStart;

    //
    //  Set our copy of the thread's tid. This is so we can do quick
    //  searches without dereferencing every thread.
    //
    __aThreads[pThread->__c4ListInd].tidThread = pThread->tidThis();

    // Remember the list index in a temp
    tCIDLib::TCard4 c4ListIndex = pThread->__c4ListInd;

    //
    //  Install a system structured exception handler first. We will do
    //  C++ exception block also, nested inside this one. We also have to
    //  do two nested structured exception blocks here. One will handle
    //  exceptions and log them. The other will handle stack unwind and
    //  clean up the thread object's state.
    //
    tCIDLib::EExitCodes eRet = tCIDLib::EExit_Normal;
    __try
    {
        // Call the init method
        pThread->_Init();

        //
        //  Now call the other thread start, which we have to have 
        //  because of VC++'s stupid limitation of only one kind of
        //  exception handling per function.
        //
        __try
        {
            eRet = __eThreadStart2(pThread, pUserData);
        }

        // Log any exception that occur, but let them pass on
        __except(__i4ExceptFilter(GetExceptionInformation()))
        {
            // Set the exit code to indicate system exception
            eRet = tCIDLib::EExit_SystemException;
        }
    }

    // During thread unwind, we need to do any needed shutdown processing
    __finally
    {
        pThread->_ShutdownProcessing();
    }

    //
    //  If this was the primary thread, then thats the main thread that
    //  was first started by the user and we want to die now. Pass along
    //  the primary thread's return value, which will become the process
    //  exit code if another thread does not call ExitProcess() somewhere
    //  during shutdown and override it.
    //
    if (pThread == __pthrPrimary)
    {
        __pthrPrimary = 0;
        _ReleaseMainThread(eRet);
    }

    // Return the thread's return value
    return eRet;
}


// ----------------------------------------------------------------------------
//  Functions for intrafacility use
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _InitTermThread
//
// DESCRIPTION:
//
//	This function is called from the CIDLib.Dll's initialization. It handles
//  any module init/term we need.
// ---------------------------------------
//   INPUT: eInitTerm indicates what initialization phase we are in.
//          eGlobals indicates whether this is before constructors or
//              after destructors for globals.
//          modThis is a module object for this module.
//          c4MaxChars is the max chars that the failure reason buffer
//              can hold.
//
//  OUTPUT: pszFailReason is filled with the reason for a failure.
//
//  RETURN: None
//
tCIDLib::TVoid _InitTermThread( const   tCIDLib::EInitTerm      eInitTerm
                                , const tCIDLib::EGlobalStates  eGlobals
                                , const TModule&                modInit
                                , const tCIDLib::TCard4         c4MaxChars
                                ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midGen_Unknown);
    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            //
            //  Create the thread list access semaphore. We create it unshared and
            //  manual mode.
            //
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midThrd_CritSecCreate);
            __pcrsListAccess = new TCriticalSection;

            // Initialize the thread list
            for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4MaxThreads; c4Ind++)
            {
                __aThreads[c4Ind].pthrThis = (TThread*)0;
                __aThreads[c4Ind].tidThread = kCIDLib::tidInvalid;
            }

            // Set the local inited flag
            __bInited = kCIDLib::True;
        }
         else if ((eInitTerm == tCIDLib::EInitTerm_Terminate)
              &&  (eGlobals == tCIDLib::EGlobalState_After))
        {
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midThrd_CritSecDelete);
            if (__pcrsListAccess)
            {
                delete __pcrsListAccess;
                __pcrsListAccess = 0;
            }
        }
    }

    catch(...)
    {
        TRawStr::CopyStr(pszFailReason, pszPhase, c4MaxChars);
        throw;
    }
}


//
// FUNCTION/METHOD NAME: _pthrCurrent
//
// DESCRIPTION:
//
//  This function will get the thread id of the current thread, lock the
//  thread list, look up the thread, and return a pointer to the thread
//  object.
//
//  NOTE:   If the thread is not found, then this is a major error, so we log
//          an error and abort. Note also that we cannot log from this call
//          because we will get into a big circle jerk with the logging
//          mechanism and overflow the stack.
//
//  NOTE:   If we have not inited, then someone is calling into here during
//          init because of some failure. So we create a bogus thread and
//          return it. It does not matter because we are going down anyway.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: A pointer to the thread object of the calling thread.
//
TThread* _pthrCurrent()
{
    if (!__bInited)
    {
        _MSGPOPUP_
        (
            L"pthrCurrent: Called before initialization is completed"
        )

        //
        //  Return a bogus thread object, since we know that this is
        //  happening due to a start up error and he just wants a thread
        //  name. He is  going to abort anyway so this memory will not
        //  get lost.
        //
        return new TThread(__pszThread1Name, 0);
    }

    // Dummy scope so we don't stay locked if popup occurs!
    TThread* pThread = 0;
    {
        // Get access to the thread list
        TCritSecLock crslAccess(__pcrsListAccess);

        // Query the thread id
        tCIDLib::TThreadId  tidTmp = TKrnlThread::tidCaller();

        // Look for this thread id
        pThread = __pthrFindId(tidTmp);
    }

    //
    //  If we did get a nul node back, then a thread with that id does not
    //  exist. This is a major error, so we need to abort.
    //
    if (!pThread)
    {
        _ERRPOPUP_
        (
            , L"_pthrCurrent: The calling thread was not found in the thread list"
            , kCIDErrs::errcPrc_ThreadNotFound
        )
        facCIDLib.ExitProcess(tCIDLib::EExit_FatalError);
    }

    // We did find the node, so return the thread object in the node
    return pThread;
}


// -----------------------------------------------------------------------------
//  Global functions
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: CIDLib_MakePrimary
//
// DESCRIPTION:
//
//  This is called by the magic startup code that is generated into the
//  user's Exe. It stores the address of the primary thread, which is the
//  first user thread started. When that thread ends, the program is
//  terminated.
// ---------------------------------------
//   INPUT: pthrPrimary is the primary thread that the user's program is
//              running on.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid CIDLib_MakePrimary(TThread* const pthrPrimary)
{
    if (__pthrPrimary)
    {
        _MSGPOPUP_
        (
            L"CIDLib_MakePrimary: This function cannot be called multiple times!"
        )
    }
    __pthrPrimary = pthrPrimary;
}



// -----------------------------------------------------------------------------
//   CLASS: TPrioJanitor
//  PREFIX: jan
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TThread: Constructors and Destructors
// -----------------------------------------------------------------------------
TPrioJanitor::TPrioJanitor(const tCIDLib::EPrioLevels eLevel) :

    __eLevel(tCIDLib::EPrioLevels(tCIDLib::EPrioLevels_Max + 1))
    , __pthrTarget(0)
{
    //
    //  Get the priority of this thread now. The class and level were set to
    //  invalid values above so that, if this fails, they will cause an error
    //  when restored to the thread.
    //
    __pthrTarget = _pthrCurrent();
    __eLevel = __pthrTarget->ePriority();

    // Set the new priority
    __pthrTarget->SetPriority(__eLevel);
}

TPrioJanitor::~TPrioJanitor()
{
    __pthrTarget->SetPriority(__eLevel);
}




// ----------------------------------------------------------------------------
//   CLASS: TThread
//  PREFIX: thr
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TThread: Constructors and Destructors
// ----------------------------------------------------------------------------

TThread::TThread(   const   TString&                strName
                    ,       tCIDLib::TThreadFuncPtr pfnFunc
                    , const tCIDLib::TCard4         c4StackSz
                    , const tCIDLib::TBoolean       bSelfPrio) :

    __bSelfPrio(bSelfPrio)
    , __bShutdownRequest(kCIDLib::False)
    , __bSyncRequest(kCIDLib::False)
    , __c4StackSz(c4StackSz)
    , __kthrThis()
    , __pfnFunc(pfnFunc)
    , __pfnOnExit(0)
    , __strName(strName)
{
    // Make sure that the stack size is not less than 8K
    if (__c4StackSz < 8192)
        __c4StackSz = 8192;

    // Create this thread's sync semaphore
    try
    {
        __kevSync.Create(tCIDLib::EEventState_Triggered);
        __kevResponse.Create(tCIDLib::EEventState_Triggered);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Create
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TString(kCIDLib::pszEmptyZStr)
        );
    }

    // Bump up the count of threads
    _pmtrgCore->OffsetCard(tCIDLib_::eCoreMetric_ThreadCnt, 1);

    // Get access to the thread list
    TCritSecLock crslAccess(__pcrsListAccess);

    //
    //  Check the thread list and see if there is already a thread with
    //  this name.
    //
    TThread* pThread = __pthrFindName(__strName);

    // If we get a thread back, then a thread with that name already exists
    if (pThread)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_AlreadyExists
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , strName
        );
    }

    // Add a new node to the list.
    __AddThread(this);
}

TThread::~TThread()
{
    // Bump down the thread count
    _pmtrgCore->OffsetCard(tCIDLib_::eCoreMetric_ThreadCnt, -1);

    if (__kthrThis.bIsRunning())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_NotWhileRunning
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , facCIDLib.strMsg(kCIDMsgs::midGen_ObjDtor)
            , __strName
        );
    }

    // Dummy scope block to make sure list gets unlocked soon as possible
    {
        // Get access to the thread list
        TCritSecLock crslAccess(__pcrsListAccess);

        if (!__bRemoveName(__strName))
        {
            facCIDLib.LogErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcPrc_ThreadNotFound
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
                , __strName
            );
        }
    }

    try
    {
        __kevSync.Close();
        __kevResponse.Close();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Close
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TString(kCIDLib::pszEmptyZStr)
        );
    }

    // Destroy the per-thread objects if any
    __DestroyPerThreadData();
}


// ----------------------------------------------------------------------------
//  TThread: Public, static methods
// ----------------------------------------------------------------------------

TThread* TThread::pthrCaller()
{
    return _pthrCurrent();
}


tCIDLib::TVoid
TThread::RegisterPerThreadObj(  const   TString&    strName
                                , const TClass&     clsDataType)
{
    TCritSecLock crslAccess(__pcrsListAccess);

    // Check to see if we have exceeded our limit
    if (__c4PTDCount >= c4MaxPerThreadObjs)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_TooManyPTOs
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_OutResource
            , TCardinal(c4MaxPerThreadObjs)
        );
    }

    // Make sure the name is unique
    tCIDLib::TCard4 c4Index;
    TPerThreadData* const pptdFind = __pptdFind(strName, c4Index);

    if (pptdFind)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_PTONotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strName
        );
    }

    // We are ok, so add the next one
    __apptdList[__c4PTDCount] = new TPerThreadData(strName, clsDataType);
    __c4PTDCount++;
}


tCIDLib::TVoid TThread::Sleep(const tCIDLib::TCard4 c4MilliSecs)
{
    TKrnlThread::Sleep(c4MilliSecs);
}



// ----------------------------------------------------------------------------
//  TThread: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TThread::bIsRunning() const
{
    try
    {
        return __kthrThis.bIsRunning();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_Running
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_CantDo
            , __strName
        );
    }
}


tCIDLib::EPrioLevels TThread::ePriority() const
{
    tCIDLib::EPrioLevels eRet;
    try
    {
        eRet = __kthrThis.ePriority();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_PrioQuery
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_CantDo
        );
    }
    return eRet;
}


tCIDLib::TExitFuncPtr
TThread::pfnSetOnExit(const tCIDLib::TExitFuncPtr pfnNew)
{
    tCIDLib::TExitFuncPtr pfnTmp = __pfnOnExit;
    __pfnOnExit = pfnNew;
    return pfnTmp;
}


TObject* TThread::pobjPerThread(const TString& strName)
{
    tCIDLib::TCard4 c4Index;

    TPerThreadData* const pptdFound = __pptdFind(strName, c4Index);
    if (!pptdFound)
        return 0;

    if (!__apobjPerThreadList[c4Index])
        __apobjPerThreadList[c4Index] = pptdFound->clsOfData().pobjMakeNew();

    return __apobjPerThreadList[c4Index];
}


tCIDLib::TVoid TThread::Release()
{
    TCritSecLock crslAccess(__pcrsListAccess);

    // Make sure the caller is the one that did the sync
    #if CID_DEBUG_ON
    if (TKrnlThread::tidCaller() != __tidSyncReq)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_NoSyncRequest
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , __strName
        );
    }
    #endif

    // If there is not an outstanding sync request, then this call is bogus
    if (!__bSyncRequest)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_NoSyncRequest
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , __strName
        );
    }

    // Clear the sync request flag
    __bSyncRequest = kCIDLib::False;

    // Post the sync semaphore to let the thread go
    try
    {
        __kevSync.Trigger();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Trigger
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TString(__kevSync.pszName())
        );
    }
}


tCIDLib::TVoid
TThread::RequestShutdown(const tCIDLib::TCard4 c4TimeOut)
{
    //
    //  First we need to sync up with this thread object. If not running,
    //  then an error is logged.
    //
    WaitSync(c4TimeOut);

    try
    {
        //
        //  Reset the response semaphore again. We will wait on this guy
        //  again, and the thread function's return will clear it.
        //
        try
        {
            __kevResponse.Reset();
        }

        catch(const TKrnlError& kerrToCatch)
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcEv_Reset
                , kerrToCatch
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
                , TString(__kevResponse.pszName())
            );
        }

        __bShutdownRequest = kCIDLib::True;

        // Release the other thread now
        Release();

        // Wait for the thread to die
        try
        {
            __kevResponse.WaitFor(c4TimeOut);
        }

        catch(const TKrnlError& kerrToCatch)
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcEv_Wait
                , kerrToCatch
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_CantDo
                , TString(__kevResponse.pszName())
            );
        }
    }

    catch(...)
    {
        __bShutdownRequest = kCIDLib::False;
        throw;
    }

    __bShutdownRequest = kCIDLib::False;
}


tCIDLib::TVoid
TThread::SetPriority(const tCIDLib::EPrioLevels ePrioLev)
{
    #if CID_DEBUG_ON
    if ((ePrioLev < tCIDLib::EPrioLevels_Min)
    ||  (ePrioLev > tCIDLib::EPrioLevels_Max))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_BadPrioLevel
            , facCIDLib.pszLoadCIDMsg(kCIDMsgs::midThrd_ClippedPrio)
            , tCIDLib::ESev_Warning
            , tCIDLib::EClass_BadParms
            , TCardinal(ePrioLev)
            , TCardinal(tCIDLib::EPrioLevels_Max)
        );
    }
    #endif

    //
    //  If this thread is in SelfPrio mode, then only it can set its own
    //  priority.
    //
    if (__bSelfPrio)
    {
        // Abort if the caller is not this thread
        _CheckCallerIsSelf
        (
            __LINE__
            , facCIDLib.pszLoadCIDMsg(kCIDMsgs::midThrd_SelfPrioMode)
        );
    }

    try
    {
        __kthrThis.SetPriority(ePrioLev);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_PrioSet
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}


tCIDLib::TVoid TThread::Start(          tCIDLib::TVoid* const   pData
                                , const tCIDLib::EThreadFlags   eFlags)
{
    if (bIsRunning())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_AlreadyRunning
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_Already
            , __strName
        );
    }

    // Reset the sync semaphore so it won't go until we tell it to
    try
    {
        __kevSync.Reset();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Reset
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TString(__kevSync.pszName())
        );
    }

    // Start the thread and pass the startup data
    TStartData*  pStartData = new TStartData;
    pStartData->pThread     = this;
    pStartData->pData       = pData;

    //
    //  We act as though a sync request was already done.
    //
    __bSyncRequest = kCIDLib::True;
    __tidSyncReq = TKrnlThread::tidCaller();

    try
    {
        __kevResponse.Reset();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Reset
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TString(__kevResponse.pszName())
        );
    }

    try
    {
        __kthrThis.BeginThread
        (
            __eThreadStart
            , __c4StackSz
            , pStartData
            , eFlags
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_ThreadStartFailed
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_CantDo
            , __strName
        );
    }

    //
    //  So now we need to wait for the new thread to call Sync() for the
    //  first time. At that point, we can return to the caller who knows
    //  that the new thread is up and going.
    //
    try
    {
        __kevResponse.WaitFor(30 Seconds);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Wait
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TString(__kevResponse.pszName())
        );
    }

    //
    //  Ok, he is up and waiting in his Sync() call, so we can let him go
    //  now.
    //
    Release();
}


tCIDLib::TVoid TThread::Sync()
{
    _CheckCallerIsSelf(__LINE__);

    // Dummy block to force semaphore unlock as quick as possible
    {
        TCritSecLock crslAccess(__pcrsListAccess);

        // If there is no sync request, then just return
        if (!__bSyncRequest)
            return;

        // Post the response semaphore to indicate that we are now waiting
        try
        {
            __kevResponse.Trigger();
        }

        catch(const TKrnlError& kerrToCatch)
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcEv_Trigger
                , kerrToCatch
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_CantDo
                , TString(__kevResponse.pszName())
            );
        }
    }

    // Block on the sync semaphore
    try
    {
        __kevSync.WaitFor();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Wait
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_CantDo
            , TString(__kevSync.pszName())
        );
    }

    // We are awake again so the syncing thread has released us
}


tCIDLib::EExitCodes TThread::eTermCode() const
{
    // Get the exit code and store it
    tCIDLib::EExitCodes eExitCode;
    try
    {
        eExitCode = __kthrThis.eQueryExitCode();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_QueryExitCode
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , facCIDLib.strMsg(kCIDMsgs::midThrd_Thread)
            , __strName
        );
    }
    return eExitCode;
}


tCIDLib::EExitCodes TThread::eWaitForDeath(const tCIDLib::TCard4 c4Timeout) const
{
    // Get the the id of the calling thread
    tCIDLib::TThreadId tidCaller = TKrnlThread::tidCaller();

    #if CID_DEBUG_ON
    if (__kthrThis.tidThis() == tidCaller)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_WaitForSelf
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , __strName
        );
    }
    #endif

    tCIDLib::EExitCodes eRet;
    try
    {
        eRet = __kthrThis.eWaitForDeath(c4Timeout);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_ThreadWait
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_CantDo
            , __strName
        );
    }
    return eRet;
}


tCIDLib::TVoid TThread::WaitSync(const tCIDLib::TCard4 c4Timeout)
{
    // Get the the id of the calling thread
    tCIDLib::TThreadId tidCaller = TKrnlThread::tidCaller();

    #if CID_DEBUG_ON
    if (__kthrThis.tidThis() == tidCaller)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcPrc_SyncWithSelf
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , __strName
        );
    }
    #endif

    {
        TCritSecLock crslAccess(__pcrsListAccess);
        try
        {
            // If there is already a request, then we can't do it
            if (__bSyncRequest)
            {
                TThread* pthrSyncer = __pthrFindId(__tidSyncReq);

                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcPrc_AlreadySynced
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_Already
                    , __strName
                    , pthrSyncer->__strName
                );
            }

            // If this thread is not running, then log an error
            if (!bIsRunning())
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcPrc_NotRunning
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_AppError
                    , __strName
                );
            }

            // Set the flag so we won't be interrupted
            __bSyncRequest = kCIDLib::True;

            // Remember the syncer's id for later
            __tidSyncReq = tidCaller;

            //
            //  Reset the sync semaphore to cause the thread to block the
            //  next timeit calls Sync().
            //
            try
            {
                __kevSync.Reset();
            }

            catch(const TKrnlError& kerrToCatch)
            {
                facCIDLib.LogKrnlErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcEv_Reset
                    , kerrToCatch
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_Internal
                    , TString(__kevSync.pszName())
                );
            }
        }

        catch(...)
        {
            // Clear the flag to cancel this operation
            __bSyncRequest = kCIDLib::False;

            // Be anal and clear the sync requester tid
            __tidSyncReq = kCIDLib::tidInvalid;

            throw;
        }
    }

    //
    //  Block on the response semaphore, which should already be reset. When
    //  we awake, the thread is blocked on the sync semaphore. So we can
    //  return.
    //
    try
    {
        __kevResponse.WaitFor(c4Timeout);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Wait
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TString(__kevResponse.pszName())
        );
    }
}


// -----------------------------------------------------------------------------
//  TThread: Protected, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _CheckCallerIsSelf
//
// DESCRIPTION:
//
//  This method will check to make sure that the calling thread is this
//  thread. If not, it will issue a fatal error message and abort.
// ---------------------------------------
//   INPUT: c4LineNum is the line from where the check was done.
//          pszAuxText is an optional text string for the aux text of the
//              logged error. It defaults to 0.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TThread::_CheckCallerIsSelf(const   tCIDLib::TCard4     c4LineNum
                            , const tCIDLib::Tch* const pszAuxText) const
{
    if (__kthrThis.tidThis() != TKrnlThread::tidCaller())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , c4LineNum
            , kCIDErrs::errcPrc_NotThisThread
            , pszAuxText ? pszAuxText : kCIDLib::pszEmptyZStr
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Authority
            , this->__strName
        );
    }
}

//
// FUNCTION/METHOD NAME: _Exiting
//
// DESCRIPTION:
//
//  This method is provided so that the standard thread exception handler
//  can clean up the thread flags when it is killed.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TThread::_Exiting()
{
    //
    //  Call the on-exit function to let the thread object clean up. This
    //  is for user code since it does not require a separate derivation to
    //  do.
    //
    if (__pfnOnExit)
        __pfnOnExit(*this);

    //
    //  Call the virtual terminate method to let derived thread classes do
    //  internal cleanup.
    //
    _Terminate();

    // Mark this thread as not running
    __kthrThis._SetNotRunning();

    // And set it in our entry of the thread list
    __aThreads[__c4ListInd].tidThread = kCIDLib::tidInvalid;
}


//
// FUNCTION/METHOD NAME: _ShutdownProcessing
//
// DESCRIPTION:
//
//  This method is called from the thread startup code when the thread is
//  ending, either by exception or normally. It handles doing all of the
//  work that needs to be done when the thread ends. Its done here because
//  the exception scheme requires that its called from multiple places.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TThread::_ShutdownProcessing()
{
    // Call the exiting virtual method to let the thread clean up
    _Exiting();

    // Make sure possible waiting sync thread gets let go
    try
    {
        __kevResponse.Trigger();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcEv_Trigger
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TString(__kevResponse.pszName())
        );
    }

    // And clear the shutdown request flag for next time
    __bShutdownRequest = kCIDLib::False;
}


// -----------------------------------------------------------------------------
//  TThread: Protected, virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TThread::_Init()
{
    // Right now we do nothing here
}


tCIDLib::EExitCodes TThread::_eProcess()
{
    facCIDLib.LogErr
    (
        __FILE__
        , __LINE__
        , kCIDErrs::errcPrc_NoProcessOverride
        , tCIDLib::ESev_ProcessFatal
        , tCIDLib::EClass_AppError
    );

    // Make the compiler happy
    return tCIDLib::EExit_Normal;
}


tCIDLib::TVoid TThread::_Terminate()
{
    // Right now we do nothing here
}


// -----------------------------------------------------------------------------
//  TThread: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TThread::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo   << facCIDLib.strMsg(kCIDMsgs::midThrd_ThreadEq) << __strName
                    << facCIDLib.strMsg(kCIDMsgs::midThrd_RunningEq) << bIsRunning()
                    << facCIDLib.strMsg(kCIDMsgs::midThrd_IdEq) << __kthrThis.tidThis();
}
