//
// NAME: CIDLib_ThisFacility.Cpp
//
// DESCRIPTION: 
//
//  This module implements the facility class for this facility. It is a
//  derivative (as all facility objects are) of TFacility.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/26/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"

// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TFacCIDLib,TFacility)



// ----------------------------------------------------------------------------
//  Intra-facility methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _InitTermThisFac
//
// DESCRIPTION:
//
//  This function is called during process init/term to initialize and
//  clean up this module.
// ---------------------------------------
//   INPUT: eInitTerm indicates what initialization phase we are in.
//          eGlobals indicates whether this is before constructors or
//              after destructors for globals.
//          modInit is a temp module object for this module.
//          c4MaxChars is the max chars that the failure reason buffer
//              can hold.
//
//  OUTPUT: pszFailReason is filled with the reason for a failure.
//
//  RETURN: None
//
tCIDLib::TVoid
_InitTermThisFac(   const   tCIDLib::EInitTerm      eInitTerm
                    , const tCIDLib::EGlobalStates  eGlobals
                    , const TModule&                modInit
                    , const tCIDLib::TCard4         c4MaxChars
                    ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midGen_Unknown);
    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            // Create default audio and encrytion objects
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midFac_DefAudioCreate);
            TFacCIDLib::__paudioDef = new TAudio;
        }
    }

    catch(...)
    {
        TRawStr::CopyStr(pszFailReason, pszPhase, c4MaxChars);
        throw;
    }
}


// ----------------------------------------------------------------------------
//   CLASS: TFacCIDLib
//  PREFIX: fac
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TFacCIDLib: Static data members
// ----------------------------------------------------------------------------
TKrnlSafeCounter CIDLIBEXP  TFacCIDLib::__scntObjectCount   = 0;
TAudio* CIDLIBEXP           TFacCIDLib::__paudioDef         = 0;


// ----------------------------------------------------------------------------
//  TFacCIDLib: Constructors and Destructors
// ----------------------------------------------------------------------------

TFacCIDLib::TFacCIDLib() :

    TFacility
    (
        L"CIDLib"
        , tCIDLib::EModType_Dll
        , kCIDLib::c4MajVersion
        , kCIDLib::c4MinVersion
    )
{
}

TFacCIDLib::~TFacCIDLib()
{
}


// ----------------------------------------------------------------------------
//  TFacCIDLib: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TFacCIDLib::Beep(   const   tCIDLib::TCard4 c4Frequency
                    , const tCIDLib::TCard4 c4Duration)
{
    __paudioDef->Beep(c4Frequency, c4Duration);
}


tCIDLib::TVoid TFacCIDLib::ExitProcess(const tCIDLib::EExitCodes eExit)
{
    //
    //  The application may have already set the process state to
    //  terminating, but go ahead and set it for those that don't.
    //
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    // Let the main thread code and give him the exit code
    _ReleaseMainThread(eExit);

    // And just sleep until we die
    TKrnlThread::Sleep(kCIDLib::c4MaxWait);
}


TAudio* TFacCIDLib::paudioAdoptDefAudioObj(TAudio* const paudioNew)
{
    TAudio* const paudioOrg = __paudioDef;
    __paudioDef = paudioNew;
    return paudioOrg;
}


tCIDLib::TVoid
TFacCIDLib::PopUpMsg(   const   tCIDLib::Tsch* const    pszModule
                        , const tCIDLib::TCard4         c4LineNum
                        , const tCIDLib::Tch* const     pszTitle1
                        , const tCIDLib::Tch* const     pszTitle2
                        , const tCIDLib::Tch* const     pszMsg
                        , const tCIDLib::Tch* const     pszMsg2)
{
    TKrnlPopup::Show
    (
        pszModule
        , c4LineNum
        , pszTitle1
        , pszTitle2
        , 0
        , 0
        , 0
        , pszMsg
        , pszMsg2
    );
}


tCIDLib::TVoid
TFacCIDLib::PopUpErr(   const   tCIDLib::Tsch* const    pszModule
                        , const tCIDLib::TCard4         c4LineNum
                        , const tCIDLib::Tch* const     pszTitle1
                        , const tCIDLib::Tch* const     pszTitle2
                        , const tCIDLib::TErrCode       errcId
                        , const tCIDLib::TErrCode       errcKrnlId
                        , const tCIDLib::TOSErrCode     errcHostId
                        , const tCIDLib::Tch* const     pszMsg
                        , const tCIDLib::Tch* const     pszMsg2)
{
    TKrnlPopup::Show
    (
        pszModule
        , c4LineNum
        , pszTitle1
        , pszTitle2
        , errcId
        , errcKrnlId
        , errcHostId
        , pszMsg
        , pszMsg2
    );
}
