//
// NAME: CIDLib_TextStream.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_TextStream.Cpp module. This module
//  implements the TTextStream class. TTextStream is a stream that supports
//  text mode input and output of data.
//
//  The abstract base class TTextStreamImpl is defined here. It is the basis
//  for any stream implementation classes that provide data sink/source
//  services for text streams. TTextStream is implemented in terms of the
//  implementation class, so it is independent of where the data comes from
//  or goes to.
//
//  And the TStreamJanitor class is implemented here too. It is a stream
//  format janitor class, which will save and restore the formatting settings
//  of a text stream.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/03/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


// -----------------------------------------------------------------------------
//  Forward references
// -----------------------------------------------------------------------------
class   TTextStream;


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//   CLASS: TTextStreamImpl
//  PREFIX: strmi
// -----------------------------------------------------------------------------
class CIDLIBEXP TTextStreamImpl : public TObject
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        ~TTextStreamImpl() {}


        // ---------------------------------------------------------------------
        //  Public, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TBoolean bEndOfStream() const = 0;

        virtual tCIDLib::TCard4 c4GetLine
        (
                    tCIDLib::Tch* const     pszBufToFill
            , const tCIDLib::TCard4         c4BufChars
            , const tCIDLib::TBoolean       bStripWhitespace = kCIDLib::True
        ) = 0;

        virtual tCIDLib::Tch chGet() = 0;

        virtual tCIDLib::TFilePos fposCurPos() const = 0;

        virtual tCIDLib::TFilePos fposLogicalEnd() const = 0;

        virtual tCIDLib::TFilePos fposPhysicalEnd() const = 0;

        virtual tCIDLib::TFilePos fposSeekToEnd() = 0;

        virtual tCIDLib::TFilePos fposSkipForwardBy
        (
            const   tCIDLib::TCard4         c4SkipBy
        ) = 0;

        virtual tCIDLib::TVoid Reset() = 0;

        virtual tCIDLib::TVoid PutCh
        (
            const   tCIDLib::Tch            chToWrite
        ) = 0;

        virtual tCIDLib::TVoid PutCh
        (
            const   tCIDLib::Tsch           schToWrite
        ) = 0;

        virtual tCIDLib::TVoid PutLine
        (
            const   tCIDLib::Tch* const     pszBufToWrite
        ) = 0;

        virtual tCIDLib::TVoid PutLine
        (
            const   tCIDLib::Tsch* const    pszBufToWrite
        ) = 0;

        virtual tCIDLib::Tsch schGet() = 0;

        virtual tCIDLib::TVoid TruncateAtZero() = 0;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::ETextFormats eTextFormat() const;

        tCIDLib::ETextFormats eTextFormat
        (
            const   tCIDLib::ETextFormats   eFormat
        );

    protected :
        // ---------------------------------------------------------------------
        //  Hidden constructors
        // ---------------------------------------------------------------------
        TTextStreamImpl();

        TTextStreamImpl
        (
            const   tCIDLib::ETextFormats   eTextFormat
        );


        // ---------------------------------------------------------------------
        //  Protected, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TVoid _FormatChange
        (
            const   tCIDLib::ETextFormats   eNewFormat
        );


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TTextStreamImpl(const TTextStreamImpl&);

        tCIDLib::TVoid operator=(const TTextStreamImpl&);


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __eTextFmtIn
        //      These indicate how text is translated to from the provided
        //      format when we read/write the data.
        // ---------------------------------------------------------------------
        tCIDLib::ETextFormats   __eTextFmt;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TTextStreamImpl,TObject)
};



// -----------------------------------------------------------------------------
//   CLASS: TTextStream
//  PREFIX: strm
// -----------------------------------------------------------------------------
class CIDLIBEXP TTextStream : public TObject
{
    public  :
        // ---------------------------------------------------------------------
        //  Public, static data
        //
        //  strmfDefaults
        //      This object contains the default stream format. So you can
        //      reset the defaults by dumping it to the text stream.
        // ---------------------------------------------------------------------
        static const TStreamFmt strmfDefault;


        // ---------------------------------------------------------------------
        //  Public types
        //
        //  ESpecial
        //      This is used to put special string combinations into the
        //      stream, very conveniently. They can all be done via strings,
        //      but those strings are subject to formatting controls. For
        //      the most part, you don't want these to be justified and filled
        //      and so on.
        //
        //  ESpaces
        //      Used to insert spaces into the stream without any formatting.
        //      Just case the number of spaces desired, e.g. ESpaces(32) to
        //      insert 32 spaces.
        // ---------------------------------------------------------------------
        enum ESpecial
        {
            ENewLine
            , EDblNewLine
            , EReturn
            , ELineFeed

            , ESpecial_Min      = ENewLine
            , ESpecial_Max      = ELineFeed
        };

        enum ESpaces
        {
            EDummy
        };

        class Fill
        {
            public :
                Fill(const tCIDLib::Tch chNewFill)
                    : chFill(chNewFill)
                {
                }
                tCIDLib::Tch chFill;
        };

        class Justify
        {
            public :
                Justify(const tCIDLib::EHJustify  eNewJustification)
                    : eJustification(eNewJustification)
                {
                }
                tCIDLib::EHJustify eJustification;
        };

        class Precision
        {
            public :
                Precision(const tCIDLib::TCard4 c4NewPrecision)
                    : c4Precision(c4NewPrecision)
                {
                }
                tCIDLib::TCard4 c4Precision;
        };

        class Width
        {
            public :
                Width(const tCIDLib::TCard4 c4NewWidth)
                    : c4Width(c4NewWidth)
                {
                }
                tCIDLib::TCard4 c4Width;
        };


        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TTextStream
        (
                    TTextStreamImpl* const  pstrmiToAdopt
            , const TStreamFmt&             strmfToUse = NUL_TStreamFmt
        );

        ~TTextStream();



        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TTextStream& operator<<
        (
            const   tCIDLib::TBoolean       bToWrite
        );

        TTextStream& operator<<
        (
            const   tCIDLib::TCard1         c1ToWrite
        );

        TTextStream& operator<<
        (
            const   tCIDLib::TCard2         c2ToWrite
        );

        TTextStream& operator<<
        (
            const   tCIDLib::TCard4         c4ToWrite
        );

        TTextStream& operator<<
        (
            const   tCIDLib::TFloat4        f4ToWrite
        );

        TTextStream& operator<<
        (
            const   tCIDLib::TFloat8&       f8ToWrite
        );

        TTextStream& operator<<
        (
            const   tCIDLib::TInt1          i1ToWrite
        );

        TTextStream& operator<<
        (
            const   tCIDLib::TInt2          i2ToWrite
        );

        TTextStream& operator<<
        (
            const   tCIDLib::TInt4          i4ToWrite
        );

        TTextStream& operator<<
        (
            const   tCIDLib::TInt8          i8ToWrite
        );

        TTextStream& operator<<
        (
            const   tCIDLib::TUInt          uToWrite
        );

        TTextStream& operator<<
        (
            const   tCIDLib::Tch* const     pszToWrite
        );

        TTextStream& operator<<
        (
            const   TTextStream::Width      NewWidth
        );

        TTextStream& operator<<
        (
            const   TTextStream::Precision  NewPrecision
        );

        TTextStream& operator<<
        (
            const   TTextStream::ESpaces    eSpaces
        );

        TTextStream& operator<<
        (
            const   TTextStream::ESpecial   eSpecial
        );

        TTextStream& operator<<
        (
            const   TTextStream::Justify    NewJustification
        );

        TTextStream& operator<<
        (
            const   TTextStream::Fill       NewFillChar
        );

        TTextStream& operator<<
        (
            const   TStreamFmt&             strmfNewFormat
        );


        TTextStream& operator>>
        (
                    tCIDLib::TBoolean&      bToFill
        );

        TTextStream& operator>>
        (
                    tCIDLib::TCard1&        c1ToFill
        );

        TTextStream& operator>>
        (
                    tCIDLib::TCard2&        c2ToFill
        );

        TTextStream& operator>>
        (
                    tCIDLib::TCard4&        c4ToFill
        );

        TTextStream& operator>>
        (
                    tCIDLib::TFloat4&       f4ToFill
        );

        TTextStream& operator>>
        (
                    tCIDLib::TFloat8&       f8ToFill
        );

        TTextStream& operator>>
        (
                    tCIDLib::TInt1&         i1ToFill
        );

        TTextStream& operator>>
        (
                    tCIDLib::TInt2&         i2ToFill
        );

        TTextStream& operator>>
        (
                    tCIDLib::TInt4&         i4ToFill
        );

        TTextStream& operator>>
        (
                    tCIDLib::TInt8&         i8ToFill
        );

        TTextStream& operator>>
        (
                    tCIDLib::TUInt&         uToFill
        );


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bEndOfStream() const;

        tCIDLib::TBoolean bRead();

        tCIDLib::TCard4 c4GetLine
        (
                    TString&                strToFill
            , const tCIDLib::TCard4         c4MaxChars = 0
            , const tCIDLib::TBoolean       bStripWhitespace = kCIDLib::False
        );

        tCIDLib::Tch chGet();

        tCIDLib::TCard1 c1Read
        (
            const   tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TCard2 c2Read
        (
            const   tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TCard4 c4Read
        (
            const   tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::ETextFormats eTextFormat() const;

        tCIDLib::ETextFormats eTextFormat
        (
            const   tCIDLib::ETextFormats   eFormat
        );

        tCIDLib::TFloat4 f4Read();

        tCIDLib::TFloat8 f8Read();

        tCIDLib::TFilePos fposCurPos() const;

        tCIDLib::TFilePos fposLogicalEnd() const;

        tCIDLib::TFilePos fposPhysicalEnd() const;

        tCIDLib::TFilePos fposSeekToEnd();

        tCIDLib::TFilePos fposSkipForwardBy
        (
            const   tCIDLib::TCard4         c4SkipBy
        );

        tCIDLib::TInt1 i1Read
        (
            const   tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TInt2 i2Read
        (
            const   tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TInt4 i4Read
        (
            const   tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TInt8 i8Read
        (
            const   tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TVoid PutCh
        (
            const   tCIDLib::Tch            chToWrite
        );

        tCIDLib::TVoid PutCh
        (
            const   tCIDLib::Tsch           schToWrite
        );

        tCIDLib::TVoid PutLine
        (
            const   TString&                strToWrite
        );

        tCIDLib::TVoid PutLine
        (
            const   tCIDLib::Tch* const     pszToWrite
        );

        tCIDLib::TVoid PutLine
        (
            const   tCIDLib::Tsch* const    pszToWrite
        );

        tCIDLib::TVoid Reset();

        tCIDLib::Tsch schGet();

        tCIDLib::TVoid SetDefaultFormat();

        tCIDLib::TVoid SetFormat
        (
            const   TStreamFmt&             strmfToSet
        );

        tCIDLib::TVoid TruncateAtZero();


    protected   :
        // ---------------------------------------------------------------------
        //  Declare our friends
        // ---------------------------------------------------------------------
        friend class TStreamFmt;


        // --------------------------------------------------------------------
        //  Hidden constructors and operators
        // --------------------------------------------------------------------
        TTextStream();


        // ---------------------------------------------------------------------
        //  Protected, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _AdoptImplObject
        (
                    TTextStreamImpl* const  pstrmiToAdopt
        );

        tCIDLib::TBoolean _bImplSet() const;

        TTextStreamImpl& _strmiThis();

        const TTextStreamImpl& _strmiThis() const;


    private :
        // ---------------------------------------------------------------------
        //  Unimplimented constructors and operators
        // ---------------------------------------------------------------------
        TTextStream(const TTextStream&);

        tCIDLib::TVoid operator=(const TTextStream&);


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __chFill
        //      The file character to use when text formatting into a field
        //      larger than the text being formatted.
        //
        //  __c4Precision
        //      The number of decimal digits to use when formatting a floating
        //      point value.
        //
        //  __c4Width
        //      The field width to format into.
        //
        //  __eJustification
        //      They justification to use when formatting a value that is
        //      smaller than the current field width.
        //
        //  __pstrmiThis
        //      This is a pointer to our stream implementation object. We
        //      adopt it and kill it when we die. We basically pass most of
        //      methods on to it or write them in terms of it.
        // ---------------------------------------------------------------------
        tCIDLib::Tch            __chFill;
        tCIDLib::TCard4         __c4Precision;
        tCIDLib::TCard4         __c4Width;
        tCIDLib::EHJustify      __eJustification;
        TTextStreamImpl*        __pstrmiThis;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TTextStream,TObject)
};



// -----------------------------------------------------------------------------
//   CLASS: TStreamJanitor
//  PREFIX: jan
// -----------------------------------------------------------------------------
class CIDLIBEXP TStreamJanitor : public TObject
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        TStreamJanitor
        (
                    TTextStream* const      pstrmToSanitize
        );

        ~TStreamJanitor();


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and opeators
        // ---------------------------------------------------------------------
        TStreamJanitor();

        TStreamJanitor(const TStreamJanitor&);

        tCIDLib::TVoid operator=(const TStreamJanitor&);


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __pstrmToSanitize
        //      This is the pointer to the textstream we are providing
        //      janitorial services for.
        //
        //  __strmfSave
        //      This is the saved stream format state to be restored.
        // ---------------------------------------------------------------------
        TTextStream*            __pstrmToSanitize;
        TStreamFmt              __strmfSave;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TStreamJanitor,TObject)
};

#pragma pack(pop)




// ---------------------------------------------------------------------------
//  TTextStreamImpl: Public, non-virtual methods
// ---------------------------------------------------------------------------
inline tCIDLib::ETextFormats TTextStreamImpl::eTextFormat() const
{
    return __eTextFmt;
}

inline tCIDLib::ETextFormats
TTextStreamImpl::eTextFormat(const tCIDLib::ETextFormats eFormat)
{
    // Give derived class a chance to reject
    _FormatChange(eFormat);

    // They did not reject so take it
    __eTextFmt = eFormat;
    return __eTextFmt;
}

inline TTextStreamImpl::TTextStreamImpl() :

    __eTextFmt(tCIDLib::ETextFmt_UNICode)
{
}

inline
TTextStreamImpl::TTextStreamImpl(const tCIDLib::ETextFormats eTextFmt) :

    __eTextFmt(eTextFmt)
{
}

inline tCIDLib::TVoid
TTextStreamImpl::_FormatChange(const tCIDLib::ETextFormats eNewFormat)
{
    // Default is to do nothing
}


// -----------------------------------------------------------------------------
//  TTextStream: Pubic operators
// -----------------------------------------------------------------------------
inline TTextStream& TTextStream::operator<<(const TTextStream::Fill NewFill)
{
    __chFill = NewFill.chFill;
    return *this;
}

inline TTextStream&
TTextStream::operator<<(const TTextStream::Justify NewJustification)
{
    __eJustification = NewJustification.eJustification;
    return *this;
}

inline TTextStream&
TTextStream::operator<<(const TTextStream::Precision NewPrecision)
{
    __c4Precision = NewPrecision.c4Precision;
    return *this;
}

inline TTextStream&
TTextStream::operator<<(const TTextStream::Width NewWidth)
{
    __c4Width = NewWidth.c4Width;
    return *this;
}

inline TTextStream& TTextStream::operator<<(const TStreamFmt& strmfNewFormat)
{
    __chFill = strmfNewFormat.__chFill;
    __c4Precision = strmfNewFormat.__c4Precision;
    __c4Width = strmfNewFormat.__c4Width;
    __eJustification = strmfNewFormat.__eJustification;
    return *this;
}


inline TTextStream& TTextStream::operator>>(tCIDLib::TBoolean& bToFill)
{
    bToFill = bRead();
    return *this;
}

inline TTextStream& TTextStream::operator>>(tCIDLib::TCard1& c1ToFill)
{
    c1ToFill = c1Read();
    return *this;
}

inline TTextStream& TTextStream::operator>>(tCIDLib::TCard2& c2ToFill)
{
    c2ToFill = c2Read();
    return *this;
}

inline TTextStream& TTextStream::operator>>(tCIDLib::TCard4& c4ToFill)
{
    c4ToFill = c4Read();
    return *this;
}

inline TTextStream& TTextStream::operator>>(tCIDLib::TFloat4& f4ToFill)
{
    f4ToFill = f4Read();
    return *this;
}

inline TTextStream& TTextStream::operator>>(tCIDLib::TFloat8& f8ToFill)
{
    f8ToFill = f8Read();
    return *this;
}

inline TTextStream& TTextStream::operator>>(tCIDLib::TInt1& i1ToFill)
{
    i1ToFill = i1Read();
    return *this;
}

inline TTextStream& TTextStream::operator>>(tCIDLib::TInt2& i2ToFill)
{
    i2ToFill = i2Read();
    return *this;
}

inline TTextStream& TTextStream::operator>>(tCIDLib::TInt4& i4ToFill)
{
    i4ToFill = i4Read();
    return *this;
}

inline TTextStream& TTextStream::operator>>(tCIDLib::TInt8& i8ToFill)
{
    i8ToFill = i8Read();
    return *this;
}

inline TTextStream& TTextStream::operator>>(tCIDLib::TUInt& uToFill)
{
    uToFill = tCIDLib::TUInt(c4Read());
    return *this;
}


// ----------------------------------------------------------------------------
//  TTextStream: Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TTextStream::bEndOfStream() const
{
    return _strmiThis().bEndOfStream();
}

inline tCIDLib::Tch TTextStream::chGet()
{
    return _strmiThis().chGet();
}

inline tCIDLib::ETextFormats TTextStream::eTextFormat() const
{
    return _strmiThis().eTextFormat();
}

inline tCIDLib::ETextFormats
TTextStream::eTextFormat(const tCIDLib::ETextFormats eFormat)
{
    return _strmiThis().eTextFormat(eFormat);
}

inline tCIDLib::TFilePos TTextStream::fposCurPos() const
{
    return _strmiThis().fposCurPos();
}

inline tCIDLib::TFilePos TTextStream::fposLogicalEnd() const
{
    return _strmiThis().fposLogicalEnd();
}

inline tCIDLib::TFilePos TTextStream::fposPhysicalEnd() const
{
    return _strmiThis().fposPhysicalEnd();
}

inline tCIDLib::TFilePos TTextStream::fposSeekToEnd()
{
    return _strmiThis().fposSeekToEnd();
}

inline tCIDLib::TFilePos
TTextStream::fposSkipForwardBy(const tCIDLib::TCard4 c4SkipBy)
{
    return _strmiThis().fposSkipForwardBy(c4SkipBy);
}

inline tCIDLib::TVoid TTextStream::PutLine(const TString& strToWrite)
{
    _strmiThis().PutLine(strToWrite.pszData());
}

inline tCIDLib::TVoid TTextStream::PutCh(const tCIDLib::Tch chToWrite)
{
    _strmiThis().PutCh(chToWrite);
}

inline tCIDLib::TVoid TTextStream::PutCh(const tCIDLib::Tsch schToWrite)
{
    _strmiThis().PutCh(schToWrite);
}

inline tCIDLib::TVoid
TTextStream::PutLine(const tCIDLib::Tch* const pszToWrite)
{
    _strmiThis().PutLine(pszToWrite);
}

inline tCIDLib::TVoid
TTextStream::PutLine(const tCIDLib::Tsch* const pszToWrite)
{
    _strmiThis().PutLine(pszToWrite);
}

inline tCIDLib::TVoid TTextStream::Reset()
{
    _strmiThis().Reset();
}

inline tCIDLib::Tsch TTextStream::schGet()
{
    return _strmiThis().schGet();
}

inline tCIDLib::TVoid TTextStream::TruncateAtZero()
{
    _strmiThis().TruncateAtZero();
}


// -----------------------------------------------------------------------------
//  TTextStream: Protected, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean TTextStream::_bImplSet() const
{
    return (__pstrmiThis != 0);
}


// -----------------------------------------------------------------------------
//  TStreamJanitor: Constructors and Destructors
// -----------------------------------------------------------------------------
inline TStreamJanitor::TStreamJanitor(TTextStream* const pstrmToSanitize) :

    __pstrmToSanitize(pstrmToSanitize)
    , __strmfSave(*pstrmToSanitize)
{
}

inline TStreamJanitor::~TStreamJanitor()
{
    __pstrmToSanitize->SetFormat(__strmfSave);
}


// -----------------------------------------------------------------------------
//  Create some short cut values for the special text stream enum. These
//  are often used so having a short cut is very convenient.
// -----------------------------------------------------------------------------
const   TTextStream::ESpecial   NewLn   = TTextStream::ENewLine;
const   TTextStream::ESpecial   DNewLn  = TTextStream::EDblNewLine;
const   TTextStream::ESpecial   Cr      = TTextStream::EReturn;
const   TTextStream::ESpecial   Lf      = TTextStream::ELineFeed;
