//
// NAME: CIDLib_TextStream.Cpp
//
// DESCRIPTION:
//
//  This module implements the TTextStream class, which provides text mode
//  input and output over a stream.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/03/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData(TTextStream,TStream)
RTTIData(TTextStreamImpl,TStreamImpl)
RTTIData(TStreamJanitor,TObject)


// -----------------------------------------------------------------------------
//  Local, static data
//
//  __pszFalse
//  __pszTrue
//      These are for loaded text for the boolean values, true and false. We
//      have to compare user input in bRead() to see if its true of false.
//      Since it has to be language sensitive, we load them during init. We
//      set defaults that will be there in case of failure of the load.
// -----------------------------------------------------------------------------
static const tCIDLib::Tch*  __pszFalse = L"False";
static const tCIDLib::Tch*  __pszTrue  = L"True";


// ----------------------------------------------------------------------------
//  Intra-facility methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _InitTermTextStream
//
// DESCRIPTION:
//
//  This function is called during process init/term to initialize and
//  clean up this module.
// ---------------------------------------
//   INPUT: eInitTerm indicates what initialization phase we are in.
//          eGlobals indicates whether this is before constructors or
//              after destructors for globals.
//          modInit is a temp module object for this module.
//          c4MaxChars is the max chars that the failure reason buffer
//              can hold.
//
//  OUTPUT: pszFailReason is filled with the reason for a failure.
//
//  RETURN: None
//
tCIDLib::TVoid
_InitTermTextStream(const   tCIDLib::EInitTerm      eInitTerm
                    , const tCIDLib::EGlobalStates  eGlobals
                    , const TModule&                modInit
                    , const tCIDLib::TCard4         c4MaxChars
                    ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midGen_Unknown);
    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midTStrm_LoadBoolStrs);

            const tCIDLib::Tch* pszTmp;
            pszTmp = modInit.pszLoadCIDMsg(kCIDMsgs::midGen_False);
            if (pszTmp)
                __pszFalse = pszTmp;

            pszTmp = modInit.pszLoadCIDMsg(kCIDMsgs::midGen_True);
            if (pszTmp)
                __pszTrue = pszTmp;
        }
    }

    catch(...)
    {
        TRawStr::CopyCatStr
        (
            pszFailReason
            , c4MaxChars
            , pszPhase
            , L" failed"
        );
        throw;
    }
}



// -----------------------------------------------------------------------------
//   CLASS: TTextStream
//  PREFIX: strm
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TTextStream: Public, static data
// -----------------------------------------------------------------------------
const TStreamFmt TTextStream::strmfDefault
(
    0
    , 2
    , tCIDLib::EHJustify_Left
    , kCIDLib::chSpace
);

// -----------------------------------------------------------------------------
//  TTextStream: Constructors and operators
// -----------------------------------------------------------------------------

inline TTextStream::TTextStream() :

    __c4Precision(2)
    , __c4Width(0)
    , __chFill(kCIDLib::chSpace)
    , __eJustification(tCIDLib::EHJustify_Left)
    , __pstrmiThis(0)
{
}

TTextStream::TTextStream(           TTextStreamImpl* const  pstrmiToAdopt
                            , const TStreamFmt&             strmfToUse) :
     __c4Precision(2)
    , __c4Width(0)
    , __chFill(kCIDLib::chSpace)
    , __eJustification(tCIDLib::EHJustify_Left)
    , __pstrmiThis(pstrmiToAdopt)
{
    //
    //  If there was a stream format object provided, then override the
    //  default values we set up.
    //
    if (&strmfToUse)
    {
        __c4Precision       = strmfToUse.__c4Precision;
        __c4Width           = strmfToUse.__c4Width;
        __chFill            = strmfToUse.__chFill;
        __eJustification    = strmfToUse.__eJustification;
    }
}

TTextStream::~TTextStream()
{
    delete __pstrmiThis;
    __pstrmiThis = 0;
}


// -----------------------------------------------------------------------------
//  TTextStream: Public operators
// -----------------------------------------------------------------------------

TTextStream& TTextStream::operator<<(const tCIDLib::TBoolean bVal)
{
    tCIDLib::TZStr8 szTmp;
    if (bVal)
        TRawStr::CopyStr(szTmp, __pszTrue, c4MaxBufChars(szTmp));
    else
        TRawStr::CopyStr(szTmp, __pszFalse, c4MaxBufChars(szTmp));

    if (__c4Width)
    {
        TString strFmt(kCIDLib::pszEmptyZStr, __c4Width, __c4Width);
        strFmt.FormatToFld
        (
            szTmp
            , __c4Width
            , __eJustification
            , __chFill
        );

        _strmiThis().PutLine(strFmt.pszData());
    }
     else
    {
        _strmiThis().PutLine(szTmp);
    }
    return *this;
}

TTextStream& TTextStream::operator<<(const tCIDLib::TCard1 c1Val)
{
    tCIDLib::TZStr64 szTmp;
    TRawStr::FormatVal(c1Val, szTmp, 64, tCIDLib::ERadix_Dec);

    if (__c4Width)
    {
        TString  strFmt(kCIDLib::pszEmptyZStr, __c4Width, __c4Width);
        strFmt.FormatToFld
        (
            szTmp
            , __c4Width
            , __eJustification
            , __chFill
        );

        _strmiThis().PutLine(strFmt.pszData());
    }
     else
    {
        _strmiThis().PutLine(szTmp);
    }
    return *this;
}

TTextStream& TTextStream::operator<<(const tCIDLib::TCard2 c2Val)
{
    tCIDLib::TZStr64 szTmp;
    TRawStr::FormatVal(c2Val, szTmp, 64, tCIDLib::ERadix_Dec);

    if (__c4Width)
    {
        TString  strFmt(kCIDLib::pszEmptyZStr, __c4Width, __c4Width);
        strFmt.FormatToFld
        (
            szTmp
            , __c4Width
            , __eJustification
            , __chFill
        );

        _strmiThis().PutLine(strFmt.pszData());
    }
     else
    {
        _strmiThis().PutLine(szTmp);
    }
    return *this;
}

TTextStream& TTextStream::operator<<(const tCIDLib::TCard4 c4Val)
{
    tCIDLib::TZStr64 szTmp;
    TRawStr::FormatVal(c4Val, szTmp, 64, tCIDLib::ERadix_Dec);

    if (__c4Width)
    {
        TString  strFmt(kCIDLib::pszEmptyZStr, __c4Width, __c4Width);
        strFmt.FormatToFld
        (
            szTmp
            , __c4Width
            , __eJustification
            , __chFill
        );

        _strmiThis().PutLine(strFmt.pszData());
    }
     else
    {
        _strmiThis().PutLine(szTmp);
    }
    return *this;
}

TTextStream& TTextStream::operator<<(const tCIDLib::TFloat4 f4Val)
{
    tCIDLib::TZStr256   szTmp;
    TRawStr::FormatVal
    (
        f4Val
        , szTmp
        , __c4Precision
        , 256
        , tCIDLib::ETrail_Ignore
    );

    if (__c4Width)
    {
        TString  strFmt(kCIDLib::pszEmptyZStr, __c4Width, __c4Width);
        strFmt.FormatToFld
        (
            szTmp
            , __c4Width
            , __eJustification
            , __chFill
        );
        _strmiThis().PutLine(strFmt.pszData());
    }
     else
    {
        _strmiThis().PutLine(szTmp);
    }
    return *this;
}

TTextStream& TTextStream::operator<<(const tCIDLib::TFloat8& f8Val)
{
    tCIDLib::TZStr256   szTmp;
    TRawStr::FormatVal
    (
        f8Val
        , szTmp
        , __c4Precision
        , 256
        , tCIDLib::ETrail_Ignore
    );

    if (__c4Width)
    {
        TString  strFmt(kCIDLib::pszEmptyZStr, __c4Width, __c4Width);
        strFmt.FormatToFld
        (
            szTmp
            , __c4Width
            , __eJustification
            , __chFill
        );
        _strmiThis().PutLine(strFmt.pszData());
    }
     else
    {
        _strmiThis().PutLine(szTmp);
    }
    return *this;
}

TTextStream& TTextStream::operator<<(const tCIDLib::TInt1 i1Val)
{
    tCIDLib::TZStr64 szTmp;
    TRawStr::FormatVal(i1Val, szTmp, 64, tCIDLib::ERadix_Dec);

    if (__c4Width)
    {
        TString  strFmt(kCIDLib::pszEmptyZStr, __c4Width, __c4Width);
        strFmt.FormatToFld
        (
            szTmp
            , __c4Width
            , __eJustification
            , __chFill
        );
        _strmiThis().PutLine(strFmt.pszData());
    }
     else
    {
        _strmiThis().PutLine(szTmp);
    }
    return *this;
}

TTextStream& TTextStream::operator<<(const tCIDLib::TInt2 i2Val)
{
    tCIDLib::TZStr64 szTmp;
    TRawStr::FormatVal(i2Val, szTmp, 64, tCIDLib::ERadix_Dec);

    if (__c4Width)
    {
        TString  strFmt(kCIDLib::pszEmptyZStr, __c4Width, __c4Width);
        strFmt.FormatToFld
        (
            szTmp
            , __c4Width
            , __eJustification
            , __chFill
        );
        _strmiThis().PutLine(strFmt.pszData());
    }
     else
    {
        _strmiThis().PutLine(szTmp);
    }
    return *this;
}

TTextStream& TTextStream::operator<<(const tCIDLib::TInt4 i4Val)
{
    tCIDLib::TZStr64 szTmp;
    TRawStr::FormatVal(i4Val, szTmp, 64, tCIDLib::ERadix_Dec);

    if (__c4Width)
    {
        TString  strFmt(kCIDLib::pszEmptyZStr, __c4Width, __c4Width);
        strFmt.FormatToFld
        (
            szTmp
            , __c4Width
            , __eJustification
            , __chFill
        );
        _strmiThis().PutLine(strFmt.pszData());
    }
     else
    {
        _strmiThis().PutLine(szTmp);
    }
    return *this;
}

TTextStream& TTextStream::operator<<(const tCIDLib::TInt8 i8Val)
{
    tCIDLib::TZStr128 szTmp;
    TRawStr::FormatVal(i8Val, szTmp, 128, tCIDLib::ERadix_Dec);

    if (__c4Width)
    {
        TString  strFmt(kCIDLib::pszEmptyZStr, __c4Width, __c4Width);
        strFmt.FormatToFld
        (
            szTmp
            , __c4Width
            , __eJustification
            , __chFill
        );
        _strmiThis().PutLine(strFmt.pszData());
    }
     else
    {
        _strmiThis().PutLine(szTmp);
    }
    return *this;
}

TTextStream& TTextStream::operator<<(const tCIDLib::TUInt uVal)
{
    tCIDLib::TZStr64 szTmp;
    TRawStr::FormatVal(uVal, szTmp, 64, tCIDLib::ERadix_Dec);

    if (__c4Width)
    {
        TString  strFmt(kCIDLib::pszEmptyZStr, __c4Width, __c4Width);
        strFmt.FormatToFld
        (
            szTmp
            , __c4Width
            , __eJustification
            , __chFill
        );
        _strmiThis().PutLine(strFmt.pszData());
    }
     else
    {
        _strmiThis().PutLine(szTmp);
    }
    return *this;
}

TTextStream&
TTextStream::operator<<(const tCIDLib::Tch* const pszToWrite)
{
    if (__c4Width)
    {
        TString  strFmt(kCIDLib::pszEmptyZStr, __c4Width, __c4Width);
        strFmt.FormatToFld
        (
            pszToWrite
            , __c4Width
            , __eJustification
            , __chFill
        );
        _strmiThis().PutLine(strFmt.pszData());
    }
     else
    {
        _strmiThis().PutLine(pszToWrite);
    }
    return *this;
}


TTextStream& TTextStream::operator<<(const TTextStream::ESpaces eSpaces)
{
    // Get the count into an int for easier use
    tCIDLib::TInt4 i4Count = eSpaces;

    // If 0, then nothing to do
    if (!i4Count)
        return *this;

    //
    //  Since most of these requests will have pretty reasonable numbers of
    //  spaces, we just create a local string of 64 chars and output the
    //  spaces in blocks of 64. If they are crazy enough to insert a million
    //  spaces, then they will just have deal with a little inefficiency.
    //
    //  In order to avoid massive inefficiency, we find out the output
    //  format of the stream and build either a short or UNICode string
    //  accordingingly. Otherwise, it would covert every block of spaces
    //  just to output them.
    //
    if (_strmiThis().eTextFormat() == tCIDLib::ETextFmt_UNICode)
    {
        tCIDLib::TZStr64    szBuf;

        TRawMem::SetMemBuf(szBuf, kCIDLib::chSpace, 64);
        szBuf[64] = kCIDLib::chNull;

        //
        //  Handle the multiples of 64 first by just filling the string
        //  and bulk outputting until we get below 64.
        //
        if (i4Count > 64)
        {
            while (i4Count > 64)
            {
                _strmiThis().PutLine(szBuf);
                i4Count -= 64;
            }
        }

        // If not an even multiple of 64, then handle the last bit
        if (i4Count)
        {
            szBuf[i4Count] = kCIDLib::chNull;
            _strmiThis().PutLine(szBuf);
        }
    }
     else
    {
        tCIDLib::Tsch   szBuf[65];

        TRawMem::SetMemBuf(szBuf, tCIDLib::TCard1(' '), 64);
        szBuf[64] = 0;

        //
        //  Handle the multiples of 64 first by just filling the string
        //  and bulk outputting until we get below 64.
        //
        if (i4Count > 64)
        {
            while (i4Count > 64)
            {
                _strmiThis().PutLine(szBuf);
                i4Count -= 64;
            }
        }

        // If not an even multiple of 64, then handle the last bit
        if (i4Count)
        {
            szBuf[i4Count] = 0;
            _strmiThis().PutLine(szBuf);
        }
    }
    return *this;
}

TTextStream& TTextStream::operator<<(const TTextStream::ESpecial eSpecial)
{
    if ((eSpecial < ESpecial_Min) || (eSpecial > ESpecial_Max))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_BadSpecial
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(eSpecial)
        );
    }

    if (_strmiThis().eTextFormat() == tCIDLib::ETextFmt_UNICode)
    {
        const tCIDLib::Tch* pszOut = kCIDLib::pszEmptyZStr;
        if (eSpecial == TTextStream::EDblNewLine)
            pszOut = L"\r\n\n";
        else if (eSpecial == TTextStream::ENewLine)
            pszOut = L"\r\n";
        else if (eSpecial == TTextStream::EReturn)
            pszOut = L"\r";
        else if (eSpecial == TTextStream::ELineFeed)
            pszOut = L"\n";

        _strmiThis().PutLine(pszOut);
    }
     else
    {
        const tCIDLib::Tsch* pszOut = "";
        if (eSpecial == TTextStream::EDblNewLine)
            pszOut = "\r\n\n";
        else if (eSpecial == TTextStream::ENewLine)
            pszOut = "\r\n";
        else if (eSpecial == TTextStream::EReturn)
            pszOut = "\r";
        else if (eSpecial == TTextStream::ELineFeed)
            pszOut = "\n";

        _strmiThis().PutLine(pszOut);
    }
    return *this;
}



// -----------------------------------------------------------------------------
//  TTextStream: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TBoolean TTextStream::bRead()
{
    TString strInput(kCIDLib::pszEmptyZStr, 64, 64);
    if (!c4GetLine(strInput, 64, kCIDLib::True))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoInput
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TString("TBoolean")
        );
    }

    if (!strInput.eICompare(__pszTrue) || !strInput.eICompare(L"1"))
         return kCIDLib::True;
    else if (!strInput.eICompare(__pszFalse) || !strInput.eICompare(L"0"))
        return kCIDLib::False;
    else
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputValue
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , strInput
            , TString("TBoolean")
        );
    }

    // Make compiler happy
    return kCIDLib::False;
}

tCIDLib::TCard4
TTextStream::c4GetLine(         TString&            strInput
                        , const tCIDLib::TCard4     c4MaxChars
                        , const tCIDLib::TBoolean   bStripWhitespace)
{
    // Clear the string out
    strInput.Clear();

    //
    //  Figure out what our real maximum input is. Assume the passed
    //  max. If its 0 or greater than the target string, then we go with
    //  the size of the target string.
    //
    tCIDLib::TCard4 c4MaxInput = c4MaxChars;
    if (!c4MaxChars || (c4MaxChars > strInput.c4MaxChars()))
        c4MaxInput = strInput.c4MaxChars();

    // Can't make any use of a 0 sized buffer
    if (!c4MaxInput)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_0_BufSize
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }

    // Force the string to reallocate up to the max input if needed
    if (c4MaxInput > strInput.c4BufChars())
        strInput.ReallocateTo(c4MaxInput);

    return _strmiThis().c4GetLine
    (
        strInput.pszData()
        , c4MaxInput
        , bStripWhitespace
    );
}

tCIDLib::TCard1 TTextStream::c1Read(const tCIDLib::ERadices eRadix)
{
    TString strInput(kCIDLib::pszEmptyZStr, 64, 64);
    if (!c4GetLine(strInput, 64, kCIDLib::True))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoInput
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TString("TCard1")
        );
    }

    // If the first character is a negative sign, then its invalid
    if (strInput[0] == TLocale::chNegSign())
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_UnsignedFmt
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
        );
    }

    tCIDLib::TBoolean   bValid;
    tCIDLib::TCard4     c4Value = strInput.c4Val(eRadix, bValid);

    if (!bValid)
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputValue
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , strInput
            , TString("TCard1")
        );
    }

    if (c4Value > kCIDLib::c1MaxCard)
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputRange
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TCardinal(c4Value)
            , TString("TCard1")
        );
    }
    return tCIDLib::TCard1(c4Value);
}

tCIDLib::TCard2 TTextStream::c2Read(const tCIDLib::ERadices eRadix)
{
    TString strInput(kCIDLib::pszEmptyZStr, 64, 64);
    if (!c4GetLine(strInput, 64, kCIDLib::True))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoInput
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TString("TCard2")
        );
    }

    // If the first character is a negative sign, then its invalid
    if (strInput[0] == TLocale::chNegSign())
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_UnsignedFmt
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
        );
    }

    tCIDLib::TBoolean   bValid;
    tCIDLib::TCard4     c4Value = strInput.c4Val(eRadix, bValid);

    if (!bValid)
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputValue
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , strInput
            , TString("TCard2")
        );
    }

    if (c4Value > kCIDLib::c2MaxCard)
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputRange
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Value)
            , TString("TCard2")
        );
    }
    return tCIDLib::TCard2(c4Value);
}

tCIDLib::TCard4 TTextStream::c4Read(const tCIDLib::ERadices eRadix)
{
    TString strInput(kCIDLib::pszEmptyZStr, 64, 64);
    if (!c4GetLine(strInput, 64, kCIDLib::True))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoInput
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TString("TCard4")
        );
    }

    // If the first character is a negative sign, then its invalid
    if (strInput[0] == TLocale::chNegSign())
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_UnsignedFmt
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
        );
    }

    tCIDLib::TBoolean   bValid;
    tCIDLib::TCard4     c4Value = strInput.c4Val(eRadix, bValid);
    if (!bValid)
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputValue
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , strInput
            , TString("TCard4")
        );
    }
    return c4Value;
}

tCIDLib::TFloat4 TTextStream::f4Read()
{
    TString strInput(kCIDLib::pszEmptyZStr, 64, 64);
    if (!c4GetLine(strInput, 64, kCIDLib::True))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoInput
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TString("TFloat4")
        );
    }

    tCIDLib::TBoolean   bValid;
    tCIDLib::TFloat8    f8Value = strInput.f8Val(bValid);

    if (!bValid)
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputValue
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , strInput
            , TString("TFloat4")
        );
    }

    if ((f8Value < kCIDLib::f4MinFloat) || (f8Value > kCIDLib::f4MaxFloat))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputRange
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TFloat(f8Value)
            , TString("TFloat4")
        );
    }
    return tCIDLib::TFloat4(f8Value);
}

tCIDLib::TFloat8 TTextStream::f8Read()
{
    TString strInput(kCIDLib::pszEmptyZStr, 64, 64);
    if (!c4GetLine(strInput, 64, kCIDLib::True))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoInput
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TString("TFloat8")
        );
    }

    tCIDLib::TBoolean   bValid;
    tCIDLib::TFloat8    f8Value = strInput.f8Val(bValid);

    if (!bValid)
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputValue
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , strInput
            , TString("TFloat8")
        );
    }

    return f8Value;
}


tCIDLib::TInt1 TTextStream::i1Read(const tCIDLib::ERadices eRadix)
{
    TString strInput(kCIDLib::pszEmptyZStr, 64, 64);
    if (!c4GetLine(strInput, 64, kCIDLib::True))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoInput
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TString("TInt1")
        );
    }

    tCIDLib::TBoolean   bValid;
    tCIDLib::TInt4      i4Value = strInput.i4Val(eRadix, bValid);

    if (!bValid)
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputValue
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , strInput
            , TString("TInt1")
        );
    }

    if ((i4Value < kCIDLib::i1MinInt)
    ||  (i4Value > kCIDLib::i1MaxInt))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputRange
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TInteger(i4Value)
            , TString("TInt1")
        );
    }
    return tCIDLib::TInt1(i4Value);
}

tCIDLib::TInt2 TTextStream::i2Read(const tCIDLib::ERadices eRadix)
{
    TString strInput(kCIDLib::pszEmptyZStr, 64, 64);
    if (!c4GetLine(strInput, 64, kCIDLib::True))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoInput
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TString("TInt2")
        );
    }

    tCIDLib::TBoolean   bValid;
    tCIDLib::TInt4      i4Value = strInput.i4Val(eRadix, bValid);

    if (!bValid)
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputValue
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , strInput
            , TString("TInt2")
        );
    }

    if ((i4Value < kCIDLib::i2MinInt)
    ||  (i4Value > kCIDLib::i2MaxInt))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputRange
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TInteger(i4Value)
            , TString("TInt2")
        );
    }
    return tCIDLib::TInt2(i4Value);
}

tCIDLib::TInt4 TTextStream::i4Read(const tCIDLib::ERadices eRadix)
{
    TString strInput(kCIDLib::pszEmptyZStr, 64, 64);
    if (!c4GetLine(strInput, 64, kCIDLib::True))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoInput
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TString("TInt4")
        );
    }

    tCIDLib::TBoolean   bValid;
    tCIDLib::TInt4      i4Value = strInput.i4Val(eRadix, bValid);

    if (!bValid)
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputValue
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , strInput
            , TString("TInt4")
        );
    }
    return i4Value;
}

tCIDLib::TInt8 TTextStream::i8Read(const tCIDLib::ERadices eRadix)
{
    TString strInput(kCIDLib::pszEmptyZStr, 64, 64);
    if (!c4GetLine(strInput, 64, kCIDLib::True))
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoInput
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , TString("TInt8")
        );
    }

    tCIDLib::TBoolean   bValid;
    tCIDLib::TInt8      i8Value = strInput.i8Val(eRadix, bValid);

    if (!bValid)
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InputValue
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_UserInput
            , strInput
            , TString("TInt8")
        );
    }
    return i8Value;
}

tCIDLib::TVoid TTextStream::SetDefaultFormat()
{
    __c4Precision = 2;
    __c4Width = 0;
    __chFill = kCIDLib::chSpace;
    __eJustification = tCIDLib::EHJustify_Left;
}

tCIDLib::TVoid TTextStream::SetFormat(const TStreamFmt& strmfToSet)
{
    __c4Precision = strmfToSet.__c4Precision;
    __c4Width = strmfToSet.__c4Width;
    __chFill = strmfToSet.__chFill;
    __eJustification = strmfToSet.__eJustification;
}


// -----------------------------------------------------------------------------
//  TTextStream: Protected, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TTextStream::_AdoptImplObject(TTextStreamImpl* const pstrmiToAdopt)
{
    if (__pstrmiThis)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_ImplAlreadySet
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , clsIsA()
        );
    }
    __pstrmiThis = pstrmiToAdopt;
}

TTextStreamImpl& TTextStream::_strmiThis()
{
    if (!__pstrmiThis)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoImplementation
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , clsIsA()
        );
    }
    return *__pstrmiThis;
}

const TTextStreamImpl& TTextStream::_strmiThis() const
{
    if (!__pstrmiThis)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_NoImplementation
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , clsIsA()
        );
    }
    return *__pstrmiThis;
}
