//
// NAME: CIDLib_TexTTextFileLogger.Cpp
//
// DESCRIPTION:
//
//  This module implements a file logger, which sends logged errors to a
//  text output file, as formatted text strings.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/27/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData2(TTextFileLogger,TLogger)


// ----------------------------------------------------------------------------
//  TTextFileLogger: Constructors and operators
// ----------------------------------------------------------------------------

TTextFileLogger::TTextFileLogger() :

    __pmtxSync(0)
    , __pstrmTarget(0)
{
}

TTextFileLogger::TTextFileLogger(   const  tCIDLib::Tch* const      pszFile
                                    , const tCIDLib::ETextFormats   eTextFmt
                                    , const TResourceName&          rsnMutex) :
    __pmtxSync(0)
    , __pstrmTarget(0)
{
    Create(pszFile, eTextFmt, rsnMutex);
}

TTextFileLogger::~TTextFileLogger()
{
    delete __pmtxSync;
    delete __pstrmTarget;
}


// ----------------------------------------------------------------------------
//  TTextFileLogger: Public, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TTextFileLogger::LogMsg(const TError& errToLog)
{
    //
    //  Create a static string. We know we can do this because calls to
    //  this method are thread serialized by the CIDLib code that calls it.
    //
    static tCIDLib::TBoolean    bFirstTime = kCIDLib::True;
    static TStringStream        strmLog(4000);
    static TTime                tmCurrent;

    //
    //  If this is the first time through, then we need to set the default
    //  format for the time object so that it will naturally format in the
    //  style we want.
    //
    if (bFirstTime)
    {
        tmCurrent.strDefaultFormat(L"%(M,2,0)/%(D,2,0)  %(R,2,0):%(h,2,0)");
        bFirstTime = kCIDLib::False;
    }

    // Tell the time object to set itself to the current time
    tmCurrent.SetToNow();

    // Reset the stream before we start formatting again
    strmLog.Reset();

    //
    //  Build up the string that we are going to log. Its relatively
    //  complex and we have to use low level stuff here because of the
    //  low level.
    //
    //  We start by putting the date and time into the message, which
    //  is particularly important when looking back after the fact.
    //
    strmLog << L"(" << tmCurrent << L")  - "
            << TKrnlSysInfo::pszProcessName()
            << NewLn << L"     ("
            << errToLog.strModName()
            << L", " << TThread::pthrCaller()->strName()
            << L", " << errToLog.strFileName() << L"."
            << errToLog.c4LineNum()
            << L", " << errToLog.eSeverity()
            << L"/" << errToLog.eClass();

    // Only do the error stuff if there are any non-zero codes
    if (errToLog.bAnyError())
    {
        strmLog << L", Error:" << errToLog.errcId()
                << L"/" << errToLog.errcKrnlId()
                << L"/" << errToLog.errcHostId();
    }
    strmLog << L")" << NewLn;

    // Now append the error text itself, indenting it over
    strmLog << L"     " << errToLog.strErrText();

    // If there is aux text, log it next and indent it also.
    if (!errToLog.strAuxText().bEmpty())
        strmLog << NewLn << L"     " << errToLog.strAuxText();

    strmLog << DNewLn;

    //
    //  According to whether we have a sync semaphore, we have to do
    //  one or another way here.
    //
    if (__pmtxSync)
    {
        TMtxLock mtxlSync(__pmtxSync);
        *__pstrmTarget << strmLog.strData();
    }
     else
    {
        *__pstrmTarget << strmLog.strData();
    }
}


// ----------------------------------------------------------------------------
//  TTextFileLogger: Private, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TTextFileLogger::__Create(  const   tCIDLib::Tch* const     pszFile
                            , const tCIDLib::ETextFormats   eTextFmt
                            , const TResourceName&          rsnMutex)
{
    if (__pstrmTarget)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcLgr_AlreadyInitialized
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_Already
        );
    }

    // Create the text file stream
    __pstrmTarget = new TTextFileStream
    (
        pszFile
        , eTextFmt
        , tCIDLib::EAccess_Write
        , tCIDLib::ECreateAct_OpenOrCreate
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileFlags
          (
            tCIDLib::EFileFlag_WriteThrough
            | tCIDLib::EFileFlag_SequentialScan
          )
    );

    // Seek to the end so that we append
    __pstrmTarget->fposSeekToEnd();

    // Try to create or open the named mutex, if the user provided a name
    if (&rsnMutex)
        __pmtxSync = new TMutex(rsnMutex);
}
