//
// NAME: CIDLib_SystemInfo.Cpp
//
// DESCRIPTION:
//        
//  This module implements the TSsyInfo class, which provides access to system
//  information.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/25/92
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData(TSysInfo,TObject)



// -----------------------------------------------------------------------------
//  Intra-facility methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _InitTermSysIfno
//
// DESCRIPTION:
//
//  This is called from the DLL init. It checks the previously parsed command
//  line arguments (which the kernel has already done) and pulls out the magic
//  ones from the list, setting appropriate static data members to indicate
//  what these args indicate.
// ---------------------------------------
//   INPUT: eInitTerm indicates what initialization phase we are in.
//          eGlobals indicates whether this is before constructors or
//              after destructors for globals.
//          modInit is a temp module object for this module.
//          c4MaxChars is the max chars that the failure reason buffer
//              can hold.
//
//  OUTPUT: pszFailReason is filled with the reason for a failure.
//
//  RETURN: None
//
tCIDLib::TVoid
_InitTermSysInfo(   const   tCIDLib::EInitTerm      eInitTerm
                    , const tCIDLib::EGlobalStates  eGlobals
                    , const TModule&                modInit
                    , const tCIDLib::TCard4         c4MaxChars
                    ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszPhase = modInit.pszLoadCIDMsg(kCIDMsgs::midGen_Unknown);
    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            //
            //  Get the logging severity parm text since its used more than
            //  once below.
            //
            const tCIDLib::Tch* pszLogSevParm = modInit.pszLoadCIDMsg
            (
                kCIDMsgs::midSysI_LogSeverity
            );

            // Start after the magic 0th parameter
            tCIDLib::TCard4 c4Index = 1;
            while (1)
            {
                // Get the new command line argument count
                tCIDLib::TCard4 c4ArgCount = TKrnlSysInfo::c4CmdLineArgCount();

                // If we've gotten down to the magic 1st parameter we are done
                if (c4ArgCount == 1)
                    break;

                // If we've hit the end of the list, we are done
                if (c4Index >= c4ArgCount)
                    break;

                // Get a pointer to this param's text
                const tCIDLib::Tch* const pszVal
                                        = TKrnlSysInfo::pszCmdLineArg(c4Index);

                //
                //  Check to see if its a special parm. If so, then we set the
                //  appropriate flag and remove it from the list.
                //
                tCIDLib::TBoolean bMagic = kCIDLib::True;
                if (!TRawStr::eICompareStr
                (
                    pszVal, modInit.pszLoadCIDMsg(kCIDMsgs::midSysI_Verbose)))
                {
                    TSysInfo::__bVerboseMode = kCIDLib::True;
                    TFacility::eSevThreshold(tCIDLib::ESev_Information);
                }
                 else if (!TRawStr::eICompareStr
                      (pszVal, modInit.pszLoadCIDMsg(kCIDMsgs::midSysI_TestMode)))
                {
                    TSysInfo::__bTestMode = kCIDLib::True;
                }
                 else if (!TRawStr::eICompareStr
                      (pszVal, modInit.pszLoadCIDMsg(kCIDMsgs::midSysI_ObjMetrics)))
                {
                    TSysInfo::__bObjectMetrics = kCIDLib::True;
                }
                 else if (!TRawStr::eICompareStr
                      (pszVal, modInit.pszLoadCIDMsg(kCIDMsgs::midSysI_Install)))
                {
                    TSysInfo::__bInstallMode = kCIDLib::True;
                }
                 else if (!TRawStr::eNICompareStr(pszVal, pszLogSevParm, 8))
                {
                    //
                    //  We need to set the logging level. We use the static
                    //  access method to call eSevThreshold() so we don't worry
                    //  about whether our facility object exists yet.
                    //
                    if (!TRawStr::eICompareStr
                    (
                          &(pszVal[8]), modInit.pszLoadCIDMsg(kCIDMsgs::midSysI_Info)))
                    {
                        TFacility::eSevThreshold(tCIDLib::ESev_Information);
                    }
                     else if (!TRawStr::eICompareStr
                          (&(pszVal[8]), modInit.pszLoadCIDMsg(kCIDMsgs::midSysI_Warning)))
                    {
                        TFacility::eSevThreshold(tCIDLib::ESev_Warning);
                    }
                    else if (!TRawStr::eICompareStr
                         (&(pszVal[8]), modInit.pszLoadCIDMsg(kCIDMsgs::midSysI_APIFailed)))
                    {
                        TFacility::eSevThreshold(tCIDLib::ESev_APIFailed);
                    }
                    else if (!TRawStr::eICompareStr
                         (&(pszVal[8]), modInit.pszLoadCIDMsg(kCIDMsgs::midSysI_ProcFatal)))
                    {
                        TFacility::eSevThreshold(tCIDLib::ESev_ProcessFatal);
                    }
                     else
                    {
                        TRawStr::CopyCatStr
                        (
                            pszFailReason
                            , c4MaxChars
                            , pszLogSevParm
                            , modInit.pszLoadCIDMsg(kCIDMsgs::midSysI_ParamInvalid)
                        );

                        facCIDLib.LogErr
                        (
                            __FILE__
                            , __LINE__
                            , kCIDErrs::errcSysI_BadStdParm
                            , tCIDLib::ESev_ProcessFatal
                            , tCIDLib::EClass_BadParms
                            , TString(&(pszVal[TRawStr::c4StrLen(pszLogSevParm)]))
                            , TString(pszLogSevParm)
                        );
                    }
                }
                 else
                {
                    // Wasn't a magic parameter
                    bMagic = kCIDLib::False;
                }

                //
                //  If a magic parameter, pull this one out of the list. Else
                //  bump up the index. We don't bump the index if a magic param
                //  because removing it will bring the others down to us.
                //
                if (bMagic)
                    TKrnlSysInfo::RemoveCmdLineArg(c4Index);
                else
                    c4Index++;
            }
        }
    }

    catch(...)
    {
        TRawStr::CopyStr(pszFailReason, pszPhase, c4MaxChars);
        throw;
    }
}



// ----------------------------------------------------------------------------
//   CLASS: TSysInfo
//  PREFIX: sysi
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TSysInfo: Static data members
// ----------------------------------------------------------------------------
tCIDLib::TBoolean   TSysInfo::__bInstallMode;
tCIDLib::TBoolean   TSysInfo::__bObjectMetrics;
tCIDLib::TBoolean   TSysInfo::__bTestMode;
tCIDLib::TBoolean   TSysInfo::__bVerboseMode;


// ----------------------------------------------------------------------------
//  TSysInfo: Public, static methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid
TSysInfo::CmdLineArg(const tCIDLib::TCard4 c4ArgNum, TString& strBuf)
{
    // See if the argument number is ok, else return eFalse.
    if (c4ArgNum >= TKrnlSysInfo::c4CmdLineArgCount())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSysI_CmdLineIndex
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4ArgNum)
            , TCardinal(TKrnlSysInfo::c4CmdLineArgCount()-1)
        );
    }

    // The index is ok, so get the value into the passed string buffer
    strBuf = TKrnlSysInfo::pszCmdLineArg(c4ArgNum);
}


TRGBClr TSysInfo::rgbSystemColor(const tCIDLib::ESysColors eSysColor)
{
    tCIDLib::TCard4 c4Color;
    try
    {
        c4Color = TKrnlSysInfo::c4SystemColor(eSysColor);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSysI_SysColor
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TInteger(eSysColor)
        );
    }
    return TRGBClr(c4Color);
}


TString TSysInfo::strUserName()
{
    // Use a local buffer plenty long enough
    tCIDLib::TZStr256   pszTmp;

    try
    {
        TKrnlSysInfo::QueryUserName(pszTmp, c4MaxBufChars(pszTmp));
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSysI_GetUser
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    return TString(pszTmp);
}
