//
// NAME: CIDLib_StringStreamImpl.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDLib_StringStreamImpl.Cpp module. This
//  module implements a stream implementation object. It allows text streaming
//  to and from a string object.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/28/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//   CLASS: TStringStreamImpl
//  PREFIX: strmi
// -----------------------------------------------------------------------------
class CIDLIBEXP TStringStreamImpl : public TTextStreamImpl
{
    public  :
        // ---------------------------------------------------------------------
        //  Public types
        // ---------------------------------------------------------------------
        enum ECheck
        {
            ECheck_ForRead
            , ECheck_ForWrite
        };


        // ---------------------------------------------------------------------
        //  Constructors and destructors
        // ---------------------------------------------------------------------
        TStringStreamImpl
        (
                    TString* const          pstrData
            , const tCIDLib::EAdoptOpts     eAdopt = tCIDLib::EAdoptOpt_Adopt
        );

        TStringStreamImpl
        (
            const   tCIDLib::TCard4         c4InitSize
            , const tCIDLib::TCard4         c4MaxSize = 0
        );

        ~TStringStreamImpl();


        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bEndOfStream() const;

        tCIDLib::TCard4 c4GetLine
        (
                    tCIDLib::Tch* const     pszBufToFill
            , const tCIDLib::TCard4         c4MaxChars
            , const tCIDLib::TBoolean       bStripWhitespace = kCIDLib::True
        );

        tCIDLib::Tch chGet();

        tCIDLib::TFilePos fposCurPos() const;

        tCIDLib::TFilePos fposLogicalEnd() const;

        tCIDLib::TFilePos fposPhysicalEnd() const;

        tCIDLib::TFilePos fposSeekToEnd();

        tCIDLib::TFilePos fposSkipForwardBy
        (
            const   tCIDLib::TCard4         c4SkipBy
        );

        tCIDLib::TVoid PutCh
        (
            const   tCIDLib::Tch            chToWrite
        );

        tCIDLib::TVoid PutCh
        (
            const   tCIDLib::Tsch           schToWrite
        );

        tCIDLib::TVoid PutLine
        (
            const   tCIDLib::Tch* const     pszBufToWrite
        );

        tCIDLib::TVoid PutLine
        (
            const   tCIDLib::Tsch* const    pszBufToWrite
        );

        tCIDLib::TVoid Reset();

        tCIDLib::Tsch schGet();

        tCIDLib::TVoid TruncateAtZero();


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid AdoptString();

        tCIDLib::TBoolean bStringIsAdopted() const;

        TString& strData();

        const TString& strData() const;


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _FormatChange
        (
            const   tCIDLib::ETextFormats   eNewFormat
        );


    private     :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TStringStreamImpl();

        TStringStreamImpl(const TStringStreamImpl&);

        tCIDLib::TVoid operator=(const TStringStreamImpl&);


        // ---------------------------------------------------------------------
        //  Private, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid __CheckAvail
        (
            const   tCIDLib::TCard4         c4Line
            , const tCIDLib::TCard4         c4Chars
            , const ECheck                  eCheckFor
        );

        tCIDLib::TVoid __WriteChars
        (
            const   tCIDLib::Tch* const     pchBufToWrite
            , const tCIDLib::TCard4         c4CharCount
        );


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __c4CurPos
        //      This is the current position that we are in the string.
        //
        //  __eAdopted
        //      This indicates whether we've adopted the string in
        //      __pstrData.
        //
        //  __pstrData
        //      This is the pointer to the string that provides the data
        //      sink/source for this implementation.
        // ---------------------------------------------------------------------
        tCIDLib::TCard4         __c4CurPos;
        tCIDLib::EAdoptOpts     __eAdopted;
        TString*                __pstrData;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TStringStreamImpl,TTextStreamImpl)
};

#pragma pack(pop)


// -----------------------------------------------------------------------------
//  Public, inherited methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean TStringStreamImpl::bEndOfStream() const
{
    if (__c4CurPos >= __pstrData->c4Length())
        return kCIDLib::True;
    return kCIDLib::False;
}

inline tCIDLib::TFilePos TStringStreamImpl::fposCurPos() const
{
    return tCIDLib::TFilePos(__c4CurPos);
}

inline tCIDLib::TFilePos TStringStreamImpl::fposLogicalEnd() const
{
    return tCIDLib::TFilePos(__pstrData->c4Length());
}

inline tCIDLib::TFilePos TStringStreamImpl::fposPhysicalEnd() const
{
    return tCIDLib::TFilePos(__pstrData->c4MaxChars());
}

inline tCIDLib::TFilePos TStringStreamImpl::fposSeekToEnd()
{
    __c4CurPos = __pstrData->c4Length();
    return tCIDLib::TFilePos(__c4CurPos);
}

inline tCIDLib::TVoid TStringStreamImpl::Reset()
{
    __c4CurPos = 0;
}

inline tCIDLib::TVoid TStringStreamImpl::TruncateAtZero()
{
    // Empty the string and set the current position back to 0
    __c4CurPos = 0;
    __pstrData->Clear();
}


// -----------------------------------------------------------------------------
//  Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TVoid TStringStreamImpl::AdoptString()
{
    __eAdopted = tCIDLib::EAdoptOpt_Adopt;
}

inline tCIDLib::TBoolean TStringStreamImpl::bStringIsAdopted() const
{
    if (__eAdopted)
        return kCIDLib::True;
    return kCIDLib::False;
}

inline TString& TStringStreamImpl::strData()
{
    return *__pstrData;
}

inline const TString& TStringStreamImpl::strData() const
{
    return *__pstrData;
}
