//
// NAME: CIDLib_StringStreamImpl.Cpp
//
// DESCRIPTION: 
//
//  This method implements the stream implementation derivative that allows
//  a stream to use a string as a data sink/source.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/28/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData(TStringStreamImpl,TTextStreamImpl)


// -----------------------------------------------------------------------------
//   CLASS: TStringStreamImpl
//  PREFIX: strm
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TStringStreamImpl: Constructors and Destructors
// -----------------------------------------------------------------------------

TStringStreamImpl::TStringStreamImpl(       TString* const      pstrToUse
                                    , const tCIDLib::EAdoptOpts eAdopt) :

    TTextStreamImpl(tCIDLib::ETextFmt_UNICode)
    , __c4CurPos(0)
    , __eAdopted(eAdopt)
    , __pstrData(pstrToUse)
{
}

TStringStreamImpl::TStringStreamImpl(   const   tCIDLib::TCard4 c4InitSize
                                        , const tCIDLib::TCard4 c4MaxSize) :

    TTextStreamImpl(tCIDLib::ETextFmt_UNICode)
    , __c4CurPos(0)
    , __eAdopted(tCIDLib::EAdoptOpt_Adopt)
    , __pstrData(0)
{
    // If max size is 0, then its really same as init
    const tCIDLib::TCard4 c4ActualMax = c4MaxSize ? c4MaxSize : c4InitSize;

    // Create the string
    __pstrData = new TString(kCIDLib::pszEmptyZStr, c4InitSize, c4ActualMax);
}

TStringStreamImpl::~TStringStreamImpl()
{
    if (__eAdopted)
    {
        delete __pstrData;
        __pstrData = 0;
    }
}


// -----------------------------------------------------------------------------
//  TStringStreamImpl: Public, inherted methods
// -----------------------------------------------------------------------------

tCIDLib::TCard4
TStringStreamImpl::c4GetLine(           tCIDLib::Tch* const     pszBufToFill
                                , const tCIDLib::TCard4         c4BufChars
                                , const tCIDLib::TBoolean       bStripWhitespace)
{
    if (bEndOfStream())
    {
        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_EndOfStream
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_OutResource
        );
    }

    tCIDLib::Tch    chCur;
    tCIDLib::TCard4 c4Count = 0;
    tCIDLib::TCard4 c4Len = __pstrData->c4Length();
    while (c4Count < c4Len)
    {
        chCur = __pstrData->chAt(__c4CurPos);
        __c4CurPos++;

        // If a newline, then cap the string and return
        if (chCur == kCIDLib::chLF)
        {
            pszBufToFill[c4Count] = 0;
            break;
        }

        // Put it in the buffer and bump up the count
        pszBufToFill[c4Count] = chCur;
        c4Count++;

        //
        //  Don't overflow the buffer. And leave room for the terminating
        //  null.
        //
        if (c4Count >= c4BufChars-1)
            break;
    }

    // Cap it off for safety
    pszBufToFill[c4Count] = 0;

    //
    //  If the last character was a carraige return, then do away with
    //  it, since its just part of a CR/LF at the line end.
    //
    if (c4Count)
    {
        if (pszBufToFill[c4Count-1] == kCIDLib::chCR)
        {
            pszBufToFill[c4Count-1] = kCIDLib::chNull;
            c4Count--;
        }
    }

    // Strip white space off it requested, and adjust the count
    if (bStripWhitespace)
    {
        TRawStr::StripStr
        (
            pszBufToFill
            , kCIDLib::szWhitespace
            , tCIDLib::EStripMode_LeadTrail
        );
        c4Count = TRawStr::c4StrLen(pszBufToFill);
    }

    return c4Count;
}


tCIDLib::Tch TStringStreamImpl::chGet()
{
    __CheckAvail(__LINE__, 1, ECheck_ForRead);
    tCIDLib::Tch chRet = __pstrData->chAt(__c4CurPos);
    __c4CurPos++;
    return chRet;
}


tCIDLib::TFilePos
TStringStreamImpl::fposSkipForwardBy(const tCIDLib::TCard4 c4SkipBy)
{
    const tCIDLib::TCard4 c4Length = __pstrData->c4Length();
    const tCIDLib::TCard4 c4Max    = __pstrData->c4MaxChars();
    const tCIDLib::TCard4 c4NewPos = __c4CurPos + c4SkipBy;

    //
    //  See whether this would seek beyond the physical end. If so, this
    //  is an error.
    //
    if ((c4SkipBy > kCIDLib::c4MaxCard) || (c4NewPos > c4Max))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_InvalidRelSeek
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(c4SkipBy)
        );
    }

    //
    //  If this would be beyond the string's current buffer, then reallocate
    //  it now for efficiency. The default is to preserve current content.
    //
    if (c4NewPos > __pstrData->c4BufChars())
        __pstrData->ReallocateTo(c4NewPos);

    //
    //  If we are seeking past the logical end, the we need to space fill
    //  it out to there.
    //
    if (c4NewPos > c4Length)
    {
        for (tCIDLib::TCard4 c4Index = c4Length; c4Index < c4NewPos; c4Index++)
            (*__pstrData)[c4Index] = kCIDLib::chSpace;
    }

    // Move up to the new position
    __c4CurPos = c4NewPos;

    return tCIDLib::TFilePos(__c4CurPos);
}


tCIDLib::TVoid TStringStreamImpl::PutCh(const tCIDLib::Tch chToWrite)
{
    tCIDLib::Tch szPut[2];
    szPut[0] = chToWrite;
    szPut[1] = kCIDLib::chNull;
    PutLine(szPut);
}

tCIDLib::TVoid TStringStreamImpl::PutCh(const tCIDLib::Tsch schToWrite)
{
    tCIDLib::Tch szPut[2];
    szPut[0] = TRawStr::chConvert(schToWrite);
    szPut[1] = kCIDLib::chNull;
    PutLine(szPut);
}


tCIDLib::TVoid
TStringStreamImpl::PutLine(const tCIDLib::Tch* const pszBufToWrite)
{
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(pszBufToWrite);
    if (!c4Len)
        return;

    __CheckAvail(__LINE__, c4Len, ECheck_ForWrite);
    __WriteChars(pszBufToWrite, c4Len);
}

tCIDLib::TVoid
TStringStreamImpl::PutLine(const tCIDLib::Tsch* const pszBufToWrite)
{
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(pszBufToWrite);
    if (!c4Len)
        return;

    __CheckAvail(__LINE__, c4Len, ECheck_ForWrite);

    tCIDLib::Tch*   pszTmp = TRawStr::pszConvert(pszBufToWrite);
    THeapJanitor    janTmpStr(pszTmp);

    __WriteChars(pszTmp, c4Len);
}


tCIDLib::Tsch TStringStreamImpl::schGet()
{
    __CheckAvail(__LINE__, 1, ECheck_ForRead);
    tCIDLib::Tch chRet = __pstrData->chAt(__c4CurPos);
    __c4CurPos++;
    return TRawStr::schConvert(chRet);
}



// -----------------------------------------------------------------------------
//  TStringStreamImpl: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TStringStreamImpl::_FormatChange(const tCIDLib::ETextFormats eNewFormat)
{
    if (eNewFormat == tCIDLib::ETextFmt_ASCII)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStrm_TextFormat
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TString(L"TString")
            , TString(L"UNICode")
        );
    }
}


// -----------------------------------------------------------------------------
//  TStringStreamImpl: Private, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TStringStreamImpl::__CheckAvail(const   tCIDLib::TCard4 c4Line
                                , const tCIDLib::TCard4 c4Chars
                                , const ECheck          eCheck)
{
    tCIDLib::TCard4 c4Avail;
    if (eCheck == ECheck_ForWrite)
        c4Avail = __pstrData->c4MaxChars() - __c4CurPos;
    else
        c4Avail = __pstrData->c4Length() - __c4CurPos;

    if (c4Avail < c4Chars)
    {
        tCIDLib::TErrCode errcToLog;
        if (!c4Avail)
            errcToLog = kCIDErrs::errcStrm_EndOfStream;
        else
            errcToLog = kCIDErrs::errcStrm_NotAllData;

        facCIDLib.ThrowErr
        (
            __FILE__
            , __LINE__
            , errcToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_OutResource
        );
    }
}

tCIDLib::TVoid
TStringStreamImpl::__WriteChars(const   tCIDLib::Tch* const pchBufToWrite
                                , const tCIDLib::TCard4     c4CharCount)
{
    //
    //  If it would exceed the current buffer size, then reallocate. To
    //  avoid doing it too often, actually reallocate an extra 64 bytes,
    //  or up to that much if we can.
    //
    if (__c4CurPos + c4CharCount > __pstrData->c4BufChars())
    {
        tCIDLib::TCard4 c4NewSize =  __c4CurPos
                                     + c4CharCount + 64;
        if (c4NewSize > __pstrData->c4MaxChars())
            c4NewSize = __pstrData->c4MaxChars();

        // The default is to preserve the content, which is what we want
        __pstrData->ReallocateTo(c4NewSize);
    }

    //
    //  Get the raw string buffer and copy the new data into it. Then
    //  cap it off. I know its ugly but its legal, and keeps us from
    //  having to provide a method in the string class to import a raw
    //  binary buffer.
    //
    tCIDLib::Tch* pszTmp = __pstrData->pszData();
    TRawMem::CopyMemBuf
    (
        &pszTmp[__c4CurPos]
        , pchBufToWrite
        , c4CharCount * kCIDLib::c4CharBytes
    );

    // Bump up the current position by the chars written
    __c4CurPos += c4CharCount;

    // Cap it off here at the new current position
    __pstrData->CapAt(__c4CurPos);
}
