//
// NAME: CIDLib_String.Hpp
//
// DESCRIPTION:
//
//  This module implements the TString class, which (suprise) implements a
//  wrapper for nul terminated strings. Strings have a max size that they
//  can expand to, and a current size that represents the actual allocated
//  amount of data. This class is very full featured because a lot of grunt
//  work effort is otherwise often wasted doing the same string operations
//  over and over.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/16/92
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  The current buffer size is not counted in the == operator. Only the
//      max size must be equal (because the actual buffer size is just an
//      internal abstraction.) The compared string would still operate just
//      the same as the orginal, since it will grow to the same max size.
//
//      HOWEVER, when you extract raw string buffer pointers from string
//      objects, don't count on their raw buffer sizes being the same just
//      because you just compared them!!!
//
//  2)  If you just want to compare the text buffers, use bSameText(). The
//      == operator compares the whole object, subject to #1 above.
//


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
class   TFacility;
class   TStreamFmt;


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//  CLASS: TString
// PREFIX: str
// ----------------------------------------------------------------------------
class CIDLIBEXP TString :

    public TObject, public MFormattable, public MDuplicable, public MStreamable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors.
        // --------------------------------------------------------------------
        TString();

        TString
        (
            const   tCIDLib::Tch            chInitValue
        );

        TString
        (
            const   TString&                strToCopy
        );

        TString
        (
            const   tCIDLib::Tch* const     pszInitValue1
            , const tCIDLib::Tch* const     pszInitValue2
        );

        TString
        (
            const   tCIDLib::Tch* const     pszInitValue
            , const tCIDLib::TCard4         c4InitChars = 0
            , const tCIDLib::TCard4         c4MaxChars = 2048
        );

        TString
        (
            const   tCIDLib::Tsch* const    pszInitValue
            , const tCIDLib::TCard4         c4InitChars = 0
            , const tCIDLib::TCard4         c4MaxChars = 2048
        );

        TString
        (
            const   tCIDLib::TMsgId         midToLoad
            , const TFacility&              facSource
            , const tCIDLib::TCard4         c4Extra = 64
        );

        TString
        (
            const   MFormattable&           fmtblInitValue
            , const TStreamFmt&             strmfToUse = NUL_TStreamFmt
        );
 
        TString
        (
            const   tCIDLib::TMsgId         midToLoad
            , const TFacility&              facSource
            , const tCIDLib::TCard4         c4Extra
            , const MFormattable&           fmtblToken1
            , const MFormattable&           fmtblToken2 = NUL_MFormattable
            , const MFormattable&           fmtblToken3 = NUL_MFormattable
            , const MFormattable&           fmtblToken4 = NUL_MFormattable
        );

        ~TString();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        tCIDLib::TBoolean operator==
        (
            const   TString&                strToCompare
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TString&                strToComprae
        )   const;

        tCIDLib::TBoolean operator==
        (
            const   tCIDLib::Tch* const     pszToCompare
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   tCIDLib::Tch* const     pszToCompare
        )   const;

        tCIDLib::TBoolean operator==
        (
            const   tCIDLib::Tsch* const    pszToCompare
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   tCIDLib::Tsch* const    pszToCompare
        )   const;

        tCIDLib::Tch& operator[]
        (
            const   tCIDLib::TCard4         c4Ind
        );

        const tCIDLib::Tch operator[]
        (
            const   tCIDLib::TCard4         c4Ind
        )   const;

        TString& operator=
        (
            const   tCIDLib::Tch* const     pszToAssign
        );

        TString& operator=
        (
            const   tCIDLib::Tsch* const    pszToAssign
        );

        TString& operator=
        (
            const   TString&                strToAssign
        );


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Append
        (
            const   TString&                strToAppend
        );

        tCIDLib::TVoid Append
        (
            const   TString&                strToAppend1
            , const TString&                strToAppend2
        );

        tCIDLib::TVoid Append
        (
            const   tCIDLib::Tch* const     pszToAppend
        );

        tCIDLib::TVoid Append
        (
                    tCIDLib::Tch            chToAppend
        );

        tCIDLib::TVoid AppendSubStr
        (
            const   TString&                strToCopyFrom
            , const tCIDLib::TCard4         c4Start
            , const tCIDLib::TCard4         c4Len = kCIDLib::c4MaxCard
        );

        tCIDLib::TBoolean bErrOnOverflow() const;

        tCIDLib::TBoolean bErrOnOverflow
        (
            const   tCIDLib::TBoolean       bNewState
        );

        tCIDLib::TBoolean bEmpty() const;

        tCIDLib::TBoolean bFull() const;

        tCIDLib::TBoolean bFirstOccurrence
        (
            const   tCIDLib::Tch            chToFind
            ,       tCIDLib::TCard4&        c4Pos
        )   const;

        tCIDLib::TBoolean bFirstOccurrence
        (
            const   TString&                strSubStrToFind
            ,       tCIDLib::TCard4&        c4Pos
            , const tCIDLib::TBoolean       bAnyChar = kCIDLib::False
        )   const;

        tCIDLib::TBoolean bIsFullyAllocated() const;

        tCIDLib::TBoolean bLastOccurrence
        (
            const   tCIDLib::Tch            chToFind
            ,       tCIDLib::TCard4&        c4Pos
        )   const;

        tCIDLib::TBoolean bLastOccurrence
        (
            const   TString&                strSubStrToFind
            ,       tCIDLib::TCard4&        c4Pos
            , const tCIDLib::TBoolean       bAnyChar = kCIDLib::False
        )   const;

        tCIDLib::TBoolean bNextOccurrence
        (
            const   tCIDLib::Tch            chToFind
            ,       tCIDLib::TCard4&        c4StartPos
        )   const;

        tCIDLib::TBoolean bNextOccurrence
        (
            const   TString&                strSubStrToFind
            ,       tCIDLib::TCard4&        c4StartPos
            , const tCIDLib::TBoolean       bAnyChar = kCIDLib::False
        )   const;

        tCIDLib::TBoolean bPrevOccurrence
        (
            const   tCIDLib::Tch            chToFind
            ,       tCIDLib::TCard4&        c4StartPos
        )   const;

        tCIDLib::TBoolean bPrevOccurrence
        (
            const   TString&                strSubStrToFind
            ,       tCIDLib::TCard4&        c4StartPos
            , const tCIDLib::TBoolean       bAnyChar = kCIDLib::False
        )   const;

        tCIDLib::TBoolean bTokenExists
        (
             const  tCIDLib::Tch            chToken
        );

        tCIDLib::TBoolean bSameText
        (
            const   TString&                strToCompare
            , const tCIDLib::TBoolean       bCaseSensitive = kCIDLib::True
        )   const;

        tCIDLib::TBoolean bSameText
        (
            const   tCIDLib::Tch* const     pszToCompare
            , const tCIDLib::TBoolean       bCaseSensitive = kCIDLib::True
        )   const;

        tCIDLib::TBoolean bSameText
        (
            const   tCIDLib::Tsch* const    pszToCompare
            , const tCIDLib::TBoolean       bCaseSensitive = kCIDLib::True
        )   const;

        tCIDLib::TBoolean bStripToken
        (
                    TString&                strResult
            , const tCIDLib::TCard4         c4TokenInd
            , const TString&                strSeparators
        )   const;

        tCIDLib::TCard4 c4BufChars() const;

        tCIDLib::TCard4 c4CopyWhileIn
        (
                    TString&                strResult
            , const TString&                strCharList
            , const tCIDLib::TCard4         cStartInd = 0
        )   const;

        tCIDLib::TCard4 c4CopyWhileOut
        (
                    TString&                strResult
            , const TString&                strCharList
            , const tCIDLib::TCard4         cStartInd   = 0
        )   const;

        tCIDLib::TCard4 c4Length() const;

        tCIDLib::TCard4 c4MaxChars() const;

        tCIDLib::TCard4 c4Val
        (
            const   tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
            ,       tCIDLib::TBoolean&      bValid = NUL_TBoolean
        )   const;

        tCIDLib::Tch chFirst() const;

        tCIDLib::Tch chLast() const;

        tCIDLib::Tch chAt
        (
            const   tCIDLib::TCard4         c4Index
        )   const;

        tCIDLib::TVoid Cap() const;

        tCIDLib::TVoid CapAt
        (
            const   tCIDLib::TCard4         c4Index
        )   const;

        tCIDLib::TVoid Clear();

        tCIDLib::TVoid CopyInSubStr
        (
            const   TString&                strSource
            , const tCIDLib::TCard4         c4Start
            , const tCIDLib::TCard4         c4Len = kCIDLib::c4MaxCard
        );

        tCIDLib::TVoid CopyOutSubStr
        (
                    TString&                strTarget
            , const tCIDLib::TCard4         c4Start
            , const tCIDLib::TCard4         c4Len = kCIDLib::c4MaxCard
        )   const;

        tCIDLib::TVoid CopyOutSubStr
        (
                    tCIDLib::Tch* const     pszTarget
            , const tCIDLib::TCard4         c4MaxChars
            , const tCIDLib::TCard4         c4Start
            , const tCIDLib::TCard4         c4Len = kCIDLib::c4MaxCard
        )   const;

        tCIDLib::TVoid Cut
        (
            const   tCIDLib::TCard4         c4Start
            , const tCIDLib::TCard4         c4Len = kCIDLib::c4MaxCard
        );

        tCIDLib::TVoid DeleteLast();

        tCIDLib::ESortComps eCompare
        (
            const   tCIDLib::Tch* const     pszToCompare
        )   const;
        
        tCIDLib::ESortComps eCompare
        (
            const   TString&                strToCompare
        )   const;

        tCIDLib::ESortComps eCompareN
        (
            const   tCIDLib::Tch* const     pszToCompare
            , const tCIDLib::TCard4         c4MaxComp
        )   const;

        tCIDLib::ESortComps eCompareN
        (
            const   TString&                strToCompare
            , const tCIDLib::TCard4         c4MaxComp
        )   const;

        tCIDLib::ESortComps eICompare
        (
            const   tCIDLib::Tch* const     pszToCompare
        )   const;

        tCIDLib::ESortComps eICompare
        (
            const   TString&                strToCompare
        )   const;

        tCIDLib::ESortComps eICompareN
        (
            const   tCIDLib::Tch* const     pszToCompare
            , const tCIDLib::TCard4         c4MaxComp
        )   const;

        tCIDLib::ESortComps eICompareN
        (
            const   TString&                strToCompare
            , const tCIDLib::TCard4         c4MaxComp
        )   const;

        tCIDLib::TVoid FormatToFld
        (
            const   TString&                strToFormat
            , const tCIDLib::TCard4         c4FldWidth  = 0
            , const tCIDLib::EHJustify      eJustify = tCIDLib::EHJustify_Left
            , const tCIDLib::Tch            chFill = kCIDLib::chSpace
        );

        tCIDLib::TVoid FromZStr
        (
            const   tCIDLib::Tch* const     pszSource
            , const tCIDLib::TCard4         c4Count
        );

        tCIDLib::TFloat8 f8Val
        (
                    tCIDLib::TBoolean&      bValid = NUL_TBoolean
        )   const;

        tCIDLib::THashVal hshCalcHash
        (
            const   tCIDLib::TCard4         c4Modulus
        )   const;

        tCIDLib::TInt4 i4Val
        (
            const   tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
            ,       tCIDLib::TBoolean&      bValid = NUL_TBoolean
        )   const;

        tCIDLib::TInt8 i8Val
        (
            const   tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
            ,       tCIDLib::TBoolean&      bValid = NUL_TBoolean
        )   const;

        tCIDLib::TVoid Insert
        (
            const   TString&                strToInsert
            , const tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid Insert
        (
            const   tCIDLib::Tch* const     pszToInsert
            , const tCIDLib::TCard4         c4Index
        );

        tCIDLib::TVoid LoadFromMsg
        (
            const   tCIDLib::TMsgId         midToLoad
            , const TFacility&              facSource
        );

        tCIDLib::TVoid LoadFromMsg
        (
            const   tCIDLib::TMsgId         midToLoad
            , const TFacility&              facSource
            , const MFormattable&           fmtblToken1
            , const MFormattable&           fmtblToken2 = NUL_MFormattable
            , const MFormattable&           fmtblToken3 = NUL_MFormattable
            , const MFormattable&           fmtblToken4 = NUL_MFormattable
        );

        const tCIDLib::Tch* pszData() const;

        tCIDLib::Tch* pszData();

        tCIDLib::TVoid Prepend
        (
            const   TString&                strToPrepend
        );

        tCIDLib::TVoid Prepend
        (
            const   tCIDLib::Tch* const     pszToPrepend
        );

        tCIDLib::TVoid Prepend
        (
            const   tCIDLib::Tch            chToPrepend
        );

        tCIDLib::TVoid PutAt
        (
            const   tCIDLib::TCard4         c4Index
            , const tCIDLib::Tch            chToPut
        );

        tCIDLib::TVoid ReallocateTo
        (
            const   tCIDLib::TCard4         c4NewSize
            , const tCIDLib::TBoolean       bPreserveContent = kCIDLib::True
        );

        tCIDLib::TVoid ReplaceToken
        (
            const   MFormattable&           fmtblVal
            , const tCIDLib::Tch            chToken
        );

        tCIDLib::TVoid ReplaceToken
        (
             const   tCIDLib::Tch* const    pszVal
            , const tCIDLib::Tch            chToken
        );

        tCIDLib::TVoid ReplaceToken
        (
            const   tCIDLib::TBoolean       bVal
            , const tCIDLib::Tch            chToken
        );

        tCIDLib::TVoid ReplaceToken
        (
            const   tCIDLib::TCard1         c1Val
            , const tCIDLib::Tch            chToken
            , const tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TVoid ReplaceToken
        (
            const   tCIDLib::TCard2         c2Val
            , const tCIDLib::Tch            chToken
            , const tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TVoid ReplaceToken
        (
            const   tCIDLib::TCard4         c4Val
            , const tCIDLib::Tch            chToken
            , const tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TVoid ReplaceToken
        (
            const   tCIDLib::TFloat8&       f8Val
            , const tCIDLib::Tch            chToken
            , const tCIDLib::TCard1         c1Precision = 2
        );

        tCIDLib::TVoid ReplaceToken
        (
            const   tCIDLib::TInt1          i1Val
            , const tCIDLib::Tch            chToken
            , const tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TVoid ReplaceToken
        (
            const   tCIDLib::TInt2          i2Val
            , const tCIDLib::Tch            chToken
            , const tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TVoid ReplaceToken
        (
            const   tCIDLib::TInt4          i4Val
            , const tCIDLib::Tch            chToken
            , const tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TVoid ReplaceToken
        (
            const   tCIDLib::TInt8&         i8Val
            , const tCIDLib::Tch            chToken
            , const tCIDLib::ERadices       eRadix = tCIDLib::ERadix_Dec
        );

        tCIDLib::TVoid ReplaceWith
        (
            const   tCIDLib::Tch            chToReplace
            , const tCIDLib::Tch            chReplaceWith
        );

        tCIDLib::TVoid SetLast
        (
            const   tCIDLib::Tch            chNew
        );

        tCIDLib::TVoid Strip
        (
            const   tCIDLib::Tch* const     pszChars
            , const tCIDLib::EStripModes    eStripmode = tCIDLib::EStripMode_LeadTrail
        );

        tCIDLib::TVoid Strip
        (
            const   TString&                strChars
            , const tCIDLib::EStripModes    eStripmode = tCIDLib::EStripMode_LeadTrail
        );

        tCIDLib::TVoid StripWhitespace
        (
            const   tCIDLib::EStripModes    eStripMode = tCIDLib::EStripMode_LeadTrail
        );

        tCIDLib::TVoid ToLower
        (
            const   tCIDLib::TCard4         c4StartInd  = 0
            , const tCIDLib::TCard4         c4Len = kCIDLib::c4MaxCard
        );

        tCIDLib::TVoid ToUpper
        (
            const   tCIDLib::TCard4         c4StartInd = 0
            , const tCIDLib::TCard4         c4Len = kCIDLib::c4MaxCard
        );

        tCIDLib::TVoid ToZStr
        (
                    tCIDLib::Tch* const     pszTarget
            , const tCIDLib::TCard4         c4MaxChars
            , const tCIDLib::TCard4         c4StartInd  = 0
        )   const;


    protected :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmDest
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private :
        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __Reallocate
        (
            const   tCIDLib::TCard4         c4NewSize
            , const tCIDLib::TBoolean       bPreserve = kCIDLib::True
        )   const;


        // --------------------------------------------------------------------
        //  Private data members.
        //
        //  __pszBuffer
        //      This is the buffer for this string.
        //
        //  __bErrOnOverflow
        //      Indicates whether this string treats an overflow of the
        //      buffer as an error, or just clips the operation so as to
        //      take as much as possible and no more. The latter is the
        //      default.
        //
        //  __c4BufChars
        //      This is the current number of chars the buffer can hold.
        //      It can grow up to the __c4MaxChars value.
        //
        //  __c4MaxChars
        //      This is the maximum chars that this string can grow up to.
        // --------------------------------------------------------------------
        tCIDLib::TBoolean       __bErrOnOverflow;
        tCIDLib::TCard4         __c4MaxChars;

        // These are mutable because of reallocation
        mutable tCIDLib::Tch*   __pszBuffer;
        mutable tCIDLib::TCard4 __c4BufChars;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TString,TObject)
        DefPolyDup(TString)
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  Global operators
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean operator<(const TString& str1, const TString& str2)
{
    return (TRawStr::eCompareStr(str1.pszData(), str2.pszData()) < 0);
}

inline tCIDLib::TBoolean operator>(const TString& str1, const TString& str2)
{
    return (TRawStr::eCompareStr(str1.pszData(), str2.pszData()) > 0);
}

inline TString operator+(const TString& str1, const TString& str2)
{
    return TString(str1.pszData(), str2.pszData());
}


// ----------------------------------------------------------------------------
//  Public operators
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TString::operator!=(const TString& strSrc) const
{
    return !operator==(strSrc);
}

inline tCIDLib::TBoolean
TString::operator!=(const tCIDLib::Tch* const pszSrc) const
{
    return !operator==(pszSrc);
}

inline tCIDLib::TBoolean
TString::operator!=(const tCIDLib::Tsch* const pszSrc) const
{
    return !operator==(pszSrc);
}


// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TVoid TString::Append(const TString& strSrc)
{
    Append(strSrc.__pszBuffer);
}

inline tCIDLib::TVoid
TString::Append(const TString&  strSrc1, const TString&  strSrc2)
{
    Append(strSrc1.__pszBuffer);
    Append(strSrc2.__pszBuffer);
}

inline tCIDLib::TBoolean TString::bEmpty() const
{
    if (__pszBuffer[0])
        return kCIDLib::False;
    return kCIDLib::True;
}

inline tCIDLib::TBoolean TString::bFull() const
{
    if (c4Length() >= __c4MaxChars)
        return kCIDLib::True;
    
    return kCIDLib::False;
}

inline tCIDLib::TBoolean TString::bErrOnOverflow() const
{
    return __bErrOnOverflow;
}

inline tCIDLib::TBoolean
TString::bErrOnOverflow(const tCIDLib::TBoolean bNewState)
{
    __bErrOnOverflow = bNewState;
    return __bErrOnOverflow;
}

inline tCIDLib::TBoolean TString::bIsFullyAllocated() const
{
    if (__c4BufChars == __c4MaxChars)
        return kCIDLib::True;
    return kCIDLib::False;
}

inline tCIDLib::TCard4 TString::c4BufChars() const
{
    return __c4BufChars;
}

inline tCIDLib::TCard4 TString::c4Length() const
{
    __pszBuffer[__c4BufChars] = 0;
    return TRawStr::c4StrLen(__pszBuffer);
}

inline tCIDLib::TCard4 TString::c4MaxChars() const
{
    return __c4MaxChars;
}

inline tCIDLib::TVoid TString::Cap() const
{
    __pszBuffer[__c4BufChars] = 0;
}

inline tCIDLib::TVoid TString::Clear()
{
    __pszBuffer[0] = 0;
}

inline tCIDLib::TVoid
TString::CopyInSubStr(  const   TString&        strSource
                        , const tCIDLib::TCard4 c4Start
                        , const tCIDLib::TCard4 c4Len)
{
    Clear();
    AppendSubStr(strSource, c4Start, c4Len);
}

inline tCIDLib::TVoid
TString::CopyOutSubStr(         TString&        strSource
                        , const tCIDLib::TCard4 c4Start
                        , const tCIDLib::TCard4 c4Len) const
{
    strSource.Clear();
    strSource.AppendSubStr(*this, c4Start, c4Len);
}

inline tCIDLib::THashVal
TString::hshCalcHash(const tCIDLib::TCard4 c4Modulus) const
{
    return TRawStr::hshHashStr(__pszBuffer, c4Modulus);
}

inline tCIDLib::TVoid
TString::Insert(const TString& strInsert, const tCIDLib::TCard4 c4Ind)
{
    Insert(strInsert.__pszBuffer, c4Ind);
}

inline const tCIDLib::Tch* TString::pszData() const
{
    return __pszBuffer;
}

inline tCIDLib::Tch* TString::pszData()
{
    return __pszBuffer;
}

inline tCIDLib::TVoid TString::Prepend(const TString& strPrepend)
{
    Insert(strPrepend.__pszBuffer, 0);
}

inline tCIDLib::TVoid TString::Prepend(const tCIDLib::Tch* const pszPrepend)
{
    Insert(pszPrepend, 0);
}

inline tCIDLib::TVoid
TString::ReallocateTo(  const   tCIDLib::TCard4     c4NewSize
                        , const tCIDLib::TBoolean   bPreserveContent)
{
    __Reallocate(c4NewSize, bPreserveContent);
}

inline tCIDLib::TVoid
TString::Strip( const   TString&                strChars
                , const tCIDLib::EStripModes    eStripMode)
{
    Strip(strChars.__pszBuffer, eStripMode);
}
