//
// NAME: CIDLib_String.Cpp
//
// DESCRIPTION:
//
//  This module implements the standard string classes. These classes provide
//  all of the usual string manipulation services.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/16/92
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  Token replacement is an important component of the string class. A
//      token has the following potential formats:
//
//          %(X)        - X is the token character
//          %(X,W)      - W is the field width, negative means left justify
//          %(X,W,F)    - F is the fill character for the field
//


// -----------------------------------------------------------------------------
//  Includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"



// -----------------------------------------------------------------------------
//  Local, constant data
//
//  __c4ReallocBlock
//      The steps that we try to alloc strings in.
// -----------------------------------------------------------------------------
static const tCIDLib::TCard4    __c4ReallocBlock    = 64;

// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData2(TString,TObject)



// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __pszFindToken
//
// DESCRIPTION:
//
//  This method will find a token in the passed string, parse it out, and
//  give back the token information.
// ------------------------------------
//   INPUT: pszSource is the source string to search
//          chToken is the token to find.
//
//  OUTPUT: eJustify is filled in with the justification indicated in the
//              token. If none, then it will be right justification.
//          c4FldWidth is filled in with the field width indicated in the
//              token. If none, then it will be 0 to indicate none.
//          chFill is filled in with the fill character indicated in the
//              token. If none, then it will be a space.
//          c4TokenChars is filled in with the count of chars in the token,
//              which will allow the caller to cut the token out.
//
//  RETURN: A pointer to the % sign of the token found, or 0if not found.
//
static const tCIDLib::Tch*
__pszFindToken( const   tCIDLib::Tch* const pszSource
                , const tCIDLib::Tch        chToken
                ,       tCIDLib::EHJustify& eJustify
                ,       tCIDLib::TCard4&    c4FldWidth
                ,       tCIDLib::Tch&       chFill
                ,       tCIDLib::TCard4&    c4TokenChars)
{
    const tCIDLib::Tch* pszCur;
    const tCIDLib::Tch* pszClose;

    tCIDLib::Tch        chTokenType;
    tCIDLib::TInt4      i4WidthVal;
    tCIDLib::Tch*       pszTok;
    tCIDLib::Tch        szTmp[64];

    // Set up the returns with correct default values
    eJustify        = tCIDLib::EHJustify_Right;
    c4FldWidth      = 0;
    c4TokenChars    = 0;
    chFill          = kCIDLib::chSpace;

    //
    //  Do an initial search for a percent sign. After we find one, check the
    //  next character to see what we are looking for.
    //
    pszCur = TRawStr::pszFindChar(pszSource, L'%');
    while (pszCur != 0)
    {
        // See if the next char is the end of the string. If so, then no match
        if (*(pszCur+1) == 0)
            return 0;

        // If the next char is a (, then it is a token so check it out
        if (*(pszCur+1) == L'(')
        {
            // Search for the closing paren
            pszClose = TRawStr::pszFindChar(pszCur+1, L')');

            //
            //  There was no closing parm so we need to just search for the
            //  next %, because this was a false alarm.
            //
            if (!pszClose)
            {
                pszCur++;
                pszCur = TRawStr::pszFindChar(pszCur, L'%');
                continue;
            }

            //
            //  If the enclosed text is larger than the tmp buffer, then
            //  something is wrong.
            //
            if (pszClose-pszCur > sizeof(szTmp) / kCIDLib::c4CharBytes)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcStr_InvalidToken
                    , tCIDLib::ESev_Warning
                    , tCIDLib::EClass_BadParms
                );
                return 0;
            }

            // If it is empty, then can't be right
            if (pszClose-pszCur == 1)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcStr_InvalidToken
                    , tCIDLib::ESev_Warning
                    , tCIDLib::EClass_BadParms
                );
                return 0;
            }

            //
            //  We need to parse the token contents. So copy the substring of
            //  the token to a temp buffer.
            //
            TRawMem::CopyMemBuf
            (
                szTmp
                , pszCur
                , (pszClose-pszCur) * kCIDLib::c4CharBytes
            );
            szTmp[pszClose-pszCur] = 0;
            pszTok = TRawStr::pszStrTokenize(szTmp, L"(,)%");

            //
            //  If no token or the more than a character, then invalid.
            //  Otherwise, pull out the token as the token character.
            //
            if (!pszTok || (TRawStr::c4StrLen(pszTok) > 1))
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcStr_InvalidToken
                    , tCIDLib::ESev_Warning
                    , tCIDLib::EClass_BadParms
                );
                return 0;
            }
            chTokenType = pszTok[0];

            //
            //  See if the char is the token character. If not, then start
            //  looking for the next character.
            //
            if (chToken != chTokenType)
            {
                pszCur++;
                pszCur = TRawStr::pszFindChar(pszCur, L'%');
                continue;
            }

            // We know the token width now
            c4TokenChars = (pszClose-pszCur) + 1;

            //
            //  Look for another token. If so, then it will be the field
            //  width. It must be a valid integral value. If it is negative,
            //  that means left justification.
            //
            pszTok = TRawStr::pszStrTokenize(0, L"(,)%");
            if (pszTok)
            {
                tCIDLib::TBoolean  bValid;
                i4WidthVal = TRawStr::i4AsBinary
                (
                    pszTok
                    , tCIDLib::ERadix_Dec
                    , bValid
                );

                if (!bValid)
                {
                    facCIDLib.LogErr
                    (
                        __FILE__
                        , __LINE__
                        , kCIDErrs::errcStr_InvalidToken
                        , tCIDLib::ESev_Warning
                        , tCIDLib::EClass_BadParms
                    );
                    return 0;
                }

                if (i4WidthVal < 0)
                {
                    eJustify = tCIDLib::EHJustify_Left;
                    if (i4WidthVal < 0)
                        c4FldWidth = i4WidthVal * -1;
                    else
                        c4FldWidth = i4WidthVal;
                }
                 else
                {
                    c4FldWidth = i4WidthVal;
                }

                pszTok = TRawStr::pszStrTokenize(0, L"(,)%");
                if (pszTok)
                {
                    if (TRawStr::c4StrLen(pszTok) > 1)
                    {
                        facCIDLib.LogErr
                        (
                            __FILE__
                            , __LINE__
                            , kCIDErrs::errcStr_InvalidToken
                            , tCIDLib::ESev_Warning
                            , tCIDLib::EClass_BadParms
                        );
                        return 0;
                    }
                    chFill = pszTok[0];
                }
            }

            // Break out to return
            break;
        }

        // Look for another percent sign
        pszCur++;
        pszCur = TRawStr::pszFindChar(pszCur, L'%');
    }
    return pszCur;
}



// -----------------------------------------------------------------------------
//   CLASS: TString
//  PREFIX: str
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TString: Constructors and Destructors
// -----------------------------------------------------------------------------

TString::TString() :

    __pszBuffer(0)
    , __bErrOnOverflow(kCIDLib::False)
    , __c4BufChars(64)
    , __c4MaxChars(4095)
{
    __pszBuffer = new tCIDLib::Tch[__c4BufChars+1];
    __pszBuffer[0] = 0;
}

TString::TString(const tCIDLib::Tch chInitValue) :

    __pszBuffer(0)
    , __bErrOnOverflow(kCIDLib::False)
    , __c4BufChars(16)
    , __c4MaxChars(4095)
{
    __pszBuffer = new tCIDLib::Tch[__c4BufChars+1];
    __pszBuffer[0] = chInitValue;
    __pszBuffer[1] = 0;
}

TString::TString(const TString& strToCopy) :

    __pszBuffer(0)
    , __bErrOnOverflow(kCIDLib::False)
    , __c4BufChars(strToCopy.__c4BufChars)
    , __c4MaxChars(strToCopy.__c4MaxChars)
{
    #if CID_DEBUG_ON
    if (__c4BufChars > __c4MaxChars)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_InitSize
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TCardinal(__c4BufChars)
            , TCardinal(__c4MaxChars)
        );
    }
    #endif

    __pszBuffer = new tCIDLib::Tch[__c4BufChars+1];
    __pszBuffer[0] = kCIDLib::chNull;
    TRawStr::CopyStr(__pszBuffer, strToCopy.__pszBuffer, __c4BufChars);
}

// This one is for supporting efficient string concatentation via + operator
TString::TString(   const   tCIDLib::Tch* const pszInit1
                    , const tCIDLib::Tch* const pszInit2) :

    __pszBuffer(0)
    , __bErrOnOverflow(kCIDLib::False)
    , __c4BufChars(0)
    , __c4MaxChars(0)
{
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(pszInit1)
                            + TRawStr::c4StrLen(pszInit2);

    // Now set the buf to the combined length and the max a little larger
    __c4BufChars = c4Len;
    __c4MaxChars = c4Len + 32;

    // Allocate the buffer and terminate it
    __pszBuffer = new tCIDLib::Tch[__c4BufChars+1];
    __pszBuffer[0] = 0;

    // Now concatenate the two source strings
    TRawStr::CopyCatStr(__pszBuffer, __c4BufChars, pszInit1, pszInit2);
}

TString::TString(   const   tCIDLib::Tch* const pszInit
                    , const tCIDLib::TCard4     c4InitChars
                    , const tCIDLib::TCard4     c4MaxChars) :
    __pszBuffer(0)
    , __bErrOnOverflow(kCIDLib::False)
    , __c4BufChars(c4InitChars)
    , __c4MaxChars(c4MaxChars)
{
    // The initial size can never be > than the max
    if (__c4BufChars > __c4MaxChars)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_InitSize
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TCardinal(__c4BufChars)
            , TCardinal(__c4MaxChars)
        );
    }

    // Get the length of the initialization string
    tCIDLib::TCard4 c4SrcLen = TRawStr::c4StrLen(pszInit);

    //
    //  If the source length is longer than the initial len, then up it
    //  clipped to the max.
    //
    if (c4SrcLen > __c4BufChars)
        __c4BufChars = (c4SrcLen > __c4MaxChars) ? __c4MaxChars : c4SrcLen;

    //
    //  If passed init size is zero, then take the source length, clipped
    //  to the max size.
    //
    if (!__c4BufChars)
    {
        __c4BufChars = c4SrcLen;
        if (__c4BufChars > __c4MaxChars)
        {
            facCIDLib.LogErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcStr_Overflow
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_BadParms
                , TString("assignment")
                , TCardinal(__c4MaxChars)
                , TCardinal(c4SrcLen)
            );
            __c4BufChars = __c4MaxChars;
        }
    }

    // Allocate and copy over what we can of the init string
    __pszBuffer = new tCIDLib::Tch[__c4BufChars+1];
    __pszBuffer[0] = kCIDLib::chNull;
    TRawStr::CopyStr(__pszBuffer, pszInit, __c4BufChars);
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
}

TString::TString(   const   tCIDLib::Tsch* const    pszInit
                    , const tCIDLib::TCard4         c4InitChars
                    , const tCIDLib::TCard4         c4MaxChars) :
    __pszBuffer(0)
    , __bErrOnOverflow(kCIDLib::False)
    , __c4BufChars(c4InitChars)
    , __c4MaxChars(c4MaxChars)
{
    // The initial size can never be > than the max
    if (__c4BufChars > __c4MaxChars)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_InitSize
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
            , TCardinal(__c4BufChars)
            , TCardinal(__c4MaxChars)
        );
    }

    // Get the length of the initialization string
    tCIDLib::TCard4 c4SrcLen = TRawStr::c4StrLen(pszInit);

    //
    //  If the source length is longer than the initial len, then up it
    //  clipped to the max.
    //
    if (c4SrcLen > __c4BufChars)
        __c4BufChars = (c4SrcLen > __c4MaxChars) ? __c4MaxChars : c4SrcLen;

    //
    //  If passed init size is zero, then take the source length, clipped
    //  to the max size.
    //
    if (!__c4BufChars)
    {
        __c4BufChars = c4SrcLen;
        if (__c4BufChars > __c4MaxChars)
        {
            facCIDLib.LogErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcStr_Overflow
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_BadParms
                , TString(L"assignment")
                , TCardinal(__c4MaxChars)
                , TCardinal(c4SrcLen)
            );
            __c4BufChars = __c4MaxChars;
        }
    }

    __pszBuffer = new tCIDLib::Tch[__c4BufChars+1];
    __pszBuffer[0] = kCIDLib::chNull;
    TRawStr::pszConvert(pszInit, __pszBuffer, __c4BufChars);
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
}

TString::TString(   const   tCIDLib::TErrCode   errcToLoad
                    , const TFacility&          facSrc
                    , const tCIDLib::TCard4     c4Extra) :

    __pszBuffer(0)
    , __bErrOnOverflow(kCIDLib::False)
    , __c4BufChars(0)
    , __c4MaxChars(0)
{
    const tCIDLib::Tch* pszTmp = facSrc.pszLoadCIDMsg(errcToLoad);

    // We got it, so how allocate our buffer with extra bytes and copy it
    __c4BufChars = TRawStr::c4StrLen(pszTmp);
    __c4MaxChars = __c4BufChars + c4Extra;

    __pszBuffer = new tCIDLib::Tch[__c4BufChars+1];
    TRawStr::CopyStr(__pszBuffer, pszTmp, __c4BufChars);
}

TString::TString(   const  tCIDLib::TErrCode    errcToLoad
                    , const TFacility&          facSrc
                    , const tCIDLib::TCard4     c4Extra
                    , const MFormattable&       fmtblToken1
                    , const MFormattable&       fmtblToken2
                    , const MFormattable&       fmtblToken3
                    , const MFormattable&       fmtblToken4) :
    __pszBuffer(0)
    , __bErrOnOverflow(kCIDLib::False)
    , __c4BufChars(0)
    , __c4MaxChars(0)
{
    tCIDLib::TBoolean bLoaded;
    const tCIDLib::Tch* pszTmp = facSrc.pszLoadCIDMsg(errcToLoad, bLoaded);

    // We got it, so how allocate our buffer with extra bytes and copy it
    __c4BufChars = TRawStr::c4StrLen(pszTmp);
    __c4MaxChars = __c4BufChars + c4Extra;

    __pszBuffer = new tCIDLib::Tch[__c4BufChars+1];
    TRawStr::CopyStr(__pszBuffer, pszTmp, __c4BufChars);

    // If we loaded it ok, then do token replacement
    if (bLoaded)
    {
        if (&fmtblToken1)
            ReplaceToken(fmtblToken1, L'1');
        if (&fmtblToken2)
            ReplaceToken(fmtblToken2, L'2');
        if (&fmtblToken3)
            ReplaceToken(fmtblToken3, L'3');
        if (&fmtblToken4)
            ReplaceToken(fmtblToken4, L'4');
    }
}

TString::TString(   const   MFormattable&   fmtblInitValue
                    , const TStreamFmt&     strmfToUse) :

    __pszBuffer(0)
    , __bErrOnOverflow(kCIDLib::False)
    , __c4BufChars(64)
    , __c4MaxChars(8192)
{
    //
    //  Create the buffer with an initially reasonable size. Most things
    //  that get formatted this way will be moderately sized.
    //
    __pszBuffer = new tCIDLib::Tch[__c4BufChars+1];
    __pszBuffer[0] = 0;

    //
    //  Create a string base text stream that we can use to format the
    //  passed object into. We tell it to format into us, but not to adopt
    //  us. We pass on the stream format object. If its a nul reference,
    //  that is ok because the string stream will just then ignore it.
    //
    TStringStream strmTmp(this, strmfToUse, tCIDLib::EAdoptOpt_NoAdopt);

    // Now we just format the value into ourself
    strmTmp << fmtblInitValue;
}
 

TString::~TString()
{
    delete __pszBuffer;
    __pszBuffer = 0;
}


// -----------------------------------------------------------------------------
//  Public operators
// -----------------------------------------------------------------------------

tCIDLib::TBoolean
TString::operator==(const TString& strToTest) const
{
    if (__bErrOnOverflow != strToTest.__bErrOnOverflow)
        return kCIDLib::False;

    if (__c4MaxChars != strToTest.__c4MaxChars)
        return kCIDLib::False;

    if (TRawStr::eCompareStr(__pszBuffer, strToTest.__pszBuffer))
        return kCIDLib::False;

    return kCIDLib::True;
}

tCIDLib::TBoolean
TString::operator==(const tCIDLib::Tch* const pszToTest) const
{
    if (TRawStr::eCompareStr(__pszBuffer, pszToTest))
        return kCIDLib::False;

    return kCIDLib::True;
}

tCIDLib::TBoolean
TString::operator==(const tCIDLib::Tsch* const pszToTest) const
{
    // Convert the short string so we can convert it
    tCIDLib::Tch*   pszConverted = TRawStr::pszConvert(pszToTest);
    THeapJanitor    janConverted(pszConverted);

    if (TRawStr::eCompareStr(__pszBuffer, pszConverted))
        return kCIDLib::False;

    return kCIDLib::True;
}


tCIDLib::Tch& TString::operator[](const tCIDLib::TCard4 c4Ind)
{
    // If the index is out of range, then log a message
    if (c4Ind >= c4Length())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Ind)
            , TCardinal(c4Length()-1)
        );
    }
    return __pszBuffer[c4Ind];
}

const tCIDLib::Tch TString::operator[](const tCIDLib::TCard4 c4Ind) const
{
    if (c4Ind >= c4Length())
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Ind)
            , TCardinal(c4Length()-1)
        );
    }
    return __pszBuffer[c4Ind];
}


TString& TString::operator=(const tCIDLib::Tch* const pszSrc)
{
    // If the source string is the same address as our buffer, then return
    if (pszSrc == __pszBuffer)
        return *this;

    // If the source string is within our buffer, then an error
    if ((pszSrc > __pszBuffer) && (pszSrc < __pszBuffer+__c4BufChars))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_SourceInDest
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
        );
    }

    // Get the source length for efficiency
    tCIDLib::TCard4 c4SrcLen = TRawStr::c4StrLen(pszSrc);

    // If its bigger than our max, check for overflow error
    if (c4SrcLen > __c4MaxChars)
    {
        if (__bErrOnOverflow)
        {
            facCIDLib.LogErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcStr_Overflow
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_BadParms
                , TString(L"assignment")
                , TCardinal(__c4MaxChars)
                , TCardinal(c4SrcLen)
            );
        }

        c4SrcLen = __c4MaxChars;
    }

    //
    //  If its larger than our current buffer, then reallocate. The last parm
    //  says don't maintain current content
    //
    if (c4SrcLen > __c4BufChars)
        __Reallocate(c4SrcLen, kCIDLib::False);

    Clear();
    Append(pszSrc);
    return *this;
}

TString& TString::operator=(const tCIDLib::Tsch* const pszSrc)
{
    tCIDLib::Tch* pszTmp = TRawStr::pszConvert(pszSrc);
    THeapJanitor  janTmp(pszTmp);
    TString::operator=(pszTmp);
    return *this;
}

TString& TString::operator=(const TString& strSrc)
{
    // Make sure we are not assigning to ourself
    if (this == &strSrc)
        return *this;

    //
    //  If our current buffer is not as large as the new string text or
    //  larger than our new max. Otherwise we can just keep our current
    //  one.
    //
    if ((__c4BufChars < strSrc.c4Length())
    ||  (__c4BufChars > strSrc.__c4MaxChars))
    {
        __c4BufChars = strSrc.__c4BufChars;
       delete __pszBuffer;
       __pszBuffer = 0;
        __pszBuffer = new tCIDLib::Tch[__c4BufChars+1];
        __pszBuffer[0] = kCIDLib::chNull;
    }

    // Copy over the max chars and overflow flag
    __c4MaxChars = strSrc.__c4MaxChars;
    __bErrOnOverflow = strSrc.__bErrOnOverflow;

    // Copy over the source text
    TRawStr::CopyStr(__pszBuffer, strSrc.__pszBuffer, __c4BufChars);

    return *this;
}



// -----------------------------------------------------------------------------
//  Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TString::Append(tCIDLib::Tch chSrc)
{
    // Put it into a temporary asciiz string
    tCIDLib::Tch     pszTmp[2];
    pszTmp[0] = chSrc;
    pszTmp[1] = kCIDLib::chNull;

    // And append it
    Append(pszTmp);
}

tCIDLib::TVoid TString::Append(const tCIDLib::Tch* const pszSrc)
{
    if (!pszSrc)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_NullSrc
            , tCIDLib::ESev_Warning
            , tCIDLib::EClass_BadParms
        );
        return;
    }

    // Get the current buffer len and the source one
    tCIDLib::TCard4 c4SrcSize = TRawStr::c4StrLen(pszSrc);
    tCIDLib::TCard4 c4OurSize = TRawStr::c4StrLen(__pszBuffer);

    // If no source text, then nothing to do
    if (!c4SrcSize)
        return;

    tCIDLib::TCard4 c4Actual = c4SrcSize;
    if ((c4OurSize + c4SrcSize) > __c4MaxChars)
    {
        if (__bErrOnOverflow)
        {
            facCIDLib.LogErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcStr_Index
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_BadParms
                , TCardinal(__c4MaxChars)
                , TCardinal(c4OurSize + c4SrcSize)
            );
        }
        c4Actual = __c4MaxChars - c4OurSize;
    }

    if (c4OurSize + c4Actual > __c4BufChars)
        __Reallocate(c4OurSize + c4Actual);

    TRawStr::CatStr(__pszBuffer, pszSrc, __c4BufChars);
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
}


tCIDLib::TVoid
TString::AppendSubStr(  const   TString&        strSource
                        , const tCIDLib::TCard4 c4Start
                        , const tCIDLib::TCard4 c4Len)
{
    // Get the source length into a local to make it more efficient
    tCIDLib::TCard4  c4SrcCnt = strSource.c4Length();

    // Is the source string is empty, then just return cause nothing to do
    if (!c4SrcCnt)
        return;

    //
    //  If the len is c4MaxCard, then set to exactly the rest of the source
    //  string.
    //
    tCIDLib::TCard4 c4Actual = c4Len;
    if (c4Actual == kCIDLib::c4MaxCard) 
    {
        c4Actual = c4SrcCnt-c4Start;
    }
     else
    {
        if (c4Start+c4Len > c4SrcCnt)
        {
            facCIDLib.LogErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcStr_Index
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_BadParms
                , TCardinal(c4SrcCnt-1)
                , TCardinal(c4Start+c4Len)
            );
        }
    }

    // And get our length too, so we don't have to keep redoing it
    tCIDLib::TCard4 c4OurCnt = c4Length();

    // See if we need to do an overflow exception
    if (c4OurCnt + c4Actual > __c4MaxChars)
    {
        if (__bErrOnOverflow)
        {
            facCIDLib.LogErr
            (
                 __FILE__
                 , __LINE__
                 , kCIDErrs::errcStr_Overflow
                 , tCIDLib::ESev_ProcessFatal
                 , tCIDLib::EClass_BadParms
                 , TString(L"sub string append")
                 , TCardinal(__c4MaxChars)
                 , TCardinal(c4OurCnt + c4Actual)
            );
        }

        // Adjust the actual again
        c4Actual = __c4MaxChars - c4OurCnt;
    }

    // If we ended up with nothing to do, then return
    if (!c4Actual)
        return;

    //
    //  See if we need to reallocate. If so it will reallocate to the
    //  nearest next realloc increment thats equal to or greater than
    //  c4Actual. The default is to maintain the current text content
    //  in the new buffer.
    //
    if (c4OurCnt + c4Actual > __c4BufChars)
        __Reallocate(c4OurCnt + c4Actual);

    // Copy over the data
    TRawMem::CopyMemBuf
    (
        &__pszBuffer[c4OurCnt]
        , &(strSource.__pszBuffer[c4Start])
        , c4Actual * kCIDLib::c4CharBytes
    );

    // Cap it off again
    __pszBuffer[c4OurCnt + c4Actual] = kCIDLib::chNull;
}


tCIDLib::TBoolean
TString::bFirstOccurrence(  const   tCIDLib::Tch        chTarget
                            ,       tCIDLib::TCard4&    c4Pos) const
{
    // Cap off for insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    // See if there is a match
    const tCIDLib::Tch* pszMatch =
                        TRawStr::pszFindChar(__pszBuffer, chTarget);

    // If not, then get out
    if (!pszMatch)
    {
        c4Pos = kCIDLib::c4MaxCard;
        return kCIDLib::False;
    }

    // Calc the position
    c4Pos = tCIDLib::TCard4(pszMatch - __pszBuffer);

    return kCIDLib::True;
}

tCIDLib::TBoolean
TString::bFirstOccurrence(  const   TString&            strSubStr
                            ,       tCIDLib::TCard4&    c4Pos
                            , const tCIDLib::TBoolean   bAnyChar) const
{
    // Cap off for insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    // See if there is a match
    const tCIDLib::Tch*    pszMatch;
    if (bAnyChar)
        pszMatch = TRawStr::pszFindChars(__pszBuffer, strSubStr.__pszBuffer);
     else
        pszMatch = TRawStr::pszFindSubStr(__pszBuffer, strSubStr.__pszBuffer);

    // If not, then get out
    if (!pszMatch)
    {
        c4Pos = kCIDLib::c4MaxCard;
        return kCIDLib::False;
    }

    // Calc the position
    c4Pos = tCIDLib::TCard4(pszMatch - __pszBuffer);

    return kCIDLib::True;
}


tCIDLib::TBoolean
TString::bLastOccurrence(   const   tCIDLib::Tch        chTarget
                            ,       tCIDLib::TCard4&    c4Pos) const
{
    // Cap off for insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    // See if there is a match
    const tCIDLib::Tch* pszMatch =
                        TRawStr::pszFindLastChar(__pszBuffer, chTarget);

    // If not, then get out
    if (!pszMatch)
    {
        c4Pos = kCIDLib::c4MaxCard;
        return kCIDLib::False;
    }

    // Calc the position
    c4Pos = tCIDLib::TCard4(pszMatch - __pszBuffer);

    return kCIDLib::True;
}

tCIDLib::TBoolean
TString::bLastOccurrence(   const   TString&            strSubStr
                            ,       tCIDLib::TCard4&    c4Pos
                            , const tCIDLib::TBoolean   bAnyChar) const
{
    tCIDLib::TCard4 c4Index;

    if (!strSubStr.c4Length())
    {
        c4Pos = kCIDLib::c4MaxCard;
        return kCIDLib::False;
    }

    // Cap off for insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    // See if there is a match
    const tCIDLib::Tch* pszMatch;

    if (bAnyChar)
        pszMatch = TRawStr::pszFindChars(__pszBuffer, strSubStr.pszData());
     else
        pszMatch = TRawStr::pszFindSubStr(__pszBuffer, strSubStr.pszData());

    // If not, then get out
    if (!pszMatch)
    {
        c4Pos = kCIDLib::c4MaxCard;
        return kCIDLib::False;
    }

    // Save the current match possition
    const tCIDLib::Tch* pszLast = pszMatch;

    while (pszMatch)
    {
        c4Index = tCIDLib::TCard4(pszMatch-__pszBuffer);

        if (bAnyChar)
        {
            pszMatch = TRawStr::pszFindChars
            (
                &__pszBuffer[c4Index+1]
                , strSubStr.pszData()
            );
        }
         else
        {
            pszMatch = TRawStr::pszFindSubStr
            (
                &__pszBuffer[c4Index+1]
                , strSubStr.pszData()
            );
        }

        if (pszMatch)
            pszLast = pszMatch;
    }

    // Calc the position of the last match
    c4Pos = tCIDLib::TCard4(pszLast - __pszBuffer);

    return kCIDLib::True;
}


tCIDLib::TBoolean
TString::bNextOccurrence(   const   tCIDLib::Tch        chTarget
                            ,       tCIDLib::TCard4&    c4Pos) const
{
    // Get the current size
    tCIDLib::TCard4 c4CurLen = TRawStr::c4StrLen(__pszBuffer);

    // If at the end then nothing to do
    if (c4Pos == c4CurLen)
    {
        c4Pos = kCIDLib::c4MaxCard;
        return kCIDLib::False;
    }

    // If over the length, then an error
    if (c4Pos >= c4CurLen)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Pos)
            , TCardinal(c4CurLen)
        );
    }

    // Cap off for insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    // See if there is a match
    const tCIDLib::Tch* pszMatch = TRawStr::pszFindChar
    (
        &__pszBuffer[c4Pos+1]
        , chTarget
    );

    // If not, then get out
    if (!pszMatch)
    {
        c4Pos = kCIDLib::c4MaxCard;
        return kCIDLib::False;
    }

    // Calc the position
    c4Pos = tCIDLib::TCard4(pszMatch - __pszBuffer);
    return kCIDLib::True;
}

tCIDLib::TBoolean
TString::bNextOccurrence(   const   TString&            strSubStr
                            ,       tCIDLib::TCard4&    c4Pos
                            , const tCIDLib::TBoolean   bAnyChar) const
{
    // If no substring, then obviously cant find it
    if (!strSubStr.c4Length())
    {
        c4Pos = kCIDLib::c4MaxCard;
        return kCIDLib::False;
    }

    // Cap off for insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    // Get the length of the string
    tCIDLib::TCard4 c4CurLen = TRawStr::c4StrLen(__pszBuffer);

    // If at the end, then we are done
    if (c4CurLen == c4Pos)
    {
        c4Pos = kCIDLib::c4MaxCard;
        return kCIDLib::False;
    }

    // If beyond the end, then an error
    if (c4Pos >= c4CurLen)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Pos)
            , TCardinal(c4CurLen)
        );
    }

    // See if there is a match
    const tCIDLib::Tch*    pszMatch;

    if (bAnyChar)
    {
        pszMatch = TRawStr::pszFindChars
        (
            &__pszBuffer[c4Pos+1]
            , strSubStr.pszData()
        );
    }
     else
    {
        pszMatch = TRawStr::pszFindSubStr
        (
            &__pszBuffer[c4Pos+1]
            , strSubStr.pszData()
        );
    }
    const tCIDLib::Tch* pszLast    = pszMatch;

    // If not, then get out
    if (!pszMatch)
    {
        c4Pos = kCIDLib::c4MaxCard;
        return kCIDLib::False;
    }

    // Calc the position of the last match
    c4Pos = tCIDLib::TCard4(pszLast - __pszBuffer);
    return kCIDLib::True;
}


tCIDLib::TBoolean
TString::bPrevOccurrence(   const   tCIDLib::Tch        chTarget
                            ,       tCIDLib::TCard4&    c4Pos) const
{
    // If the position is at 0, then nothing to do
    if (c4Pos == 0UL)
        return kCIDLib::False;

    // Cap off for insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    // Get our current length
    tCIDLib::TCard4 c4CurLen = TRawStr::c4StrLen(__pszBuffer);

    // If starting point is beyond the string length, then an error
    if (c4Pos > c4CurLen)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Pos)
            , TCardinal(c4CurLen)
        );
    }

    //
    //  See if there is a match. We just search backwards in the string to
    //  the previous match or 0 index
    //
    c4Pos--;
    while (__pszBuffer[c4Pos] != chTarget)
    {
        if (!c4Pos)
        {
            c4Pos = kCIDLib::c4MaxCard;
            return kCIDLib::False;
        }
        c4Pos--;
    }
    return kCIDLib::True;
}

tCIDLib::TBoolean
TString::bPrevOccurrence(   const   TString&            strSubStr
                            ,       tCIDLib::TCard4&    c4Pos
                            , const tCIDLib::TBoolean   bAnyChar) const
{
    tCIDLib::TCard4 c4SubLen = strSubStr.c4Length();
    if (!c4SubLen)
    {
        c4Pos = kCIDLib::c4MaxCard;
        return kCIDLib::False;
    }

    // Cap off for insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    // Get our string length
    tCIDLib::TCard4 c4CurLen = TRawStr::c4StrLen(__pszBuffer);

    // If beyond the string end, then an error
    if (c4Pos > c4CurLen)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Pos)
            , TCardinal(c4CurLen)
        );
    }

    // If the substring len is > than our len, then cannot match
    if (c4SubLen > c4CurLen)
        return kCIDLib::False;

    // Cap off for insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    //
    //  If we are looking for any character, then we can do a simpler
    //  search. If we are looking for the previous substring, then we
    //  want to do a smarter search.
    //    
    if (bAnyChar)
    {
        //
        //  All we have to do is find the the previous char that shows up
        //  in the substring anywhere.
        //
        c4Pos--;
        while (!TRawStr::pszFindChar(strSubStr.__pszBuffer, __pszBuffer[c4Pos]))
        {
            if (!c4Pos)
            {
                c4Pos = kCIDLib::c4MaxCard;
                return kCIDLib::False;
            }
            c4Pos--;
        }
    }
     else
    {
        //
        //  First of all, see if the start position plus the length of
        //  the substring is beyond the end of the string. If so, then we
        //  can back down the starting position right off the bat.
        //
        if (c4Pos + c4SubLen > c4CurLen)
        {
            // We know this will not go negative because we checked above
            c4Pos = c4CurLen - c4SubLen;
        }

        // Get the last first and last char of the substring
        tCIDLib::Tch chFirst    = strSubStr.__pszBuffer[0];
        tCIDLib::Tch chLast     = strSubStr.__pszBuffer[c4SubLen-1];

        //
        //  Now start the search. We check the first char and the last
        //  char. If that matches, then we do a full compare.
        //
        while (1)
        {
            if ((chFirst == __pszBuffer[c4Pos])
            &&  (chLast == __pszBuffer[c4Pos + (c4SubLen-1)]))
            {
                // Compare the whole substring
                if (!TRawStr::eNCompareStr
                (
                    &__pszBuffer[c4Pos]
                    , strSubStr.__pszBuffer
                    , c4SubLen))
                {
                    break;
                }
            }

            // If we hit bottom, then no match. Else, bump down and go again
            if (!c4Pos)
            {
                c4Pos = kCIDLib::c4MaxCard;
                return kCIDLib::False;
            }
            c4Pos--;
        }
    }
    return kCIDLib::True;
}


tCIDLib::TBoolean TString::bTokenExists(const tCIDLib::Tch chToken)
{
    //
    //  We don't care about these, but the searching function returns
    //  them, so...
    //
    tCIDLib::EHJustify  eJustify;
    tCIDLib::TCard4     c4Width, c4Chars;
    tCIDLib::Tch        chFill;

    // And look for the token.
    if (__pszFindToken
    (
        __pszBuffer
        , chToken
        , eJustify
        , c4Width
        , chFill
        , c4Chars))
    {
        return kCIDLib::True;
    }
    return kCIDLib::False;
}


tCIDLib::TBoolean
TString::bSameText( const   tCIDLib::Tch* const pszToCompare
                    , const tCIDLib::TBoolean   bCase) const
{
    if (bCase)
    {
        if (!TRawStr::eCompareStr(pszToCompare, __pszBuffer))
            return kCIDLib::True;
    }
     else
    {
        if (!TRawStr::eICompareStr(pszToCompare, __pszBuffer))
            return kCIDLib::True;
    }
    return kCIDLib::False;
}

tCIDLib::TBoolean
TString::bSameText( const   tCIDLib::Tsch* const    pszToCompare
                    , const tCIDLib::TBoolean       bCase) const
{
    tCIDLib::Tch*   pszTmp = TRawStr::pszConvert(pszToCompare);
    THeapJanitor    janTmp(pszTmp);
    return bSameText(pszTmp, bCase);
}

tCIDLib::TBoolean
TString::bSameText( const   TString&            strToCompare
                    , const tCIDLib::TBoolean   bCase) const
{
    if (bCase)
    {
        if (!TRawStr::eCompareStr(strToCompare.__pszBuffer, __pszBuffer))
            return kCIDLib::True;
    }
     else
    {
        if (!TRawStr::eICompareStr(strToCompare.__pszBuffer, __pszBuffer))
            return kCIDLib::True;
    }
    return kCIDLib::False;
}


tCIDLib::TBoolean
TString::bStripToken(           TString&        strResult
                        , const tCIDLib::TCard4 c4Index
                        , const TString&        strSeparators) const
{
    // Check for a very questionable index and issue a warning
    #if CID_DEBUG_ON
    if (c4Index > 1024)
    {
        facCIDLib.LogMsg
        (
            __FILE__
            , __LINE__
            , L"Token index of %(1) is a little questionable?"
            , tCIDLib::ESev_Warning
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Index)
        );
    }
    #endif

    //
    //  Allocate a temp asciiz string to use as target of strip and copy the
    //  current string value to it.
    //
    tCIDLib::Tch*   pszThis = new tCIDLib::Tch[__c4BufChars+1];
    THeapJanitor    janBuf(pszThis);
    TRawStr::CopyStr(pszThis, __pszBuffer, __c4BufChars);

    // Try to get the first token
    tCIDLib::Tch* pszTmp =
                TRawStr::pszStrTokenize(pszThis, strSeparators.pszData());

    if (pszTmp)
    {
        for (tCIDLib::TCard4 c4Tmp = 0; c4Tmp <= c4Index; c4Tmp++)
        {
            if (c4Tmp == c4Index)
            {
                strResult.Clear();
                strResult.Append(pszTmp);
                return kCIDLib::True;
            }

            pszTmp = TRawStr::pszStrTokenize(0, strSeparators.pszData());
            if (!pszTmp)
                break;
        }
    }
    return kCIDLib::False;
}


tCIDLib::TCard4 TString::c4Val( const   tCIDLib::ERadices   eRadix
                                ,       tCIDLib::TBoolean&  bValid) const
{
    tCIDLib::TBoolean bTmp;
    tCIDLib::TCard4 c4Ret = TRawStr::c4AsBinary(__pszBuffer, eRadix, bTmp);
    if (&bValid)
        bValid = bTmp;
    return c4Ret;
}


tCIDLib::TCard4
TString::c4CopyWhileIn(         TString&        strResult
                        , const TString&        strChars
                        , const tCIDLib::TCard4 c4StartInd) const
{
    // Clear the result string
    strResult.Clear();

    // If no chars in strChars then nothing to do.
    if (strChars.bEmpty())
        return 0;

    // Get our count of chars
    tCIDLib::TCard4  c4Tmp;
    tCIDLib::TCard4  c4SrcCnt = c4Length();

    // If no chars in this string, then we are done
    if (!c4SrcCnt)
        return 0;

    // If the start index is at the end, then we are done
    if (c4StartInd == c4SrcCnt-1)
        return 0;

    // If the start index is >= than the length of this string, then an error
    if (c4StartInd >= c4SrcCnt)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4StartInd)
            , TCardinal(c4SrcCnt-1)
        );
    }

    // Move up an index until we find a char not in the search chars
    for (c4Tmp = c4StartInd; c4Tmp < c4SrcCnt; c4Tmp++)
    {
        // Is this char in the list?
        if (!TRawStr::pszFindChar(strChars.pszData(), __pszBuffer[c4Tmp]))
            break;
    }

    // Copy the substring from c4StartInd, to c4Tmp
    strResult.CopyInSubStr(*this, c4StartInd, c4Tmp);

    return c4Tmp-c4StartInd;
}


tCIDLib::TCard4
TString::c4CopyWhileOut(        TString&        strResult
                        , const TString&        strChars
                        , const tCIDLib::TCard4 c4StartInd) const
{
    tCIDLib::TCard4  c4Tmp;
    tCIDLib::TCard4  c4SrcCnt = c4Length();

    // Clear the result string
    strResult.Clear();

    // If no chars in this string, then we are done
    if (!c4SrcCnt)
        return 0;

    // If the start index is at the end, then we are done
    if (c4StartInd == c4SrcCnt-1)
        return 0;

    // If the start index is > than the length of this string, then an error
    if (c4StartInd > c4SrcCnt-1)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4StartInd)
            , TCardinal(c4SrcCnt-1)
        );
    }

    for (c4Tmp = c4StartInd; c4Tmp < c4SrcCnt; c4Tmp++)
    {
        // Is this char in the list?
        if (TRawStr::pszFindChar(strChars.pszData(), __pszBuffer[c4Tmp]))
            break;
    }

    // Copy the substring from c4StartInd, to c4Tmp
    strResult.CopyInSubStr(*this, c4StartInd, c4Tmp);

    return c4Tmp-c4StartInd;
}


tCIDLib::TVoid TString::CapAt(const tCIDLib::TCard4 c4Index) const
{
    if (c4Index > __c4BufChars)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Index)
            , TCardinal(__c4BufChars)
        );
    }

    __pszBuffer[c4Index] = kCIDLib::chNull;
}


tCIDLib::Tch TString::chAt(const tCIDLib::TCard4 c4Ind) const
{
    // If the index is out of range, then log a message
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
    tCIDLib::TCard4 c4CurLen = TRawStr::c4StrLen(__pszBuffer);
    if (c4Ind >= c4CurLen)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Ind)
            , TCardinal(c4CurLen)
        );
    }
    return __pszBuffer[c4Ind];
}


tCIDLib::Tch TString::chFirst() const
{
    return __pszBuffer[0];
}

tCIDLib::Tch TString::chLast() const
{
    if (!__pszBuffer[0])
        return tCIDLib::Tch(0);

    return __pszBuffer[TRawStr::c4StrLen(__pszBuffer)-1];
}


tCIDLib::TVoid
TString::CopyOutSubStr(         tCIDLib::Tch* const pszTarget
                        , const tCIDLib::TCard4     c4MaxChars
                        , const tCIDLib::TCard4     c4Start
                        , const tCIDLib::TCard4     c4Len) const
{
    // Init the return string
    pszTarget[0] = kCIDLib::chNull;

    // If no length to copy, we are done
    if (!c4Len)
        return;

    // Get the chars in this string
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
    tCIDLib::TCard4 c4CurLen = TRawStr::c4StrLen(__pszBuffer);

    //
    //  If the start index plus the length is > than the length of this
    //  string, then an error cause we cannot read the requested lenngth.
    //
    if (c4Start+c4Len >= c4CurLen)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Start+c4Len)
            , TCardinal(c4CurLen)
        );
    }

    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Len; c4Index++)
    {
        // Don't copy over more chars than the target can take
        if (c4Index == c4MaxChars)
            break;

        pszTarget[c4Index] = __pszBuffer[c4Start+c4Index];
    }

    // Cap off the target now.
    pszTarget[c4Index] = kCIDLib::chNull;
}


tCIDLib::TVoid
TString::Cut(const tCIDLib::TCard4 c4Start, const tCIDLib::TCard4 c4Len)
{
    if (!c4Len)
        return;

    // Get the current length
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
    tCIDLib::TCard4  c4CurLen = TRawStr::c4StrLen(__pszBuffer);

    // We cannot start beyond the end
    if (c4Start > c4CurLen)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Start)
            , TCardinal(c4CurLen)
        );
    }

    tCIDLib::TCard4 c4ActualLen = c4Len;
    if ((c4ActualLen == kCIDLib::c4MaxCard)
    ||  (c4Start + c4ActualLen == c4CurLen))
    {
        __pszBuffer[c4Start] = kCIDLib::chNull;
        return;
    }
     else if (c4Start + c4ActualLen > c4CurLen)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Start + c4Len)
            , TCardinal(c4CurLen)
        );
    }

    // Copy the text at c4Start+c4Len down to c4Start
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
    TRawStr::CopyStr
    (
        &__pszBuffer[c4Start]
        , &__pszBuffer[c4Start+c4Len]
        , __c4BufChars
    );
}


tCIDLib::TVoid TString::DeleteLast()
{
    if (!__pszBuffer[0])
        return;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    // Just put a 0 in the last character
    __pszBuffer[TRawStr::c4StrLen(__pszBuffer)-1] = kCIDLib::chNull;
}


tCIDLib::ESortComps
TString::eCompare(const tCIDLib::Tch* const pszToCompare) const
{
    return TRawStr::eCompareStr(__pszBuffer, pszToCompare);
}

tCIDLib::ESortComps
TString::eCompare(const TString& strToCompare) const
{
    return TRawStr::eCompareStr(__pszBuffer, strToCompare.__pszBuffer);
}

tCIDLib::ESortComps
TString::eCompareN( const   tCIDLib::Tch* const pszToCompare
                    , const tCIDLib::TCard4     c4MaxComp) const
{
    if (!c4MaxComp)
        return tCIDLib::ESort_Equal;

    return TRawStr::eNCompareStr(__pszBuffer, pszToCompare, c4MaxComp);
}

tCIDLib::ESortComps
TString::eCompareN( const   TString&        strToCompare
                    , const tCIDLib::TCard4 c4MaxComp) const
{
    if (!c4MaxComp)
        return tCIDLib::ESort_Equal;

    return TRawStr::eNCompareStr
    (
        __pszBuffer
        , strToCompare.__pszBuffer
        , c4MaxComp
    );
}

tCIDLib::ESortComps
TString::eICompare(const tCIDLib::Tch* const pszToCompare) const
{
    return TRawStr::eICompareStr(__pszBuffer, pszToCompare);
}

tCIDLib::ESortComps
TString::eICompare(const TString& strToCompare) const
{
    return TRawStr::eICompareStr(__pszBuffer, strToCompare.__pszBuffer);
}

tCIDLib::ESortComps
TString::eICompareN(const   tCIDLib::Tch* const pszToCompare
                    , const tCIDLib::TCard4     c4MaxComp) const
{
    if (!c4MaxComp)
        return tCIDLib::ESort_Equal;
    return TRawStr::eNICompareStr(__pszBuffer, pszToCompare, c4MaxComp);
}

tCIDLib::ESortComps
TString::eICompareN(const   TString&        strToCompare
                    , const tCIDLib::TCard4 c4MaxComp) const
{
    if (!c4MaxComp)
        return tCIDLib::ESort_Equal;

    return TRawStr::eNICompareStr
    (
        __pszBuffer
        , strToCompare.__pszBuffer
        , c4MaxComp
    );
}


tCIDLib::TVoid
TString::FormatToFld(   const   TString&            strSrc
                        , const tCIDLib::TCard4     c4FldWidth
                        , const tCIDLib::EHJustify  eJustify
                        , const tCIDLib::Tch        chFill)
{
    //
    //  Check for the psycho circumstance of a string with no length. Should
    //  not happen, but need to catch it so we don't crash.
    //
    if (!__c4BufChars)
    {
        #if  CID_DEBUG_ON
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_DestStr_0Size
            , tCIDLib::ESev_Warning
            , tCIDLib::EClass_BadParms
        );
        #endif
        return;
    }

    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
    strSrc.__pszBuffer[strSrc.__c4BufChars] = kCIDLib::chNull;

    // Clip the field width if needed to fit in this string
    tCIDLib::TCard4 c4ActualWidth = c4FldWidth;
    if (c4ActualWidth > __c4MaxChars)
        c4ActualWidth = __c4MaxChars;

    //
    //  See if we need to reallocate. If so, it will reallocate up to the
    //  next realloc block size that is >= the passed actual. The second
    //  parm says don't bother preserving the text content.
    //
    if (c4ActualWidth > __c4BufChars)
        __Reallocate(c4ActualWidth, kCIDLib::False);

    TRawStr::FormatStr
    (
        strSrc.pszData()
        , __pszBuffer
        , c4ActualWidth
        , chFill
        , eJustify
    );
}


tCIDLib::TVoid
TString::FromZStr(  const   tCIDLib::Tch* const pszSource
                    , const tCIDLib::TCard4     c4Count)
{
    tCIDLib::TCard4 c4Actual = c4Count;

    if (c4Actual > __c4MaxChars)
    {
        if (__bErrOnOverflow)
        {
            facCIDLib.LogErr
            (
                 __FILE__
                 , __LINE__
                 , kCIDErrs::errcStr_Overflow
                 , tCIDLib::ESev_ProcessFatal
                 , tCIDLib::EClass_BadParms
                 , TString(L"assignment")
                 , TCardinal(__c4MaxChars)
                 , TCardinal(c4Actual)
            );
        }
        c4Actual = __c4MaxChars;
    }

    // Reallocate if we need to, telling it not to preserve the content.
    if (c4Actual > __c4BufChars)
        __Reallocate(c4Actual, kCIDLib::False);

    tCIDLib::TCard4  c4TargetInd = kCIDLib::chNull;
    for (tCIDLib::TCard4 c4SrcInd = 0; c4SrcInd < c4Actual; c4SrcInd++)
        __pszBuffer[c4TargetInd++] = pszSource[c4SrcInd];

    __pszBuffer[c4TargetInd] = kCIDLib::chNull;
}


tCIDLib::TFloat8 TString::f8Val(tCIDLib::TBoolean& bValid) const
{
    tCIDLib::TBoolean bTmp;
    tCIDLib::TFloat8 f8Ret = TRawStr::f8AsBinary(__pszBuffer, bTmp);
    if (&bValid)
        bValid = bTmp;

    return f8Ret;
}


tCIDLib::TInt4 TString::i4Val(  const   tCIDLib::ERadices   eRadix
                                ,       tCIDLib::TBoolean&  bValid) const
{
    tCIDLib::TBoolean bTmp;
    tCIDLib::TInt4 i4Ret = TRawStr::i4AsBinary(__pszBuffer, eRadix, bTmp);
    if (&bValid)
        bValid = bTmp;
    return i4Ret;
}

tCIDLib::TInt8 TString::i8Val(  const   tCIDLib::ERadices   eRadix
                                ,       tCIDLib::TBoolean&  bValid) const
{
    tCIDLib::TBoolean bTmp;
    tCIDLib::TInt8 i8Ret = TRawStr::i8AsBinary(__pszBuffer, eRadix, bTmp);
    if (&bValid)
        bValid = bTmp;
    return i8Ret;
}


tCIDLib::TVoid TString::Insert( const   tCIDLib::Tch* const pszSrc
                                , const tCIDLib::TCard4     c4Ind)
{
    tCIDLib::TCard4  c4CurLen, c4InsertLen;

    // Get the current length and the insert length
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
    c4CurLen    = TRawStr::c4StrLen(__pszBuffer);
    c4InsertLen = TRawStr::c4StrLen(pszSrc);

    // Do a quick sanity check to see if we need to do anything
    if (!c4InsertLen)
        return;

    //
    //  See if this would overflow the string. If so, and this string
    //  considers that an error, then log an exception. If its ok with an
    //  overflow, then adjust the insert length.
    //
    if (c4CurLen + c4InsertLen > __c4MaxChars)
    {
        if (__bErrOnOverflow)
        {
            facCIDLib.LogErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcStr_Overflow
                , tCIDLib::ESev_Warning
                , tCIDLib::EClass_BadParms
                , TString(L"insert")
                , TCardinal(__c4MaxChars)
                , TCardinal(c4CurLen + c4InsertLen)
            );
        }
        c4InsertLen = __c4MaxChars - c4CurLen;
    }

    //
    //  Make sure that the insert index is valid. We know that the insert
    //  length now cannot overflow, so we only need to know whether the start
    //  index is within the current length.
    //
    if (c4Ind > c4CurLen)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Ind)
            , TCardinal(c4CurLen)
        );
    }

    if (c4CurLen + c4InsertLen > __c4BufChars)
        __Reallocate(c4CurLen + c4InsertLen);

    // If the insert point is at the end, then we just append
    if (c4Ind == c4CurLen)
    {
        Append(pszSrc);
    }
     else
    {
        // Allocate a temp buffer to build into
        tCIDLib::Tch*   pszBuf = new tCIDLib::Tch[__c4BufChars+1];
        THeapJanitor    janBuf(pszBuf);
        pszBuf[0] = kCIDLib::chNull;

        //
        //  First we get the part of the string below the index over to the
        //  temp string. If we are at index 0, then there is nothing to for
        //  this part.
        //
        if (c4Ind != 0)
        {
            TRawStr::CopyStr(pszBuf, __pszBuffer, c4Ind);
            pszBuf[c4Ind] = kCIDLib::chNull;
        }

        // Cat the insert string text
        TRawStr::CatStr(pszBuf, pszSrc, __c4BufChars);
        pszBuf[__c4BufChars] = kCIDLib::chNull;

        // If there is any space left copy what we can of the remaining text
        if (TRawStr::c4StrLen(pszBuf) < __c4BufChars)
        {
            TRawStr::CatStr(pszBuf, &__pszBuffer[c4Ind], __c4BufChars);
            pszBuf[__c4BufChars] = kCIDLib::chNull;
        }

        // Copy the new text to the string object back to the string buffer
        __pszBuffer[0] = kCIDLib::chNull;
        TRawStr::CopyStr(__pszBuffer, pszBuf, __c4BufChars);
    }
}


tCIDLib::TVoid
TString::LoadFromMsg(   const   tCIDLib::TErrCode   errcToLoad
                        , const TFacility&          facSrc)
{
    const tCIDLib::Tch* pszTmp = facSrc.pszLoadCIDMsg(errcToLoad);

    // We got it, so how allocate our buffer with extra bytes and copy it
    tCIDLib::TCard4 c4NewChars = TRawStr::c4StrLen(pszTmp);

    if (c4NewChars > __c4MaxChars)
    {
        if (__bErrOnOverflow)
        {
            facCIDLib.LogErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcStr_Overflow
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_BadParms
                , TString(L"assignment")
                , TCardinal(__c4MaxChars)
                , TCardinal(c4NewChars)
            );
        }
        c4NewChars = __c4MaxChars;
    }
     else
    {
        //
        //  Allocate some extra bytes, since these strings often have
        //  replacement parameters.
        //
        c4NewChars += 128;
        if (c4NewChars > __c4MaxChars)
            c4NewChars = __c4MaxChars;
    }

    //
    //  If the new size is greater than the existing buffer size, then
    //  we have to reallocate. The default is to preserve existing text,
    //  so we pass false to make it throw it away.
    //
    if (c4NewChars > __c4BufChars)
        __Reallocate(c4NewChars, kCIDLib::False);

    // Now copy over the temp buffer
    TRawStr::CopyStr(__pszBuffer, pszTmp, __c4BufChars);
}

tCIDLib::TVoid
TString::LoadFromMsg(   const   tCIDLib::TErrCode   errcToLoad
                        , const TFacility&          facSrc
                        , const MFormattable&       fmtblToken1
                        , const MFormattable&       fmtblToken2
                        , const MFormattable&       fmtblToken3
                        , const MFormattable&       fmtblToken4)
{
    //
    //  Call the other version first. This will get the string allocated
    //  and we just have to handle the token replacement.
    //
    LoadFromMsg(errcToLoad, facSrc);

    // Handle the token replacement
    if (&fmtblToken1)
        ReplaceToken(fmtblToken1, L'1');
    if (&fmtblToken2)
        ReplaceToken(fmtblToken2, L'2');
    if (&fmtblToken3)
        ReplaceToken(fmtblToken3, L'3');
    if (&fmtblToken4)
        ReplaceToken(fmtblToken4, L'4');
}


tCIDLib::TVoid TString::Prepend(const tCIDLib::Tch chPrepend)
{
    // Put it into a temporary asciiz string
    tCIDLib::Tch pszTmp[2];
    pszTmp[0] = chPrepend;
    pszTmp[1] = kCIDLib::chNull;

    // And insert it at 0
    Insert(pszTmp, 0);
}


tCIDLib::TVoid TString::PutAt(  const   tCIDLib::TCard4 c4Index
                                , const tCIDLib::Tch    chToPut)
{
    //  Check for a put beyond the current length.
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(__pszBuffer);
    if (c4Index >= c4Len-1)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4Index)
            , TCardinal(c4Len-1)
        );
    }

    // Stick the character in and terminate the buffer size for safety
    __pszBuffer[c4Index] = chToPut;
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
}


tCIDLib::TVoid
TString::ReplaceToken(  const   MFormattable&   fmtblVal
                        , const tCIDLib::Tch    chToken)
{
    tCIDLib::Tch        chFill;
    tCIDLib::TCard4     c4Index, c4FldWidth, c4Chars;
    tCIDLib::EHJustify  eJustify;
    const tCIDLib::Tch* pszTmp;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    //  Find the first valid replacement token.
    if (!(pszTmp = __pszFindToken
    (
        __pszBuffer
        , chToken
        , eJustify
        , c4FldWidth
        , chFill
        , c4Chars)))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_TokenNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(chToken)
        );
    }

    // Calc the index where we found it
    c4Index = pszTmp-__pszBuffer;

    // Format the replacement value to a string
    TStringStream strmTmp
    (
        256
        , 8192
        , TStreamFmt(c4FldWidth, 2, eJustify, chFill)
    );
    strmTmp << fmtblVal;

    // Cut out the token from the string object and insert the new value
    Cut(c4Index, c4Chars);
    Insert(strmTmp.strData(), c4Index);
}

tCIDLib::TVoid
TString::ReplaceToken(  const   tCIDLib::Tch* const pszVal
                        , const tCIDLib::Tch    chToken)
{
    tCIDLib::Tch        chFill;
    tCIDLib::TCard4     c4Index, c4FldWidth, c4Chars;
    tCIDLib::EHJustify  eJustify;
    const tCIDLib::Tch* pszTmp;
    tCIDLib::Tch*       pszFmt;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    //  Find the first valid replacement token.
    if (!(pszTmp = __pszFindToken
    (
        __pszBuffer
        , chToken
        , eJustify
        , c4FldWidth
        , chFill
        , c4Chars)))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_TokenNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(chToken)
        );
    }

    // Calc the index where we found it
    c4Index = pszTmp-__pszBuffer;

    if (!c4FldWidth)
        c4FldWidth = TRawStr::c4StrLen(pszVal);

    pszFmt = new tCIDLib::Tch[c4FldWidth+1];
    THeapJanitor janFmt(pszFmt);
    TRawStr::FormatStr(pszVal, pszFmt, c4FldWidth, chFill, eJustify);

    // Cut out the token from the string object and insert the new value
    Cut(c4Index, c4Chars);
    Insert(pszFmt, c4Index);
}

tCIDLib::TVoid
TString::ReplaceToken(  const   tCIDLib::TBoolean   bVal
                        , const tCIDLib::Tch    chToken)
{
    tCIDLib::Tch        chFill;
    tCIDLib::TCard4     c4Index, c4FldWidth, c4Chars;
    tCIDLib::EHJustify  eJustify;
    const tCIDLib::Tch* pszTmp;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    //  Find the first valid replacement token.
    if (!(pszTmp = __pszFindToken
    (
        __pszBuffer
        , chToken
        , eJustify
        , c4FldWidth
        , chFill
        , c4Chars)))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_TokenNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(chToken)
        );
    }

    // Calc the index where we found it
    c4Index = pszTmp-__pszBuffer;

    // Cut out the token from the string object and insert the new value
    Cut(c4Index, c4Chars);

    if (bVal)
        Insert(L"True", c4Index);
    else
        Insert(L"False", c4Index);
}

tCIDLib::TVoid
TString::ReplaceToken(  const   tCIDLib::TCard1     c1Val
                        , const tCIDLib::Tch        chToken
                        , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::Tch        chFill;
    tCIDLib::TCard4     c4Index, c4FldWidth, c4Chars;
    tCIDLib::EHJustify  eJustify;
    const tCIDLib::Tch* pszTmp;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    //  Find the first valid replacement token.
    if (!(pszTmp = __pszFindToken
    (
        __pszBuffer
        , chToken
        , eJustify
        , c4FldWidth
        , chFill
        , c4Chars)))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_TokenNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(chToken)
        );
    }

    // Calc the index where we found it
    c4Index = pszTmp - __pszBuffer;

    tCIDLib::TZStr64    szTmp;
    TRawStr::FormatVal(c1Val, szTmp, 64, eRadix);

    if (!c4FldWidth)
        c4FldWidth = TRawStr::c4StrLen(szTmp);

    tCIDLib::Tch*   pszFmt = new tCIDLib::Tch[c4FldWidth+1];
    THeapJanitor    janFmt(pszFmt);
    TRawStr::FormatStr(szTmp, pszFmt, c4FldWidth, chFill, eJustify);

    // Cut out the token from the string object and insert the new value
    Cut(c4Index, c4Chars);
    Insert(pszFmt, c4Index);
}

tCIDLib::TVoid
TString::ReplaceToken(  const   tCIDLib::TCard2     c2Val
                        , const tCIDLib::Tch        chToken
                        , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::Tch        chFill;
    tCIDLib::TCard4     c4Index, c4FldWidth, c4Chars;
    tCIDLib::EHJustify  eJustify;
    const tCIDLib::Tch* pszTmp;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    //  Find the first valid replacement token.
    if (!(pszTmp = __pszFindToken
    (
        __pszBuffer
        , chToken
        , eJustify
        , c4FldWidth
        , chFill
        , c4Chars)))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_TokenNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(chToken)
        );
    }

    // Calc the index where we found it
    c4Index = pszTmp-__pszBuffer;

    tCIDLib::TZStr64    szTmp;
    TRawStr::FormatVal(c2Val, szTmp, 64, eRadix);

    if (!c4FldWidth)
        c4FldWidth = TRawStr::c4StrLen(szTmp);

    tCIDLib::Tch*   pszFmt = new tCIDLib::Tch[c4FldWidth+1];
    THeapJanitor    janFmt(pszFmt);
    TRawStr::FormatStr(szTmp, pszFmt, c4FldWidth, chFill, eJustify);

    // Cut out the token from the string object and insert the new value
    Cut(c4Index, c4Chars);
    Insert(pszFmt, c4Index);
}

tCIDLib::TVoid
TString::ReplaceToken(  const   tCIDLib::TCard4     c4Val
                        , const tCIDLib::Tch        chToken
                        , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::Tch        chFill;
    tCIDLib::TCard4     c4Index, c4FldWidth, c4Chars;
    tCIDLib::EHJustify  eJustify;
    const tCIDLib::Tch* pszTmp;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    //  Find the first valid replacement token.
    if (!(pszTmp = __pszFindToken
    (
        __pszBuffer
        , chToken
        , eJustify
        , c4FldWidth
        , chFill
        , c4Chars)))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_TokenNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(chToken)
        );
    }

    // Calc the index where we found it
    c4Index = pszTmp-__pszBuffer;

    tCIDLib::TZStr64    szTmp;
    TRawStr::FormatVal(c4Val, szTmp, 64, eRadix);

    if (!c4FldWidth)
        c4FldWidth = TRawStr::c4StrLen(szTmp);

    tCIDLib::Tch*   pszFmt = new tCIDLib::Tch[c4FldWidth+1];
    THeapJanitor    janFmt(pszFmt);
    TRawStr::FormatStr(szTmp, pszFmt, c4FldWidth, chFill, eJustify);

    // Cut out the token from the string object and insert the new value
    Cut(c4Index, c4Chars);
    Insert(pszFmt, c4Index);
}

tCIDLib::TVoid
TString::ReplaceToken(  const   tCIDLib::TFloat8&   f8Val
                        , const tCIDLib::Tch        chToken
                        , const tCIDLib::TCard1     c1Precision)
{
    tCIDLib::Tch        chFill;
    tCIDLib::TCard4     c4Index, c4FldWidth, c4Chars;
    tCIDLib::EHJustify  eJustify;
    const tCIDLib::Tch* pszTmp;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    //  Find the first valid replacement token.
    if (!(pszTmp = __pszFindToken
    (
        __pszBuffer
        , chToken
        , eJustify
        , c4FldWidth
        , chFill
        , c4Chars)))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_TokenNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(chToken)
        );
    }

    // Calc the index where we found it
    c4Index = pszTmp-__pszBuffer;

    tCIDLib::TZStr128   szTmp;
    TRawStr::FormatVal(f8Val, szTmp, c1Precision, 128, tCIDLib::ETrail_Zeroes);

    if (!c4FldWidth)
        c4FldWidth = TRawStr::c4StrLen(szTmp);

    tCIDLib::Tch*   pszFmt = new tCIDLib::Tch[c4FldWidth+1];
    THeapJanitor    janFmt(pszFmt);
    TRawStr::FormatStr(szTmp, pszFmt, c4FldWidth, chFill, eJustify);

    // Cut out the token from the string object and insert the new value
    Cut(c4Index, c4Chars);
    Insert(szTmp, c4Index);
}

tCIDLib::TVoid
TString::ReplaceToken(  const   tCIDLib::TInt1      i1Val
                        , const tCIDLib::Tch        chToken
                        , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::Tch        chFill;
    tCIDLib::TCard4     c4Index, c4FldWidth, c4Chars;
    tCIDLib::EHJustify  eJustify;
    const tCIDLib::Tch* pszTmp;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    //  Find the first valid replacement token.
    if (!(pszTmp = __pszFindToken
    (
        __pszBuffer
        , chToken
        , eJustify
        , c4FldWidth
        , chFill
        , c4Chars)))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_TokenNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(chToken)
        );
    }

    // Calc the index where we found it
    c4Index = pszTmp-__pszBuffer;

    tCIDLib::TZStr64    szTmp;
    TRawStr::FormatVal(i1Val, szTmp, 64, eRadix);

    if (!c4FldWidth)
        c4FldWidth = TRawStr::c4StrLen(szTmp);

    tCIDLib::Tch*   pszFmt = new tCIDLib::Tch[c4FldWidth+1];
    THeapJanitor    janFmt(pszFmt);
    TRawStr::FormatStr(szTmp, pszFmt, c4FldWidth, chFill, eJustify);

    // Cut out the token from the string object and insert the new value
    Cut(c4Index, c4Chars);
    Insert(pszFmt, c4Index);
}

tCIDLib::TVoid
TString::ReplaceToken(  const   tCIDLib::TInt2      i2Val
                        , const tCIDLib::Tch        chToken
                        , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::Tch        chFill;
    tCIDLib::TCard4     c4Index, c4FldWidth, c4Chars;
    tCIDLib::EHJustify  eJustify;
    const tCIDLib::Tch* pszTmp;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    //  Find the first valid replacement token.
    if (!(pszTmp = __pszFindToken
    (
        __pszBuffer
        , chToken
        , eJustify
        , c4FldWidth
        , chFill
        , c4Chars)))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_TokenNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(chToken)
        );
    }

    // Calc the index where we found it
    c4Index = pszTmp-__pszBuffer;

    tCIDLib::TZStr64 szTmp;
    TRawStr::FormatVal(i2Val, szTmp, 64, eRadix);

    if (!c4FldWidth)
        c4FldWidth = TRawStr::c4StrLen(szTmp);

    tCIDLib::Tch*   pszFmt = new tCIDLib::Tch[c4FldWidth+1];
    THeapJanitor    janFmt(pszFmt);
    TRawStr::FormatStr(szTmp, pszFmt, c4FldWidth, chFill, eJustify);

    // Cut out the token from the string object and insert the new value
    Cut(c4Index, c4Chars);
    Insert(pszFmt, c4Index);
}

tCIDLib::TVoid
TString::ReplaceToken(  const   tCIDLib::TInt4      i4Val
                        , const tCIDLib::Tch        chToken
                        , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::Tch        chFill;
    tCIDLib::TCard4     c4Index, c4FldWidth, c4Chars;
    tCIDLib::EHJustify  eJustify;
    const tCIDLib::Tch* pszTmp;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    //  Find the first valid replacement token.
    if (!(pszTmp = __pszFindToken
    (
        __pszBuffer
        , chToken
        , eJustify
        , c4FldWidth
        , chFill
        , c4Chars)))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_TokenNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(chToken)
        );
    }

    // Calc the index where we found it
    c4Index = pszTmp-__pszBuffer;

    tCIDLib::TZStr64    szTmp;
    TRawStr::FormatVal(i4Val, szTmp, 64, eRadix);

    if (!c4FldWidth)
        c4FldWidth = TRawStr::c4StrLen(szTmp);

    // Could get an exception here and leak the buffer
    tCIDLib::Tch*   pszFmt = new tCIDLib::Tch[c4FldWidth+1];
    THeapJanitor    janFmt(pszFmt);
    TRawStr::FormatStr(szTmp, pszFmt, c4FldWidth, chFill, eJustify);

    // Cut out the token from the string object and insert the new value
    Cut(c4Index, c4Chars);
    Insert(pszFmt, c4Index);
}

tCIDLib::TVoid
TString::ReplaceToken(  const   tCIDLib::TInt8&     i8Val
                        , const tCIDLib::Tch        chToken
                        , const tCIDLib::ERadices   eRadix)
{
    tCIDLib::Tch        chFill;
    tCIDLib::TCard4     c4Index, c4FldWidth, c4Chars;
    tCIDLib::EHJustify  eJustify;
    const tCIDLib::Tch* pszTmp;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    //  Find the first valid replacement token.
    if (!(pszTmp = __pszFindToken
    (
        __pszBuffer
        , chToken
        , eJustify
        , c4FldWidth
        , chFill
        , c4Chars)))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_TokenNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(chToken)
        );
    }

    // Calc the index where we found it
    c4Index = pszTmp-__pszBuffer;

    tCIDLib::TZStr128    szTmp;
    TRawStr::FormatVal(i8Val, szTmp, 128, eRadix);

    if (!c4FldWidth)
        c4FldWidth = TRawStr::c4StrLen(szTmp);

    // Could get an exception here and leak the buffer
    tCIDLib::Tch*   pszFmt = new tCIDLib::Tch[c4FldWidth+1];
    THeapJanitor    janFmt(pszFmt);
    TRawStr::FormatStr(szTmp, pszFmt, c4FldWidth, chFill, eJustify);

    // Cut out the token from the string object and insert the new value
    Cut(c4Index, c4Chars);
    Insert(pszFmt, c4Index);
}


tCIDLib::TVoid TString::ReplaceWith(const   tCIDLib::Tch    chToReplace
                                    , const tCIDLib::Tch    chReplaceWith)
{
    // To replace cannot be nul
    if (!chToReplace)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_CannotBeNul
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
        );
    }

    //
    //  Loop through the string and replace any instances of the replacement
    //  character with the replacement character.
    //
    tCIDLib::TCard4 c4CurLen = TRawStr::c4StrLen(__pszBuffer);
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4CurLen; c4Index++)
    {
        if (__pszBuffer[c4Index] == chToReplace)
            __pszBuffer[c4Index] = chReplaceWith;
    }
}


tCIDLib::TVoid TString::SetLast(const tCIDLib::Tch chNew)
{
    if (!__pszBuffer[0])
        return;

    // Insurance
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;

    // Just put the new char in the last character
    __pszBuffer[TRawStr::c4StrLen(__pszBuffer)-1] = chNew;
}


tCIDLib::TVoid
TString::Strip( const   tCIDLib::Tch* const     pszStripChars
                , const tCIDLib::EStripModes    eMode)
{
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
    TRawStr::StripStr(__pszBuffer, pszStripChars, eMode);
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
}


tCIDLib::TVoid
TString::StripWhitespace(const tCIDLib::EStripModes eStripMode)
{
    Strip(kCIDLib::szWhitespace, eStripMode);
}


tCIDLib::TVoid TString::ToLower(const   tCIDLib::TCard4 c4StartInd
                                , const tCIDLib::TCard4 c4Len)
{
    tCIDLib::TCard4 c4Actual;
    tCIDLib::TCard4 c4CurLen;

    // Get the current length
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
    c4CurLen = TRawStr::c4StrLen(__pszBuffer);

    //
    //  If the length is max card, then we want to just make it long enough
    //  to get to the end of the string.
    //
    c4Actual = c4Len;
    if (c4Actual == kCIDLib::c4MaxCard)
        c4Actual = c4CurLen - c4StartInd;

    // If the start plus the actual is too big, then an error
    if (c4StartInd + c4Actual > c4CurLen)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Overflow
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TString(L"lower case")
            , TCardinal(c4CurLen)
            , TCardinal(c4StartInd + c4Actual)
        );
    }

    // If the length would take us to the end of the string, then use strlwr()
    if (c4StartInd+c4Actual >= c4CurLen)
    {
        TRawStr::pszLowerCase(&__pszBuffer[c4StartInd]);
        return;
    }
     else
    {
        // Else, we have to use a loop
        tCIDLib::TCard4 c4Ind;
        for (c4Ind = c4StartInd; c4Ind < c4StartInd+c4Actual; c4Ind++)
            __pszBuffer[c4Ind] = TRawStr::chLower(__pszBuffer[c4Ind]);
    }

    // Cap it off for safety
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
}


tCIDLib::TVoid TString::ToUpper(const   tCIDLib::TCard4 c4StartInd
                                , const tCIDLib::TCard4 c4Len)
{
    tCIDLib::TCard4 c4Actual;
    tCIDLib::TCard4 c4CurLen;

    // Get the current length
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
    c4CurLen = TRawStr::c4StrLen(__pszBuffer);

    //
    //  If the length is max card, then we want to just make it long enough
    //  to get to the end of the string.
    //
    c4Actual = c4Len;
    if (c4Actual == kCIDLib::c4MaxCard)
        c4Actual = c4CurLen - c4StartInd;

    // If the start plus the actual is too big, then an error
    if (c4StartInd + c4Actual > c4CurLen)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Overflow
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TString(L"upper case")
            , TCardinal(c4CurLen)
            , TCardinal(c4StartInd + c4Actual)
        );
    }

    // If the length would take us to the end of the string, then use strupr()
    if (c4StartInd+c4Actual >= c4CurLen)
    {
        TRawStr::pszUpperCase(&__pszBuffer[c4StartInd]);
        return;
    }
     else
    {
        // Else, we have to use a loop
        tCIDLib::TCard4 c4Ind;
        for (c4Ind = c4StartInd; c4Ind < c4StartInd+c4Actual; c4Ind++)
            __pszBuffer[c4Ind] = TRawStr::chUpper(__pszBuffer[c4Ind]);
    }

    // Cap it off
    __pszBuffer[__c4BufChars] = kCIDLib::chNull;
}


tCIDLib::TVoid
TString::ToZStr(        tCIDLib::Tch* const pszTarget
                , const tCIDLib::TCard4     c4MaxChars
                , const tCIDLib::TCard4     c4StartInd) const
{
    // Clear the target zstring
    pszTarget[0] = kCIDLib::chNull;

    // Get our count of chars
    tCIDLib::TCard4  c4SrcCnt = c4Length();

    // If no chars in this string, then we are done
    if (!c4SrcCnt)
        return;

    // If start index is at end of string, then we are done
    if (c4StartInd == c4SrcCnt-1)
        return;

    // If the start index is > than the length of this string, then an error
    if (c4StartInd >= c4SrcCnt-1)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Index
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4StartInd)
            , TCardinal(c4SrcCnt-1)
        );
    }

    // Copy as many chars as will fit or are available, whichever is smaller
    TRawStr::CopyStr
    (
        pszTarget
        , &__pszBuffer[c4StartInd]
        , MinVal(c4MaxChars, c4SrcCnt-c4StartInd)
    );

    // Cap it off just in case
    pszTarget[c4MaxChars] = kCIDLib::chNull;
}


// -----------------------------------------------------------------------------
//  TString: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TString::_FormatTo(TTextStream& strmToWriteTo) const
{
    //
    //  Dump the asciiz string pointer to the string. DO NOT dump 'this'
    //  to the stream! This will kick off a recursive circle jerk.
    //
    strmToWriteTo << __pszBuffer;
}


tCIDLib::TVoid TString::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    //
    //  Read the stored stored max, buffer size, and actual chars stored to
    //  temporaries until we are sure they are happy.
    //
    tCIDLib::TCard4 c4NewMax;
    tCIDLib::TCard4 c4NewBufChars;
    tCIDLib::TCard4 c4CharsStored;
    strmToReadFrom >> c4NewMax;
    strmToReadFrom >> c4NewBufChars;
    strmToReadFrom >> c4CharsStored;

    if (c4NewBufChars > c4NewMax)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_SizeConflict
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4NewBufChars)
            , TCardinal(c4NewMax)
        );
    }

    if ((c4CharsStored > c4NewMax)
    ||  (c4CharsStored > c4NewBufChars))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_StoredBad
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4CharsStored)
            , TCardinal(c4NewBufChars)
            , TCardinal(c4NewMax)
        );
    }

    // Looks ok so store the new max
    __c4MaxChars = c4NewMax;

    //
    //  If the new buffer size is different from the stored one, then we
    //  have to reallocate. The second parm says  don't preserve current
    //  content. This will set __c4BufChars to the new buffer size.
    //
    if (c4NewBufChars != __c4BufChars)
        __Reallocate(c4NewBufChars, kCIDLib::False);

    //
    //  Ok, the caller agrees he can hold this much so stream in the
    //  new chars can cap off the string. Only do this if we actually
    //  stored any chars.
    //
    if (c4CharsStored)
        strmToReadFrom.ReadRawBuffer(__pszBuffer, c4CharsStored * kCIDLib::c4CharBytes);
    __pszBuffer[c4CharsStored] = kCIDLib::chNull;
}

tCIDLib::TVoid TString::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __c4MaxChars;
    strmToWriteTo << __c4BufChars;

    //
    //  Stream out the chars. We get the number of chars, and store that
    //  then store that many chars following it. So we are not storing the
    //  null, in order to save a byte.
    //
    tCIDLib::TCard4 c4Count = c4Length();
    strmToWriteTo << c4Count;
    strmToWriteTo.WriteRawBuffer(__pszBuffer, c4Count * kCIDLib::c4CharBytes);
}


// -----------------------------------------------------------------------------
//  TString: Private, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __Reallocate
//
// DESCRIPTION:
//
//  This method will reallocate the buffer to the new size, optionally
//  preserving the text content.
//
//  It affects only mutable members which are not counted in the equality
//  testing so its constant!
// ---------------------------------------
//   INPUT: c4NewSize is the new size to allocate the buffer to. It can
//              be smaller if not smaller than the current number of
//              actual chars (or if bPreserve is eFalse, in which case
//              it does not matter.)
//          bPreserve indicates whether the current contents is to be
//              preserved.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TString::__Reallocate(  const   tCIDLib::TCard4     c4NewSize
                        , const tCIDLib::TBoolean   bPreserve) const
{
    // If the same, then nothing to do
    if (c4NewSize == __c4BufChars)
        return;

    if (bPreserve && (c4NewSize < c4Length()))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Reallocate
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TString(L"the new size < current but bPreserve is true")
        );
    }

    if (c4NewSize > __c4MaxChars)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcStr_Reallocate
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TString(L"the new size is > the maximum string size")
        );
    }

    //
    //  Reallocate the buffer. If the new size is not at least one realloc
    //  block larger, then make it one realloc block larger anyway. Check
    //  though that we did not push it over the max again.
    //
    tCIDLib::TCard4 c4Actual = c4NewSize;

    if (__c4BufChars + __c4ReallocBlock > c4NewSize)
        c4Actual = __c4BufChars + __c4ReallocBlock;

    if (c4Actual > __c4MaxChars)
        c4Actual = __c4MaxChars;

    //
    //  Save the old buffer pointer and put a janitor on it so that it will
    //  get cleaned up. zero the current buffer so we give up control.
    //
    tCIDLib::Tch*   pszOldBuffer = __pszBuffer;
    THeapJanitor    janOld(pszOldBuffer);
    __pszBuffer = kCIDLib::chNull;

    // Now reallocate the new buffer and zero it out
    __pszBuffer = new tCIDLib::Tch[c4Actual+1];
    __pszBuffer[0] = kCIDLib::chNull;

    // If we are preserving, then copy the old content
    if (bPreserve)
        TRawStr::CopyStr(__pszBuffer, pszOldBuffer, c4Actual);

    // Save the new current buffer size
    __c4BufChars = c4Actual;
}
