//
// NAME: CIDLib_Streamable.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDLib_Streamable.Cpp module. This module
//  implements the MStreamable mixin class. This class is mixed in via
//  multiple inheritance to any class that needs to be streamed to or from
//  a binary stream (or derived) object, class TBinaryStream.
//
//  Also provided here are the global streaming operators. Any class that
//  implements the streamable interface gets use of these operators.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/25.96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//
//  1)  By providing the same RTTI public methods here as TObject, we
//      insure that we can use these same capabilities (that all CIDLib
//      objects have) when we are looking at them via the streamable mixin.
//


// -----------------------------------------------------------------------------
//  Forward references
// -----------------------------------------------------------------------------
class   TBinaryStream;
class   TClass;


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: MStreamable
// PREFIX: strmbl
// -----------------------------------------------------------------------------
class CIDLIBEXP MStreamable
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and Destructors
        // ---------------------------------------------------------------------
        MStreamable() {}

        ~MStreamable()  {}


        // ---------------------------------------------------------------------
        //  Public, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TBoolean bIsDescendantOf
        (
            const   TClass&                 clsTarget
        )   const = 0;

        virtual const TClass& clsIsA() const = 0;

        virtual const TClass& clsParent() const = 0;


    protected   :
        // ---------------------------------------------------------------------
        //  Declare our friends
        //
        //  The global methods below must have access to our protected stream
        //  methods.
        // ---------------------------------------------------------------------
        friend TBinaryStream& operator<<
        (
                    TBinaryStream&          strmToWriteTo
            , const MStreamable&            strmblToWrite
        );

        friend TBinaryStream& operator>>
        (
                    TBinaryStream&          strmToRead
            ,       MStreamable&            strmblToRead
        );


        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        )   = 0;

        virtual tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const = 0;


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        MStreamable(const MStreamable&);

        tCIDLib::TVoid operator=(const MStreamable&);
};

#pragma pack(pop)



// -----------------------------------------------------------------------------
//  Provide the global operators that allow any streamable object to be
//  read from or written to a binary stream. This is what the outside world
//  uses.
// -----------------------------------------------------------------------------
inline TBinaryStream&
operator<<(TBinaryStream& strmTarget, const MStreamable& strmblToWrite)
{
    strmblToWrite._StreamTo(strmTarget);
    return strmTarget;
}

inline TBinaryStream&
operator>>(TBinaryStream& strmSource, MStreamable& strmblToFill)
{
    strmblToFill._StreamFrom(strmSource);
    return strmSource;
}
