//
// NAME: CIDLib_StdOutLogger.Cpp
//
// DESCRIPTION:
//
//  This module implements a file logger, which sends logged errors to a
//  text output file.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/27/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData2(TStdOutLogger,TLogger)


// ----------------------------------------------------------------------------
//  TStdOutLogger: Constructors and operators
// ----------------------------------------------------------------------------

TStdOutLogger::TStdOutLogger() :

    __pmtxSync(0)
{
}

TStdOutLogger::TStdOutLogger(const TResourceName& rsnMutex) :

    __pmtxSync(0)
{
    __pmtxSync = new TMutex(rsnMutex, tCIDLib::ELockState_Unlocked);
}

TStdOutLogger::~TStdOutLogger()
{
    delete __pmtxSync;
}


// ----------------------------------------------------------------------------
//  TStdOutLogger: Public, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TStdOutLogger::LogMsg(const TError& errToLog)
{
    //
    //  Create some static stuff to use for logging. We know we can do this
    //  because calls to this method are serialized by the CIDLib code that
    //  calls it.
    //
    static tCIDLib::TBoolean    bFirstTime = kCIDLib::True;
    static TConsole             conOut;
    static TStringStream        strmLog(4000);
    static TTime                tmCurrent;

    if (bFirstTime)
    {
        tmCurrent.strDefaultFormat(L"%(M,2,0)/%(D,2,0)  %(R,2,0):%(h,2,0)");
        bFirstTime = kCIDLib::False;
    }

    // Set the local time object to now
    tmCurrent.SetToNow();

    // Clear the string before we start formatting into it again
    strmLog.Reset();

    //
    //  Build up the string that we are going to log. Its relatively
    //  complex and we have to use low level stuff here because of the
    //  low level.
    //
    //  We start by putting the date and time into the message, which
    //  is particularly important when looking back after the fact.
    //
    strmLog << L"(" << tmCurrent << L")  - "
            << TKrnlSysInfo::pszProcessName()
            << NewLn << L"     ("
            << errToLog.strModName()
            << L", " << TThread::pthrCaller()->strName()
            << L", " << errToLog.strFileName() << L"."
            << errToLog.c4LineNum()
            << L", " << errToLog.eSeverity()
            << L"/" << errToLog.eClass();

    // Only do the error stuff if there are any non-zero codes
    if (errToLog.bAnyError())
    {
        strmLog << L", Error:" << errToLog.errcId()
                << L"/" << errToLog.errcKrnlId()
                << L"/" << errToLog.errcHostId();
    }
    strmLog << L")" << NewLn;

    // Now append the error text itself, indenting it over
    strmLog << L"     " << errToLog.strErrText();

    // If there is aux text, log it next and indent it also.
    if (!errToLog.strAuxText().bEmpty())
        strmLog << NewLn << L"     " << errToLog.strAuxText();

    strmLog << DNewLn;

    //
    //  According to whether we have a sync semaphore, we have to do
    //  one or another way here.
    //
    if (__pmtxSync)
    {
        TMtxLock mtxlSync(__pmtxSync);
        conOut << strmLog.strData();
    }
     else
    {
        conOut << strmLog.strData();
    }
}
