//
// NAME: CIDLib_StatusControllerTTY.Hpp
//
// DESCRIPTION: 
//
//  A simple derivative of TStatusController is provided here, called
//  TStatusControllerTTY, which will serve many TTY style programs that
//  want to start a thread and just wait until the thread is done, letting
//  it output to the program's output stream while it works.
//
//  The implementation is actually in our parent class' Cpp file.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/21/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//
//  1)  The controlling thread just calls the WaitForComplete() method to
//      wait on the controlled thread.
//
//  2)  The constructor takes an output text file to write to. This lets the
//      controlling thread redirect the working thread's output easily.  It is
//      not adopted, just referenced.
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//   CLASS: TStatusControllerTTY
//  PREFIX: statc
// -----------------------------------------------------------------------------
class CIDLIBEXP TStatusControllerTTY : public TStatusController
{
    public  :
        // ---------------------------------------------------------------------
        //  Constuctors and Destructors
        // ---------------------------------------------------------------------
        TStatusControllerTTY
        (
                    TTextStream* const      pstrmToUse
        );

        TStatusControllerTTY
        (
            const   TStatusControllerTTY&   statcToCopy
        );

        ~TStatusControllerTTY() {}


        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid Starting
        (
            const   TString&                strStartText
        );

        tCIDLib::TVoid WaitForComplete
        (
            const   tCIDLib::TCard4         c4MilliSecs
        );

        tCIDLib::TVoid Working
        (
            const   TString&                strStatusText
            , const tCIDLib::TCard4         c4PercentComplete
        );

        tCIDLib::TVoid WorkComplete
        (
            const   TString&                strClosingComment
        );


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bIsDone();


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TStatusControllerTTY();

        tCIDLib::TVoid operator=(const TStatusControllerTTY&);


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __evWaitForIt
        //      This is an event semaphore that is created initially closed.
        //      When The thread calls WorkComplete() it will be posted to
        //      let anyone blocked on WaitForComplete() go again.
        //
        //  __pstrmTarget
        //      This is a pointer to the output text stream we are to send
        //      our text to. It is just a pointer to a stream that we do not
        //      own.
        // ---------------------------------------------------------------------
        TEvent              __evWaitForIt;
        TTextStream*        __pstrmTarget;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TStatusControllerTTY,TStatusController)
        DefPolyDup(TStatusControllerTTY)
};

#pragma pack(pop)


inline
TStatusControllerTTY::TStatusControllerTTY(const TStatusControllerTTY& statcToCopy) :

    __evWaitForIt(tCIDLib::EEventState_Reset)
    , __pstrmTarget(statcToCopy.__pstrmTarget)
{
}

inline
TStatusControllerTTY::TStatusControllerTTY(TTextStream* const pstrmOutput) :

    __evWaitForIt(tCIDLib::EEventState_Reset)
    , __pstrmTarget(pstrmOutput)
{
}

inline tCIDLib::TVoid
TStatusControllerTTY::Starting(const TString& strStartText)
{
    *__pstrmTarget << strStartText << NewLn;
}

inline tCIDLib::TVoid
TStatusControllerTTY::WaitForComplete(const tCIDLib::TCard4 c4MilliSecs)
{
    __evWaitForIt.WaitFor(c4MilliSecs);
}

inline tCIDLib::TVoid
TStatusControllerTTY::Working(  const   TString&    strStatusText
                                , const tCIDLib::TCard4)
{
    *__pstrmTarget << strStatusText << NewLn;
}

inline tCIDLib::TVoid
TStatusControllerTTY::WorkComplete(const TString& strClosingComment)
{
    *__pstrmTarget << strClosingComment << NewLn;
    __evWaitForIt.Trigger();
}

inline tCIDLib::TBoolean TStatusControllerTTY::bIsDone()
{
    try
    {
        __evWaitForIt.WaitFor(0);
    }

    catch(...)
    {
        return kCIDLib::False;
    }
    return kCIDLib::True;
}
