//
// NAME: CIDLib_Stack.Hpp
//
// DESCRIPTION:
//
//  This is the public header for the stack collection template. It implements
//  the TStack class template, which provides LIFO storage of its elements by
//  value.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/05/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


// ----------------------------------------------------------------------------
//  Forward reference the stack and stack cursor classse
// ----------------------------------------------------------------------------
template <class TElem> class TStackCursor;
template <class TElem> class TStack;


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TStackNode
//  PREFIX: node
// ----------------------------------------------------------------------------
template <class TElem> class TStackNode : public TSLstNode
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TStackNode(const TElem& objData) :
        
            __objData(objData)
        {
        }

        ~TStackNode()
        {
        }


    protected   :
        // --------------------------------------------------------------------
        //  The stack class is our friend
        // --------------------------------------------------------------------
        friend class TStack<TElem>;
        friend class TStackCursor<TElem>;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TStackNode();

        TStackNode(const TStackNode<TElem>&);

        tCIDLib::TVoid operator=(const TStackNode<TElem>&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __objData
        //      The user data object.
        // --------------------------------------------------------------------
        TElem   __objData;


        // --------------------------------------------------------------------
        //  Private, static data members
        //
        //  __bFirstOfType
        //      This is used to trigger the setting of the class name into
        //      the clsThis member on the first creation of an object of this
        //      type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean    __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TStackNode<TElem>,TSLstNode)
};

template <class TElem> tCIDLib::TBoolean TStackNode<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TStackNode<TElem>::clsThis;




// ----------------------------------------------------------------------------
//   CLASS: TStack
//  PREFIX: stk
// ----------------------------------------------------------------------------
template <class TElem> class TStack : public TCollection<TElem>
{
    public  :
        // --------------------------------------------------------------------
        //  Nested typedefs for the cursor and node types used by a stack.
        // --------------------------------------------------------------------
        typedef class TStackCursor<TElem>   TCursor;
        typedef TStackNode<TElem>           TNode;


        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TStack(const tCIDLib::EMTStates eMTSafe = tCIDLib::EMTState_Unsafe) :

            TCollection<TElem>(eMTSafe)
            , __llstStack(0)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
        }

        TStack( const   tCIDLib::TCard4     c4MaxElems
                , const tCIDLib::EMTStates  eMTSafe = tCIDLib::EMTState_Unsafe) :

            TCollection<TElem>(eMTSafe)
            , __llstStack(c4MaxElems)
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
        }

        ~TStack() {}


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Add(const TElem& objNew)
        {
            TLocker lockStack(this);
            Push(objNew);
            _c4IncSerialNum();
        }

        tCIDLib::TBoolean bFlushCur()
        {
            TLocker lockStack(this);
            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Save the next node and flush the iterator node
            TNode* const pnodeNext = (TNode*)__pnodeCur->pnodeNext();
            __llstStack.FlushNode(__pnodeCur);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // Put the next node back into the iterator
            __pnodeCur = pnodeNext;

            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bIsEmpty() const
        {
            TLocker lockStack(this);
            tCIDLib::TBoolean bRet = __llstStack.bIsEmpty();
            return bRet;
        }

        tCIDLib::TBoolean bNext()
        {
            TLocker lockStack(this);
            if (!__pnodeCur)
                return kCIDLib::False;

            __pnodeCur = (TNode*)__pnodeCur->pnodeNext();
            return (__pnodeCur != 0);
        }
        
        tCIDLib::TBoolean bResetIter()
        {
            TLocker lockStack(this);
            __pnodeCur = (TNode*)__llstStack.pnodeHead();
            return (__pnodeCur != 0);
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            TLocker lockStack(this);
            tCIDLib::TCard4 c4Ret = __llstStack.c4ElemCount();
            return c4Ret;
        }

        tCIDLib::TCard4 c4MaxElemCount() const
        {
            TLocker lockStack(this);
            tCIDLib::TCard4 c4Ret = __llstStack.c4MaxElemCount();
            return c4Ret;
        }

        tCIDLib::TVoid Flush()
        {
            TLocker lockStack(this);
            __pnodeCur = 0;
            __llstStack.Flush();
            _c4IncSerialNum();
        }

        TElem& objCur() const
        {
            TLocker lockStack(this);
            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pnodeCur->__objData;
        }

        TElemCursor<TElem>* pcursNew() const
        {
            TLocker lockStack(this);
            return new TCursor(this);
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid FlushAt(TCursor& cursAt)
        {
            TLocker lockStack(this);

            // Make sure the cursor belongs to this stack
            if (!cursAt.bIsCursoring(*this))
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_NotMyCursor
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_BadParms
                    , cursAt.clsIsA()
                    , clsIsA()
                );
            }

            // Get the node to flush
            TNode* pnodeToFlush = cursAt.pnodeCur();

            // Make sure the cursor's valid
            if (!pnodeToFlush)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IterNotValid
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Bump up the cursor
            cursAt.bNext();

            // If the delete node is the internal iterator node, move it up also
            if (pnodeToFlush == __pnodeCur)
                __pnodeCur = (TNode*)__pnodeCur->pnodeNext();

            // Now flush that node
            __llstStack.FlushNode(pnodeToFlush);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            // But we kept the passed cursor up to date by bumping it up
            cursAt._c4SerialNum(c4SerialNum());
        }

        TElem objPeek() const
        {
            TLocker lockStack(this);

            // See if there are any nodes. If not, throw an exception
            if (!__llstStack.c4ElemCount())
            {
                facCIDLib.ThrowErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IsEmpty
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Get the first node
            TNode* pnodeHead = (TNode*)__llstStack.pnodeHead();
            return pnodeHead->__objData;
        }

        TElem objPop()
        {
            TLocker lockStack(this);

            // See if there are any nodes. If not, throw an exception
            if (!__llstStack.c4ElemCount())
            {
                facCIDLib.ThrowErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IsEmpty
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Get the first node
            TNode* pnodeHead = (TNode*)__llstStack.pnodeHead();

            // Get the node's object before we destroy it
            TElem objRet = pnodeHead->__objData;

            // If the popped node is the iterator node, then move it up
            if (__pnodeCur == pnodeHead)
                __pnodeCur = (TNode*)__pnodeCur->pnodeNext();

            __llstStack.FlushNode(pnodeHead);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();

            return objRet;
        }

        tCIDLib::TVoid Push(const TElem& objToPush)
        {
            TLocker lockStack(this);

            // Prepend a node
            TNode* pnodeNew = new TNode(objToPush);
            TJanitor<TNode> janNode(pnodeNew);
            __llstStack.PrependNode(pnodeNew);
            janNode.Orphan();

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();
        }

        tCIDLib::TVoid TrashTop()
        {
            TLocker lockStack(this);

            // See if there are any nodes. If not, throw and exception
            if (!__llstStack.c4ElemCount())
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_IsEmpty
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }

            // Get the first node
            TNode* pnodeHead = (TNode*)__llstStack.pnodeHead();

            // If the trashed node is the iterator node, then move it up
            if (__pnodeCur == pnodeHead)
                __pnodeCur = (TNode*)__pnodeCur->pnodeNext();

            // Delete the current last node
            __llstStack.FlushNode(pnodeHead);

            // Bump the serial number to invalidate cursors
            _c4IncSerialNum();
        }


    protected  :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class TStackCursor<TElem>;

        friend tCIDLib::TBoolean bCompCollections
        (
            const   TStack<TElem>&      col1
            , const TStack<TElem>&      col2
            , const TObjEq<TElem>&      compToUse
        );

        friend tCIDLib::TVoid CopyCollectionNodes
        (
                    TStack<TElem>&      colDest
            , const TStack<TElem>&      colSource
        );

        friend TStack<TElem>* pDupCollection
        (
            const   TStack<TElem>&      colToDup
        );


        // --------------------------------------------------------------------
        //  Hidden constructors
        // --------------------------------------------------------------------
        TStack(const TStack<TElem>& colToCopy) :

            TCollection<TElem>(colToCopy)
            , __llstStack(colToCopy.c4MaxElemCount())
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
        }


        // --------------------------------------------------------------------
        //  Protected, non-virtual methods
        // --------------------------------------------------------------------
        const TSLinkedList& _llstStack() const
        {
            return __llstStack;
        }

        TSLinkedList& _llstStack()
        {
            return __llstStack;
        }


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        tCIDLib::TVoid operator=(const TStack<TElem>&);


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TStackOf");
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __llstStack
        //      This is a singly linked list that is the storage for the stack
        //      elems. Stacks are unidirectional, so we use a singly linked
        //      list for speed and efficiency.
        //
        //  __pnodeCur
        //      The current node for writable iteration.
        // --------------------------------------------------------------------
        TSLinkedList    __llstStack;
        TNode*          __pnodeCur;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a stack of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TStack<TElem>,TCollection<TElem>)
};

template <class TElem> tCIDLib::TBoolean TStack<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TStack<TElem>::clsThis;



// ----------------------------------------------------------------------------
//   CLASS: TStackCursor
//  PREFIX: curs
// ----------------------------------------------------------------------------
template <class TElem> class TStackCursor : public TElemCursor<TElem>
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TStackCursor(const TStack<TElem>* const pcolToCursor) :

            TElemCursor<TElem>(pcolToCursor->c4SerialNum())
            , __pcolCursoring(pcolToCursor)
            , __pllstCursoring(&pcolToCursor->_llstStack())
            , __pnodeCur(0)
        {
            __CheckFirstOfType();
            __pnodeCur = (TStackNode<TElem>*)__pllstCursoring->pnodeHead();
        }

        TStackCursor(const TStackCursor<TElem>& cursToCopy) :

            TElemCursor<TElem>(cursToCopy)
            , __pcolCursoring(cursToCopy.__pcolCursoring)
            , __pllstCursoring(cursToCopy.__pllstCursoring)
            , __pnodeCur(cursToCopy.__pnodeCur)
        {
            __CheckFirstOfType();
        }

        ~TStackCursor() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TStackCursor<TElem>& operator=(const TStackCursor<TElem> cursToAssign)
        {
            if (this == &cursToAssign)
                return *this;

            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            TParent::operator=(cursToAssign);
            __pcolCursoring = cursToAssign.__pcolCursoring;
            __pllstCursoring = cursToAssign.__pllstCursoring;
            __pnodeCur = cursToAssign.__pnodeCur;

            return *this;
        }


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsEmpty() const
        {
            return __pllstCursoring->bIsEmpty();
        }

        tCIDLib::TBoolean bIsValid() const
        {
            // If the serial number is off, its not valid
            if (c4SerialNum() != __pcolCursoring->c4SerialNum())
                return kCIDLib::False;
            return (__pnodeCur != 0);
        }

        tCIDLib::TBoolean bNext()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            if (!__pnodeCur)
                return kCIDLib::False;

            __pnodeCur = (TStackNode<TElem>*)__pnodeCur->pnodeNext();

            return (__pnodeCur != 0);
        }
        
        tCIDLib::TBoolean bReset()
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            __pnodeCur = (TStackNode<TElem>*)__pllstCursoring->pnodeHead();

            // Get our serial back in sync with the collection
            _c4SerialNum(__pcolCursoring->c4SerialNum());

            return (__pnodeCur != 0);
        }

        tCIDLib::TCard4 c4ElemCount() const
        {
            return __pllstCursoring->c4ElemCount();
        }

        const TElem& objCur() const
        {
            // Lock the collection
            TLocker lockCol(__pcolCursoring);

            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            if (!__pnodeCur)
            {
                facCIDLib.LogErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcCol_CursorNotValid
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_AppError
                    , clsIsA()
                );
            }
            return __pnodeCur->__objData;
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bIsCursoring(const TStack<TElem>& colToCheck) const
        {
            return (__pcolCursoring == &colToCheck);
        }

        const TStackNode<TElem>* pnodeCur() const
        {
            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            return __pnodeCur;
        }

        TStackNode<TElem>* pnodeCur()
        {
            _CheckSerialNum(__pcolCursoring->c4SerialNum());
            return __pnodeCur;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Declare our friends
        // --------------------------------------------------------------------
        friend class TStack<TElem>;
        friend tCIDLib::TBoolean bCompareStacks
        (
            const   TStack<TElem>&          col1
            , const TStack<TElem>&          col2
        );


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TStackCursor();


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString  strTmp(L"TStackCursorOf");
                strTmp.Append(TElem::clsThis.pszClassName());
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __pcolCursoring
        //      A pointer to the stack collection that we are cursoring.
        //
        //  __pllstCursoring
        //      This is the linked list we are cursoring. It is extracted from
        //      the stack during construction for fast access.
        //
        //  __pnodeCur
        //      This is the current node that we are on in our iteration. It
        //      is 0 if we are at the end.
        // --------------------------------------------------------------------
        const TStack<TElem>*    __pcolCursoring;
        const TSLinkedList*     __pllstCursoring;
        TStackNode<TElem>*      __pnodeCur;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a stack cursor of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TStackCursor<TElem>,TElemCursor<TElem>)
};

template <class TElem> tCIDLib::TBoolean TStackCursor<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TStackCursor<TElem>::clsThis;

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  Some template functions to make duplication and copying of stacks typesafe
// ----------------------------------------------------------------------------
template <class TElem> tCIDLib::TBoolean
bCompCollections(const  TStack<TElem>&  col1
                , const TStack<TElem>&  col2
                , const TObjEq<TElem>&  compToUse)
{
    // Check for comparison against self
    if (&col1 == &col2)
        return kCIDLib::True;

    // Lock both of the stacks
    TLocker lockStack1(&col1);
    TLocker lockStack2(&col2);

    //
    //  First see if the linked lists behind them have equal element counts,
    //  max element counts, and default delete mode. Note that this does
    //  not compare the elements because the linked knows nothing about how
    //  to compare them.
    //
    if (col1.__llstStack != col2.__llstStack)
        return kCIDLib::False;

    // If the element count of either is 0, then they are both zero so equal
    if (!col1.c4ElemCount())
        return kCIDLib::True;

    //
    //  Ok so we have to compare the elements. So we just create cursors
    //  for both and compare each element. We know that they have the
    //  same number of elements and are locked so this safe.
    //
    TStack<TElem>::TCursor curs1(&col1);
    TStack<TElem>::TCursor curs2(&col2);
    do
    {
        if (!compToUse.bEqual(curs1.objCur(), curs2.objCur()))
            return kCIDLib::False;
    }   while (curs1.bNext() && curs2.bNext());
    return kCIDLib::True;
}

template <class TElem> tCIDLib::TVoid
CopyCollectionNodes(TStack<TElem>& colDest, const TStack<TElem>& colSource)
{
    // Check for copy to self
    if (&colDest == &colSource)
        return;

    // Lock both collections
    TLocker lockNew(&colDest);
    TLocker lockToDup(&colSource);

    if (colSource.c4ElemCount())
    {
        TStack<TElem>::TCursor cursSource(&colSource);
        do
        {
            colDest._llstStack().AppendNode
            (
                new TStack<TElem>::TNode(cursSource.objCur())
            );
        }   while (cursSource.bNext());
    }
}

template <class TElem> TStack<TElem>*
pDupCollection(const TStack<TElem>& colToDup)
{
    // Lock the source collection
    TLocker lockToDup(&colToDup);

    // Create a new stack with the same max elem count
    TStack<TElem>* const pcolNew = new TStack<TElem>(colToDup);
    TJanitor<TStack<TElem> > janCol(pcolNew);

    // If there are any elements in the source, copy them
    if (colToDup.c4ElemCount())
    {
        TStack<TElem>::TCursor cursDup(&colToDup);
        do
        {
            pcolNew->_llstStack().AppendNode
            (
                new TStack<TElem>::TNode(cursDup.objCur())
            );
        }   while (cursDup.bNext());
    }

    // Return a pointer to the new stack
    janCol.Orphan();
    return pcolNew;
}
