//
// NAME: CIDLib_SortedBag.Hpp
//
// DESCRIPTION:
//
//  This header defines a very simple derivative of TBag, TSortedBag, which
//  just adds the capability to insert new objects in sorted order. All it
//  requires, beyond the usual RTTI and constructor/destructor stuff, is to
//  override the Add() method and find the correct sorted add position.
//
//  Note that this means that sorted bag elements must support an > operator,
//  which TBag does not require.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 01/30/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//

#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TSortedBag
//  PREFIX: bag
// ----------------------------------------------------------------------------
template <class TElem> class TSortedBag : public TBag<TElem>
{
    public :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TSortedBag(         TObjComp<TElem>*    pcompToAdopt
                    , const tCIDLib::EMTStates  eMTSafe = tCIDLib::EMTState_Unsafe) :

            TBag<TElem>(eMTSafe)
            , __pcompToUse(pcompToAdopt)
        {
            __CheckFirstOfType();
        }

        TSortedBag( const   tCIDLib::TCard4     c4MaxItems
                    ,       TObjComp<TElem>*    pcompToAdopt
                    , const tCIDLib::EMTStates  eMTSafe = tCIDLib::EMTState_Unsafe) :

            TBag<TElem>(c4MaxItems, eMTSafe)
            , __pcompToUse(pcompToAdopt)
        {
            __CheckFirstOfType();
        }

        ~TSortedBag()
        {
            delete __pcompToUse;
        }


        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid Add(const TElem& objNew)
        {
            // Optimize if the bag is empty
            if (bIsEmpty())
            {
                TBag<TElem>::Add(objNew);
                return;
            }

            //
            //  Find the position to put the new element at. We just iterate
            //  the nodes until we find one that is greater than the new
            //  object. We then insert the new element before it.
            //
            TBag<TElem>::TCursor cursAdd(this);
            do
            {
                if (__pcompToUse->eCompare
                (
                    cursAdd.objCur()
                    , objNew) == tCIDLib::ESort_FirstGreater)
                {
                    break;
                }
            }   while (cursAdd.bNext());

            // If the cursor is invalid, then we put this one at the end.
            if (!cursAdd.bIsValid())
            {
                TBag<TElem>::Add(objNew);
            }
             else
            {
                //
                //  Move back now so that we can insert after this node. If
                //  going back invalidates, then we prepend at the head.
                //  Else, we insert after the node.
                //
                TBag<TElem>::TNode* pnodeNew = new TBag<TElem>::TNode(objNew);
                if (!cursAdd.bPrevious())
                    _llstBag().PrependNode(pnodeNew);
                 else
                    _llstBag().InsertNode(cursAdd.pnodeCur(), pnodeNew);
            }
        }


    protected :
        // --------------------------------------------------------------------
        //  Hidden constructors and operators
        // --------------------------------------------------------------------
        TSortedBag(const TSortedBag<TElem>& colToCopy) :

            TBag<TElem>(colToCopy)
            , __pcompToUse(0)
        {
            __pcompToUse = ::pDupObject<TObjComp<TElem> >
            (
                *colToCopy.__pcompToUse
            );
        }


    private  :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        tCIDLib::TVoid operator=(const TSortedBag<TElem>&);


        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __CheckFirstOfType()
        {
            if (!__bFirstOfType)
            {
                __bFirstOfType = kCIDLib::True;

                TString strTmp(L"TSortedBagOf", 64);
                strTmp.Append(TElem::clsThis.pszClassName());

                // Force an update of the class object
                *((TClass*)&clsThis) = TClass(strTmp);
            }
        }


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __pcompToUse
        //      The comparator object to use for comparing objects to find
        //      their insert position. It is an allocated object that we
        //      have adopted.
        // --------------------------------------------------------------------
        TObjComp<TElem>*    __pcompToUse;


        // --------------------------------------------------------------------
        //  Private static data members
        //
        //  __bFirstOfType
        //      This is used to force the update of the clsThis member on the
        //      first construction of a bag of a particular type.
        // --------------------------------------------------------------------
        static tCIDLib::TBoolean   __bFirstOfType;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TSortedBag<TElem>,TBag<TElem>)
};

template <class TElem> tCIDLib::TBoolean TSortedBag<TElem>::__bFirstOfType = kCIDLib::False;
template <class TElem> const TClass TSortedBag<TElem>::clsThis;

#pragma pack(pop)
