//
// NAME: CIDLib_Size.Cpp
//
// DESCRIPTION:
//
//	This module implements the TSize class, which is a 2D extent in an
//  integral coordinate space.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/10/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData2(TSize,TObject)


// -----------------------------------------------------------------------------
//  Local static functions
// -----------------------------------------------------------------------------
static tCIDLib::TVoid
__CheckValue(const tCIDLib::TCard4 c4Value)
{
    if (c4Value > kCIDLib::c4MaxCard)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSize_UnderOverflow
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }
}

static tCIDLib::TVoid
__CheckOffset(const tCIDLib::TCard4 c4Value, const tCIDLib::TInt4 i4Ofs)
{
    tCIDLib::TInt8 i8Test = c4Value;
    i8Test += i4Ofs;

    if ((i8Test < 0) || (i8Test > kCIDLib::c4MaxCard))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSize_UnderOverflow
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }
}

static tCIDLib::TVoid
__CheckOffset(const tCIDLib::TCard4 c4Value, const tCIDLib::TCard4 c4Ofs)
{
    if ((tCIDLib::TInt8(c4Value) + c4Ofs) > kCIDLib::c4MaxCard)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSize_UnderOverflow
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }
}

static tCIDLib::TVoid
__CheckSub(const tCIDLib::TCard4 c4Value, const tCIDLib::TCard4 c4Ofs)
{
    if (tCIDLib::TInt8(c4Value) - c4Ofs < 0)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSize_UnderOverflow
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }
}




// -----------------------------------------------------------------------------
//   CLASS: TSize
//  PREFIX: sz
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TSize: Constructors and Destructors
// -----------------------------------------------------------------------------

TSize::TSize(const TSize& szSrc) :

    __c4CX(szSrc.__c4CX)
    , __c4CY(szSrc.__c4CY)
{
}

TSize::TSize() :

    __c4CX(0)
    , __c4CY(0)
{
}

TSize::TSize(const tCIDLib::TCard4 c4X, const tCIDLib::TCard4 c4Y) :

    __c4CX(c4X)
    , __c4CY(c4Y)
{
}

TSize::TSize(const tCIDLib::THostSize& szToCopy) :

    __c4CX(szToCopy.i4CX)
    , __c4CY(szToCopy.i4CY)
{
    #if CID_DEBUG_ON
    __CheckOffset(0, szToCopy.i4CX);
    __CheckOffset(0, szToCopy.i4CY);
    #endif
}

TSize::TSize(const tCIDLib::TCard4 c4Both) :

    __c4CX(c4Both)
    , __c4CY(c4Both)
{
}

// -----------------------------------------------------------------------------
//  TSize: Public operators
// -----------------------------------------------------------------------------

tCIDLib::TBoolean TSize::operator==(const TSize& szToTest) const
{
    return (__c4CX == szToTest.__c4CX) && (__c4CY == szToTest.__c4CY);
}

tCIDLib::TVoid TSize::operator=(const TSize& szToAssign)
{
    if (&szToAssign == this)
        return;

    __c4CX = szToAssign.__c4CX;
    __c4CY = szToAssign.__c4CY;
    return;
}

tCIDLib::TVoid TSize::operator+=(const TSize& szToAdd)
{
    #if CID_DEBUG_ON
    __CheckOffset(__c4CX, szToAdd.__c4CX);
    __CheckOffset(__c4CY, szToAdd.__c4CY);
    #endif

    __c4CX += szToAdd.__c4CX;
    __c4CY += szToAdd.__c4CY;
}

tCIDLib::TVoid TSize::operator-=(const TSize& szToSub)
{
    #if CID_DEBUG_ON
    __CheckSub(__c4CX, szToSub.__c4CX);
    __CheckSub(__c4CY, szToSub.__c4CY);
    #endif

    __c4CX -= szToSub.__c4CX;
    __c4CY -= szToSub.__c4CY;
}

TSize operator+(const TSize& sz1, const TSize& sz2)
{
    #if CID_DEBUG_ON
    __CheckOffset(sz1.__c4CX, sz2.__c4CX);
    __CheckOffset(sz1.__c4CY, sz2.__c4CY);
    #endif
    return TSize(sz1.__c4CX + sz2.__c4CX, sz1.__c4CY + sz2.__c4CY);
}

TSize operator-(const TSize& sz1, const TSize& sz2)
{
    #if CID_DEBUG_ON
    __CheckSub(sz1.__c4CX, sz2.__c4CX);
    __CheckSub(sz1.__c4CY, sz2.__c4CY);
    #endif
    return TSize(sz1.__c4CX - sz2.__c4CX, sz1.__c4CY - sz2.__c4CY);
}


// -----------------------------------------------------------------------------
//  TSize: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TSize::Adjust(const tCIDLib::TInt4 i4XOfs, const tCIDLib::TInt4 i4YOfs)
{
    #if CID_DEBUG_ON
    __CheckOffset(__c4CX, i4XOfs);
    __CheckOffset(__c4CY, i4YOfs);
    #endif

    __c4CX += i4XOfs;
    __c4CY += i4YOfs;
}

tCIDLib::TVoid TSize::Adjust(const tCIDLib::TInt4 i4Ofs)
{
    #if CID_DEBUG_ON
    __CheckOffset(__c4CX, i4Ofs);
    __CheckOffset(__c4CY, i4Ofs);
    #endif

    __c4CX += i4Ofs;
    __c4CY += i4Ofs;
}

tCIDLib::TVoid TSize::AdjustCX(const tCIDLib::TInt4 i4Ofs)
{
    #if CID_DEBUG_ON
    __CheckOffset(__c4CX, i4Ofs);
    #endif

    __c4CX += i4Ofs;
}

tCIDLib::TVoid TSize::AdjustCY(const tCIDLib::TInt4 i4Ofs)
{
    #if CID_DEBUG_ON
    __CheckOffset(__c4CY, i4Ofs);
    #endif

    __c4CY += i4Ofs;
}

tCIDLib::TCard4 TSize::c4Packed() const
{
    if ((__c4CX > kCIDLib::c2MaxCard) || (__c4CY > kCIDLib::c2MaxCard))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSize_CantPack
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    return TRawBits::c4From16(tCIDLib::TCard2(__c4CX), tCIDLib::TCard2(__c4CY));
}


tCIDLib::TVoid TSize::ToHostPt(tCIDLib::THostPoint& ptlToFill) const
{
    #if CID_DEBUG_ON
    __CheckValue(__c4CX);
    __CheckValue(__c4CY);
    #endif

    ptlToFill.i4X = tCIDLib::TInt4(__c4CX);
    ptlToFill.i4Y = tCIDLib::TInt4(__c4CY);
}

tCIDLib::TVoid TSize::ToHostSize(tCIDLib::THostSize& szToFill) const
{
    #if CID_DEBUG_ON
    __CheckValue(__c4CX);
    __CheckValue(__c4CY);
    #endif

    szToFill.i4CX = tCIDLib::TInt4(__c4CX);
    szToFill.i4CY = tCIDLib::TInt4(__c4CY);
}


// -----------------------------------------------------------------------------
//  TSize: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TSize::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo << __c4CX << L"," << __c4CY;
}

tCIDLib::TVoid TSize::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Get the vector value
    strmToReadFrom >> __c4CX;
    strmToReadFrom >> __c4CY;
}

tCIDLib::TVoid TSize::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Write out the origin and size information
    strmToWriteTo << __c4CX;
    strmToWriteTo << __c4CY;
}
