//
// NAME: CIDLib_SharedMemory.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_SharedMemory.Cpp module. This module
//  provides the TSharedMemBuf class, which is a buffer of system allocated
//  memory that is named.
//
//  It is always fully committed, unlike TSharedMemBuf, which provides
//  unnamed system memory buffers that can be committed as needed. However
//  for architectural reasons, it must provide the _ReallocateTo() method override
//  that its parent class requires. But, since its already fully commited, it does
//  not have to do anything but check for a request beyond its max size.
//
//  The default action during construction is to either open the buffer if
//  it already exists or create it if it does not.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 02/28/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//  1)  As with other memory buffers, distinguish between equality of two
//      buffer objects and the equality of the contents of them. Use
//      bBuffersEqual() for testing the actual buffer contents. Or you can
//      just compare the names because two buffers with the same name
//      actually point at the same memory.
//
//  2)  Note that creating another shared memory buffer with the same name
//      and access flags as another buffer is basically the same as doing
//      a copy constructor, since it will refer to the same buffer.
//


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: TSharedMemBuf
// PREFIX: mbuf
// -----------------------------------------------------------------------------
class CIDLIBEXP TSharedMemBuf : public TMemBuf
{
    public  :
        // ---------------------------------------------------------------------
        //  Constructors and destructors
        // ---------------------------------------------------------------------
        TSharedMemBuf
        (
            const   tCIDLib::TCard4         c4Size
            , const TResourceName&          rsnToUse
            , const tCIDLib::EMemAccFlags   eAccFlags = tCIDLib::EMemAcc_ReadWrite
            , const tCIDLib::ECreateActions eCreate = tCIDLib::ECreateAct_OpenOrCreate
        );

        TSharedMemBuf
        (
            const   TSharedMemBuf&          mbufSrc
        );

        ~TSharedMemBuf();


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TSharedMemBuf& operator=
        (
            const   TSharedMemBuf&          mbufToAssign
        );

        tCIDLib::TBoolean operator==
        (
            const   TSharedMemBuf&          mbufToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TSharedMemBuf&          mbufToTest
        )   const;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::EMemAccFlags eAccess() const;

        const TResourceName& rsnName() const;

    
    protected           :
        // ---------------------------------------------------------------------
        //  Protected methods
        // ---------------------------------------------------------------------
        TSharedMemBuf();


        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _ReallocateTo
        (
            const   tCIDLib::TCard4         c4NewSize
            , const tCIDLib::TBoolean       bPreserve
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private             :
        // ---------------------------------------------------------------------
        //  Private, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TCard4 __c4RoundedUp
        (
            const   tCIDLib::TCard4         c4ToRound
        )   const;

        tCIDLib::TVoid __CreateOrOpen
        (
            const   tCIDLib::ECreateActions eCreateFlags
        );

        tCIDLib::TVoid* __pPageAdr
        (
            const   tCIDLib::TVoid* const   pAddress
        )   const;


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __rsnThis
        //      This is the resource name object that holds the abstracted
        //      name for the buffer.
        //
        //  __ksmbThis
        //      This is the shared memory buffer object that holds the
        //      buffer information.
        // ---------------------------------------------------------------------
        TResourceName           __rsnThis;
        TKrnlSharedMemBuf       __ksmbThis;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        BefriendFactory(TSharedMemBuf)
        RTTIMacros(TSharedMemBuf,TMemBuf)
        DefPolyDup(TSharedMemBuf)
};

#pragma pack(pop)


// -----------------------------------------------------------------------------
//  Public operators
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean
TSharedMemBuf::operator!=(const TSharedMemBuf& mbufToTest) const
{
    return !operator==(mbufToTest);
}

// -----------------------------------------------------------------------------
//  Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::EMemAccFlags TSharedMemBuf::eAccess() const
{
    return __ksmbThis.eAccess();
}

inline const TResourceName& TSharedMemBuf::rsnName() const
{
    return __rsnThis;
}


// -----------------------------------------------------------------------------
//  Hidden constructors. Needed for polymorphic streaming. Note that it will
//  leave the object in a bad condition.
// -----------------------------------------------------------------------------
inline TSharedMemBuf::TSharedMemBuf() :

    TMemBuf()
{
}
