//
// NAME: CIDLib_SharedMemory.Cpp
//
// DESCRIPTION:
//
//  This method provides a derivative of TMemBuf that manages named, shared
//  memory buffers. They are much like TSysBuf objects, but are named and
//  can be shared between processes.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 02/28/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI macros
// -----------------------------------------------------------------------------
RTTIData(TSharedMemBuf,TMemBuf)



// -----------------------------------------------------------------------------
//   CLASS: TSharedMemBuf
//  PREFIX: mbuf
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TSharedMemBuf: Constructors and Destructors
// -----------------------------------------------------------------------------

TSharedMemBuf::TSharedMemBuf(   const   tCIDLib::TCard4         c4Size
                                , const TResourceName&          rsnToUse
                                , const tCIDLib::EMemAccFlags   eAccessFlags
                                , const tCIDLib::ECreateActions eCreate) :
    TMemBuf
    (
        __c4RoundedUp(c4Size)
        , __c4RoundedUp(c4Size)
        , kCIDLib::c4MemPageSize
        , 0
    )
    , __rsnThis(rsnToUse)
{
    try
    {
        __ksmbThis.Alloc
        (
            __rsnThis.strFullName(tCIDLib::ENamedRsc_Memory).pszData()
            , c4Size
            , eAccessFlags
            , eCreate
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMBuf_AllocShared
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TCardinal(c4Size)
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Memory)
        );
    }

    // Store the data pointer
    _pc1Data = (tCIDLib::TCard1*)__ksmbThis.pData();
}

TSharedMemBuf::TSharedMemBuf(const TSharedMemBuf& mbufToCopy) :

    TMemBuf(mbufToCopy)
    , __rsnThis(mbufToCopy.__rsnThis)
{
    try
    {
        __ksmbThis.Alloc
        (
            __rsnThis.strFullName(tCIDLib::ENamedRsc_Memory).pszData()
            , mbufToCopy.__ksmbThis.c4Size()
            , mbufToCopy.__ksmbThis.eAccess()
            , tCIDLib::ECreateAct_OpenIfExists
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMBuf_AllocShared
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Memory)
        );
    }

    // Set the buffer pointer
    _pc1Data = (tCIDLib::TCard1*)__ksmbThis.pData();
}

TSharedMemBuf::~TSharedMemBuf()
{
    // Clear out the parent class' data members for the buffer
    _ReleaseBuffer();

    //
    //  If we have a buffer, then free it. We cannot just let it destruct
    //  because we have to catch any errors and translate them to CIDLib
    //  level errors.
    //
    if (__ksmbThis.pData())
    {
        try
        {
            __ksmbThis.Free();
        }

        catch(const TKrnlError& kerrToCatch)
        {
            // Log as a warning, so it does not propogate out of destructor
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcMBuf_FreeShared
                , kerrToCatch
                , tCIDLib::ESev_Warning
                , tCIDLib::EClass_CantDo
                , __rsnThis.strFullName(tCIDLib::ENamedRsc_Memory)
            );
        }
    }
}


// -----------------------------------------------------------------------------
//  TSharedMemBuf: Public operators
// -----------------------------------------------------------------------------

TSharedMemBuf& TSharedMemBuf::operator=(const TSharedMemBuf& mbufToAssign)
{
    if (this == &mbufToAssign)
        return *this;

    // If there is an existing buffer, then free it
    if (__ksmbThis.pData())
    {
        _ReleaseBuffer();

        try
        {
            __ksmbThis.Free();
        }

        catch(const TKrnlError& kerrToCatch)
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcMBuf_FreeShared
                , kerrToCatch
                , tCIDLib::ESev_Warning
                , tCIDLib::EClass_CantDo
                , __rsnThis.strFullName(tCIDLib::ENamedRsc_Memory)
            );
        }
    }

    // Copy all the members over now
    _c1Fill             = mbufToAssign._c1Fill;
    _c4ExpandIncrement  = mbufToAssign._c4ExpandIncrement;
    _c4MaxSize          = mbufToAssign._c4MaxSize;
    _c4Size             = mbufToAssign._c4Size;
    __rsnThis           = mbufToAssign.__rsnThis;

    try
    {
        __ksmbThis.Alloc
        (
            __rsnThis.strFullName(tCIDLib::ENamedRsc_Memory).pszData()
            , mbufToAssign.__ksmbThis.c4Size()
            , mbufToAssign.__ksmbThis.eAccess()
            , tCIDLib::ECreateAct_OpenIfExists
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMBuf_DupShared
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Memory)
        );
    }

    // Store the buffer pointer
    _pc1Data = (tCIDLib::TCard1*)__ksmbThis.pData();

    return *this;
}


tCIDLib::TBoolean
TSharedMemBuf::operator==(const TSharedMemBuf& mbufToTest) const
{
    if (__ksmbThis != mbufToTest.__ksmbThis)
        return kCIDLib::False;

    if (__rsnThis != mbufToTest.__rsnThis)
        return kCIDLib::False;

    // Call our parent's version now to compare the actual memory
    return TParent::operator==(mbufToTest);
}


// -----------------------------------------------------------------------------
//  TSharedMemBuf: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TSharedMemBuf::_ReallocateTo(   const   tCIDLib::TCard4     c4NewSize
                                , const tCIDLib::TBoolean   bPreserve) const
{
    facCIDLib.LogErr
    (
        __FILE__
        , __LINE__
        , kCIDErrs::errcMBuf_NoRealloc
        , tCIDLib::ESev_APIFailed
        , tCIDLib::EClass_AppError
        , clsIsA()
    );
}


tCIDLib::TVoid TSharedMemBuf::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    //  Delete the current buffer
    if (__ksmbThis.pData())
    {
        _ReleaseBuffer();

        try
        {
            __ksmbThis.Free();
        }

        catch(const TKrnlError& kerrToCatch)
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcMBuf_FreeShared
                , kerrToCatch
                , tCIDLib::ESev_ProcessFatal
                , tCIDLib::EClass_Internal
                , __rsnThis.strFullName(tCIDLib::ENamedRsc_Memory)
            );
        }
    }

    // Do our parent first
    TParent::_StreamFrom(strmToReadFrom);

    // Get the new info out
    tCIDLib::EMemAccFlags   eAccess;
    strmToReadFrom >> _c4Size;
    strmToReadFrom >> eAccess;
    strmToReadFrom >> _c1Fill;
    strmToReadFrom >> _c4ExpandIncrement;
    strmToReadFrom >> _c4MaxSize;
    strmToReadFrom >> __rsnThis;

    // Create the buffer
    try
    {
        __ksmbThis.Alloc
        (
            __rsnThis.strFullName(tCIDLib::ENamedRsc_Memory).pszData()
            , _c4Size
            , eAccess
            , tCIDLib::ECreateAct_OpenOrCreate
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcMBuf_AllocShared
            , kerrToCatch
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Memory)
        );
    }

    // Store the new buffer pointer
    _pc1Data = (tCIDLib::TCard1*)__ksmbThis.pData();

    // Now read in the actual data
    strmToReadFrom.ReadRawBuffer(_pc1Data, _c4Size);
}

tCIDLib::TVoid TSharedMemBuf::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Do our parent first
    TParent::_StreamTo(strmToWriteTo);

    // Stream out our members and our parent's
    strmToWriteTo << _c4Size;
    strmToWriteTo << __ksmbThis.eAccess();
    strmToWriteTo << _c1Fill;
    strmToWriteTo << _c4ExpandIncrement;
    strmToWriteTo << _c4MaxSize;
    strmToWriteTo << __rsnThis;

    // Now stream out the actual data
    strmToWriteTo.WriteRawBuffer(_pc1Data, _c4Size);
}


// -----------------------------------------------------------------------------
//  TSharedMemBuf: Private, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __c4Rounded
//
// DESCRIPTION:
//
//  Handles our commonly needed chore of rounding a value up to the next
//  page boundary.
// ---------------------------------------
//   INPUT: c4ToRound is the value to round
//
//  OUTPUT: None
//
//  RETURN: The rounded value
//
tCIDLib::TCard4
TSharedMemBuf::__c4RoundedUp(const tCIDLib::TCard4 c4ToRound) const
{
    // if 0, then return 0
    if (!c4ToRound)
        return c4ToRound;

    // If less than the page size, then just return the page size
    if (c4ToRound < kCIDLib::c4MemPageSize)
        return kCIDLib::c4MemPageSize;

    tCIDLib::TCard4 c4Ret = c4ToRound / kCIDLib::c4MemPageSize;

    // If its already correct, then just return as is
    if (c4ToRound == (c4Ret * kCIDLib::c4MemPageSize))
        return c4ToRound;

    return (c4Ret +1 ) * kCIDLib::c4MemPageSize;
}


//
// FUNCTION/METHOD NAME: __pPageAdr
//
// DESCRIPTION:
//
//  Calculates the base page address of a given address.
// ---------------------------------------
//   INPUT: pAddress is the address to get the page for.
//
//  OUTPUT: None
//
//  RETURN: The base page address
//
tCIDLib::TVoid*
TSharedMemBuf::__pPageAdr(const tCIDLib::TVoid* const pAddress) const
{
    tCIDLib::TCard4 c4Tmp = tCIDLib::TCard4(pAddress);
    c4Tmp = c4Tmp / kCIDLib::c4MemPageSize;
    return (tCIDLib::TVoid*)(c4Tmp * kCIDLib::c4MemPageSize);
}
