//
// NAME: CIDLib_Semphore.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_Semphore.Cpp module. It implements the
//  TSemaphore class, which is a standard counting semaphore.
//
//  TSemJanitor is a janitor class for a semaphore. It makes it safe to enter
//  a semaphore, and insures that it gets exited in the face of exceptions.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 05/18/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//  CLASS: TSemaphore
// PREFIX: sem
// ----------------------------------------------------------------------------
class CIDLIBEXP TSemaphore : public TObject, public MFormattable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors.
        // --------------------------------------------------------------------
        TSemaphore
        (
            const   tCIDLib::TCard4         c4MaxCount
            , const tCIDLib::TCard4         c4InitCount
        );

        TSemaphore
        (
            const   TResourceName&          rsnToUse
            , const tCIDLib::TCard4         c4InitCount
            , const tCIDLib::TCard4         c4MaxCount
            , const tCIDLib::ECreateActions eCreateAct = tCIDLib::ECreateAct_OpenOrCreate
        );

        ~TSemaphore();


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bNamed() const;

        tCIDLib::TCard4 c4MaxCount() const;

        tCIDLib::TVoid Enter
        (
            const   tCIDLib::TCard4         c4Timeout = kCIDLib::c4MaxWait
        );

        tCIDLib::TVoid Exit();

        const TResourceName& rsnName() const;


    protected   :
        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TSemaphore();

        TSemaphore(const TSemaphore&);

        tCIDLib::TVoid operator=(const TSemaphore&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __bNamed
        //      Indicates whether this is a named or unnamed semaphore.
        //
        //  __ksemImpl
        //      This is the kernel semphore object that provides the real
        //      implementation and platform abstraction.
        //
        //  __rsnThis
        //      The name of the semaphore, in a resource name object.
        // --------------------------------------------------------------------
        tCIDLib::TBoolean   __bNamed;
        TKrnlSemaphore      __ksemImpl;
        TResourceName       __rsnThis;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TSemaphore,TObject)
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  CLASS: TSemJanitor
// PREFIX: jan
// ----------------------------------------------------------------------------
class CIDLIBEXP TSemJanitor : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors.
        // --------------------------------------------------------------------
        TSemJanitor
        (
                    TSemaphore* const       psemToEnter
            , const tCIDLib::TCard4         c4MilliSecs = kCIDLib::c4MaxWait
        );

        ~TSemJanitor();

    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TSemJanitor(const TSemJanitor&);

        tCIDLib::TVoid operator=(const TSemJanitor&);


        // --------------------------------------------------------------------
        //  Private data
        //
        //  __psemToEnter
        //      The semaphore to enter and exit. We keep a pointer to it but
        //      don't adopt it.
        // --------------------------------------------------------------------
        TSemaphore*     __psemToEnter;


        // --------------------------------------------------------------------
        //  Magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TSemJanitor, TObject)
};


// ----------------------------------------------------------------------------
//  TSemaphore: Public, inherited methods
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TSemaphore::bNamed() const
{
    return __bNamed;
}

inline tCIDLib::TCard4 TSemaphore::c4MaxCount() const
{
    return __ksemImpl.c4MaxCount();
}

inline const TResourceName& TSemaphore::rsnName() const
{
    return __rsnThis;
}
