//
// NAME: CIDLib_Semaphore.Cpp
//
// DESCRIPTION:
//
//  This module implements the TSemaphore class, which is a standard
//  counting semaphore.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 05/18/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our standard members macros
// ----------------------------------------------------------------------------
RTTIData(TSemaphore,TObject)
RTTIData(TSemJanitor,TObject)



// ----------------------------------------------------------------------------
//   CLASS: TSemaphore
//  PREFIX: sem
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TSemaphore: Constructors and Destructors
// ----------------------------------------------------------------------------

TSemaphore::TSemaphore( const   tCIDLib::TCard4 c4InitCount
                        , const tCIDLib::TCard4 c4MaxCount) :

    __bNamed(kCIDLib::False)
    , __ksemImpl(c4MaxCount)
{
    try
    {
        __ksemImpl.Create(c4InitCount);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSem_Create
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Semaphore)
        );
    }
}

TSemaphore::TSemaphore( const   TResourceName&          rsnToUse
                        , const tCIDLib::TCard4         c4InitCount
                        , const tCIDLib::TCard4         c4MaxCount
                        , const tCIDLib::ECreateActions eAction) :

    __bNamed(kCIDLib::True)
    , __ksemImpl
      (
        rsnToUse.strFullName(tCIDLib::ENamedRsc_Semaphore).pszData()
        , c4MaxCount
      )
{
    #if CID_DEBUG_ON
    //
    //  Check for illegal create actions for this type of operation.
    //  
    if ((eAction == tCIDLib::ECreateAct_ReplaceIfExists)
    ||  (eAction == tCIDLib::ECreateAct_TruncateIfExists))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSem_CreateAction
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TString(L"'Replace/truncate if exists'")
        );
    }
    #endif

    //
    //  First try to open the semaphore. Then see if the open action
    //  makes this a bad thing.
    //
    try
    {
        __ksemImpl.Open();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        //
        //  If it failed and we were asked only to open if it exists,
        //  then log and error.
        //
        if ((kerrToCatch.errcId() == kKrnlErrors::errcFileNotFound)
        &&  (eAction == tCIDLib::ECreateAct_OpenIfExists))
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcSem_Open
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , __rsnThis.strFullName(tCIDLib::ENamedRsc_Semaphore)
            );
        }

        //
        //  If we were asked to create it, then lets try that and see if
        //  it works any better. If we were not asked to create it, then
        //  give up.
        //
        if ((eAction == tCIDLib::ECreateAct_CreateIfNew)
        ||  (eAction == tCIDLib::ECreateAct_OpenOrCreate))
        {
            try
            {
                __ksemImpl.Create(c4InitCount);
            }

            catch(const TKrnlError& kerrToCatch)
            {
                facCIDLib.LogKrnlErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDErrs::errcSem_Create
                    , kerrToCatch
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_CantDo
                    , __rsnThis.strFullName(tCIDLib::ENamedRsc_Semaphore)
                );
            }
        }
         else
        {
            facCIDLib.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDErrs::errcSem_Open
                , kerrToCatch
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , __rsnThis.strFullName(tCIDLib::ENamedRsc_Semaphore)
            );
        }
    }

    //
    //  We opened it. If the open action indicates that we should fail if
    //  it exists, then close it an fail.
    //
    if (eAction == tCIDLib::ECreateAct_FailIfExists)
    {
        __ksemImpl.Close();
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSem_AlreadyExists
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_Already
            , TString(L"TSemaphore")
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Semaphore)
        );
    }
}

TSemaphore::~TSemaphore()
{
    try
    {
        __ksemImpl.Close();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        //
        //  If we get an error, log it but don't throw out of the destructor
        //  so we use a warning message.
        //
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSem_Close
            , kerrToCatch
            , tCIDLib::ESev_Warning
            , tCIDLib::EClass_Internal
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Semaphore)
        );
    }
}


// ----------------------------------------------------------------------------
//  TSemaphore: Constructors and Destructors
// ----------------------------------------------------------------------------

tCIDLib::TVoid TSemaphore::Enter(const tCIDLib::TCard4 c4Timeout)
{
    try
    {
        __ksemImpl.Enter(c4Timeout);
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSem_Enter
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Semaphore)
        );
    }
}

tCIDLib::TVoid TSemaphore::Exit()
{
    try
    {
        __ksemImpl.Exit();
    }

    catch(const TKrnlError& kerrToCatch)
    {
        facCIDLib.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcSem_Exit
            , kerrToCatch
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , __rsnThis.strFullName(tCIDLib::ENamedRsc_Semaphore)
        );
    }
}

// ----------------------------------------------------------------------------
//  TSemaphore: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TSemaphore::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo   << L"Name="
                    << __rsnThis.strFullName(tCIDLib::ENamedRsc_Semaphore)
                    << L", Max Count=" << __ksemImpl.c4MaxCount();
}



// ----------------------------------------------------------------------------
//  CLASS: TSemJanitor
// PREFIX: jan
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TSemJanitor: Constructors and Destructors.
// ----------------------------------------------------------------------------

TSemJanitor::TSemJanitor(       TSemaphore* const   psemToEnter
                        , const tCIDLib::TCard4     c4MilliSecs) :

    __psemToEnter(0)
{
    psemToEnter->Enter(c4MilliSecs);

    // We successfull entered, so save the semaphore pointer
    __psemToEnter = psemToEnter;
}

TSemJanitor::~TSemJanitor()
{
    // If it got entered, then exit
    if (__psemToEnter)
        __psemToEnter->Exit();
}
