//
// NAME: CIDLib_SLinkedList.Hpp
//
// DESCRIPTION: 
//
//  This is the header for the CIDLib_SLinkedList.Cpp module. This module
//  implements the concrete singly linked list collection type. The linked
//  list is defined in terms of an abstract node class, from which a template
//  class will later be defined. That template class will be instantiated
//  many times for particular styles of collections built upon this
//  basic linked list functionality.
//
//  Note that this concrete linked list class is not derived from the
//  base collection classes itself. The template classes that use it are
//  derived from the collection classes.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/22/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TSLstNode
//  PREFIX: node
// ----------------------------------------------------------------------------
class CIDLIBEXP TSLstNode : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and destructors
        // --------------------------------------------------------------------
        ~TSLstNode();


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        TSLstNode* pnodeNext() const;


    protected   :
        // --------------------------------------------------------------------
        //  Our linked list class is our friend. Can you say friend? Sure!
        //  That's real nice.
        // --------------------------------------------------------------------
        friend class TSLinkedList;


        // --------------------------------------------------------------------
        //  Hidden constructors
        // --------------------------------------------------------------------
        TSLstNode();


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // --------------------------------------------------------------------
        TSLstNode(const TSLstNode&);

        tCIDLib::TVoid operator=(const TSLstNode&);

        tCIDLib::TBoolean operator==(const TSLstNode&) const;

        tCIDLib::TBoolean operator!=(const TSLstNode&) const;


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __pnodeNext
        //      The next node of this node
        // --------------------------------------------------------------------
        TSLstNode*      __pnodeNext;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TSLstNode,TObject)
};



// ----------------------------------------------------------------------------
//   CLASS: TSLinkedList
//  PREFIX: llst
// ----------------------------------------------------------------------------
class CIDLIBEXP TSLinkedList : public TObject
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors/Destructors
        // --------------------------------------------------------------------
        TSLinkedList();

        TSLinkedList
        (
            const   tCIDLib::TCard4         c4MaxElems
        );

        ~TSLinkedList();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        tCIDLib::TBoolean operator==
        (
            const   TSLinkedList&            llstToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TSLinkedList&            llstToTest
        )   const;


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid AppendNode
        (
                    TSLstNode* const        pnodeNew
        );

        tCIDLib::TBoolean bIsEmpty() const;

        tCIDLib::TBoolean bIsFirstNode
        (
                    TSLstNode* const        pnodeTest
        )   const;

        tCIDLib::TBoolean bIsLastNode
        (
                    TSLstNode* const        pnodeTest
        )   const;

        tCIDLib::TCard4 c4ElemCount() const;

        tCIDLib::TCard4 c4MaxElemCount() const;

        tCIDLib::TCard4 c4MaxElemCount
        (
            const   tCIDLib::TCard4         c4NewMax
        );

        tCIDLib::TVoid Flush();

        tCIDLib::TVoid FlushNode
        (
                    TSLstNode* const        pnodeToFlush
        );

        tCIDLib::TVoid InsertNode
        (
                    TSLstNode* const        pnodeAfter
            ,       TSLstNode* const        pnodeNew
        );

        tCIDLib::TVoid MoveToHead
        (
                    TSLstNode* const        pnodeToMove
        );

        tCIDLib::TVoid MoveToTail
        (
                    TSLstNode* const        pnodeToMove
        );

        TSLstNode* pnodeHead() const;

        TSLstNode* pnodeTail() const;

        tCIDLib::TVoid PrependNode
        (
                    TSLstNode* const        pnodeNew
        );


    private :
        // --------------------------------------------------------------------
        //  Unimplemented constructors and destructors
        // --------------------------------------------------------------------
        TSLinkedList(const TSLinkedList&);

        TSLinkedList& operator=(const TSLinkedList&);


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4ElemCount
        //      The number of objects currently in the container. It is
        //      maintained as elems are added or removed.
        //
        //  __c4MaxElemCount
        //      The maximum number objects this array can hold. When the elem
        //      count exceeds this limit, a runtime error is generated.
        //
        //  __pnodeHead
        //      This is a pointer to the 1st node of the linked list. If it
        //      is 0, then there are no elements in the list.
        //
        //  __pnodTail
        //      This is the pointer to the last node of the linked list. If it
        //      is 0, then there are no elements in the list.
        // --------------------------------------------------------------------
        tCIDLib::TCard4         __c4ElemCount;
        tCIDLib::TCard4         __c4MaxElemCount;
        TSLstNode*              __pnodeHead;
        TSLstNode*              __pnodeTail;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TSLinkedList,TObject)
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  TSLstNode: Constructors and destructors
// ----------------------------------------------------------------------------
inline TSLstNode::TSLstNode() :

    __pnodeNext(0)
{
}

inline TSLstNode::~TSLstNode()
{
}


// ----------------------------------------------------------------------------
//  TSLstNode: Public, non-virtual methods
// ----------------------------------------------------------------------------
inline TSLstNode* TSLstNode::pnodeNext() const
{
    return __pnodeNext;
}



// ----------------------------------------------------------------------------
//  TSLinkedList: Public operators
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean
TSLinkedList::operator!=(const TSLinkedList& llstToTest) const
{
    return !operator==(llstToTest);
}


// ----------------------------------------------------------------------------
//  TSLinkedList: Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TSLinkedList::bIsEmpty() const
{
    if (__c4ElemCount)
        return kCIDLib::False;
    return kCIDLib::True;
}

inline tCIDLib::TBoolean
TSLinkedList::bIsFirstNode(TSLstNode* const pnodeTest) const
{
    if (pnodeTest == __pnodeHead)
        return kCIDLib::False;
    else
        return kCIDLib::True;
}

inline tCIDLib::TBoolean
TSLinkedList::bIsLastNode(TSLstNode* const pnodeTest) const
{
    if (pnodeTest == __pnodeTail)
        return kCIDLib::False;
    return kCIDLib::True;
}

inline tCIDLib::TCard4 TSLinkedList::c4ElemCount() const
{
    return __c4ElemCount;
}

inline tCIDLib::TCard4 TSLinkedList::c4MaxElemCount() const
{
    return __c4MaxElemCount;
}

inline TSLstNode* TSLinkedList::pnodeHead() const
{
    return __pnodeHead;
}

inline TSLstNode* TSLinkedList::pnodeTail() const
{
    return __pnodeTail;
}
