//
// NAME: CIDLib_SLinkedList.Cpp
//
// DESCRIPTION:
//
//  This module implements the concrete collection class, TSLinkedList, and its
//  abstract linked list node class. The linked list class is defined in terms
//  of the abstract node class, from which many type specific versions are
//  instantiated via a template. The derived types determine the type of data
//  is in a node and what capabilities that the data must provide.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/22/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our standard members macros
// ----------------------------------------------------------------------------
RTTIData(TSLstNode,TObject)
RTTIData(TSLinkedList,TObject)



// ----------------------------------------------------------------------------
//   CLASS: TSLinkedList
//  PREFIX: llst
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TSLinkedList: Constructors and Destructors
// ----------------------------------------------------------------------------

TSLinkedList::TSLinkedList(const tCIDLib::TCard4 c4MaxElems) :

    __c4ElemCount(0)
    , __c4MaxElemCount(c4MaxElems)
    , __pnodeHead(0)
    , __pnodeTail(0)
{
    // If the max items is 0, then set it to the max
    if (!__c4MaxElemCount)
        __c4MaxElemCount = kCIDLib::c4MaxCard;
}

TSLinkedList::TSLinkedList() :

    __c4ElemCount(0)
    , __c4MaxElemCount(kCIDLib::c4MaxCard)
    , __pnodeHead(0)
    , __pnodeTail(0)
{
}

TSLinkedList::~TSLinkedList()
{
    Flush();
}


// ----------------------------------------------------------------------------
//  TSLinkedList: Public operators
// ----------------------------------------------------------------------------

tCIDLib::TBoolean
TSLinkedList::operator==(const TSLinkedList& llstToTest) const
{
    if (__c4MaxElemCount != llstToTest.__c4MaxElemCount)
        return kCIDLib::False;

    if (__c4ElemCount != llstToTest.__c4ElemCount)
        return kCIDLib::False;

    return kCIDLib::True;
}


// ----------------------------------------------------------------------------
//  TSLinkedList: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TSLinkedList::AppendNode(TSLstNode* const pnodeNew)
{
    // Make sure that we are not overflowing here
    if (__c4ElemCount >= __c4MaxElemCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_TooManyNodes
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
        );
    }

    // Bump up the count of items
    __c4ElemCount++;

    //
    //  Check for special case of an empty list with no entries yet. If so,
    //  then just store away the new node.
    //
    if (!__pnodeHead)
    {
        // No current nodes, so point the head and tail at this node
        __pnodeHead  = pnodeNew;
        __pnodeTail  = pnodeNew;
    }
     else
    {
        // Point the current tail's next at the new node
        __pnodeTail->__pnodeNext = pnodeNew;

        // Set the new node's next node to 0
        pnodeNew->__pnodeNext = 0;

        // Now store it as the new tail
        __pnodeTail = pnodeNew;
    }
}


tCIDLib::TCard4
TSLinkedList::c4MaxElemCount(const tCIDLib::TCard4 c4NewMax)
{
    if (c4NewMax < __c4ElemCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_NotEnoughNodes
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TCardinal(c4NewMax)
            , TCardinal(__c4ElemCount)
        );
    }

    __c4MaxElemCount = c4NewMax;
    return c4NewMax;
}


tCIDLib::TVoid TSLinkedList::Flush()
{
    // If no head node, then an empty list so just return
    if (!__pnodeHead)
        return;

    // There is at least one node, so loop through them
    TSLstNode* pnodeCur  = __pnodeHead;
    TSLstNode* pnodeNext = 0;

    do
    {
        // Before we destroy this node, save its next node
        pnodeNext = pnodeCur->__pnodeNext;

        // Free the node object
        delete pnodeCur;

        // Put the saved next node into the current node pointer
        pnodeCur = pnodeNext;

    }   while (pnodeCur);

    // 0 out the current item count and the head/tail pointers
    __c4ElemCount   = 0;
    __pnodeHead     = 0;
    __pnodeTail     = 0;
}


tCIDLib::TVoid
TSLinkedList::FlushNode(TSLstNode* const pnodeDelete)
{
    //
    //  We need to run up to the node in question and find its previous
    //  node, and this also insures that its in this list as well.
    //
    TSLstNode* pnodePrev = 0;

    if (__pnodeHead && (pnodeDelete != __pnodeHead))
    {
        pnodePrev = __pnodeHead;
        while (pnodePrev)
        {
            if (pnodePrev->__pnodeNext == pnodeDelete)
                break;
            pnodePrev = pnodePrev->__pnodeNext;
        }
    }

    // If not found, then the node was not part of this list.
    if (!pnodePrev && (pnodeDelete != __pnodeHead))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_NodeNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }

    //
    //  Get a pointer to the next node of the passed node, so we don't
    //  have to worry about having changed it below.
    //
    TSLstNode* pnodeNext = pnodeDelete->__pnodeNext;

    //
    //  Adjust the next node of pnodePrev to skip over the current node,
    //  thus removing it from the chain. Check for the special case that
    //  this is the first or last node, or the only node.
    //
    if (__c4ElemCount == 1)
    {
        __pnodeHead = 0;
        __pnodeTail = 0;
    }
     else if (!pnodePrev)
    {
        //
        //  The delete record is the start of the line, so we have to point
        //  the list's head to nodeNext.
        //
        __pnodeHead  = pnodeNext;
    }
     else if (!pnodeNext)
    {
        //
        //  The delete record is end of the line, so this one is easy. Just
        //  make the pnodePrev's next node nul, which makes it the new end
        //  of the list and store it as the new tail node.
        //
        pnodePrev->__pnodeNext = 0;
        __pnodeTail = pnodePrev;
    }
     else
    {
        //
        //  No special conditions, just do the normal patch around the
        //  deleted node
        //
       pnodePrev->__pnodeNext = pnodeNext;
    }

    // Bump down the count of nodes
    __c4ElemCount--;

    // Delete the node object
    delete pnodeDelete;
}


tCIDLib::TVoid
TSLinkedList::InsertNode(   TSLstNode* const    pnodeAfter
                            , TSLstNode* const  pnodeNew)
{
    //
    //  If the after node is 0, then that means insert it at the head,
    //  so we just turn it into a call to PrependNode().
    //
    if (!pnodeAfter)
    {
        PrependNode(pnodeNew);
        return;
    }

    //
    //  If there are no current nodes, then an error because pnodeBefore
    //  cannot exist.
    //
    if (!__c4ElemCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_NodeNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }

    // Make sure that we are not overflowing here
    if (__c4ElemCount >= __c4MaxElemCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_TooManyNodes
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_OutResource
        );
    }

    //
    //  Get a copy of the after node's next node pointer, so we don't
    //  have to worry about when we change it below.
    //
    TSLstNode* pnodeNext = pnodeAfter->__pnodeNext;

    //
    //  Insert the node into the list. Handle the special case of the
    //  insert after node being the tail node.
    //
    if (__pnodeTail == pnodeAfter)
    {
        // Point the list tail at the new node
        __pnodeTail = pnodeNew;

        // Point the after node's next node at the new tail
        pnodeAfter->__pnodeNext = pnodeNew;
    }
     else
    {
       // No special case, just do the normal patch around
       pnodeAfter->__pnodeNext = pnodeNew;
       pnodeNew->__pnodeNext = pnodeNext;
    }

    // Bump up the count of items
    __c4ElemCount++;
}


tCIDLib::TVoid TSLinkedList::MoveToHead(TSLstNode* const pnodeToMove)
{
    // If there are not at least 2 nodes, then nothing to do
    if (__c4ElemCount < 2)
        return;

    // If this node is already the first, then nothing to do
    if (pnodeToMove == __pnodeHead)
        return;

    //
    //  Find the previous node of the passed node, which also makes sure
    //  that the node is in the list.
    //
    TSLstNode* pnodePrev = 0;

    if (__pnodeHead && (pnodeToMove != __pnodeHead))
    {
        pnodePrev = __pnodeHead;
        while (pnodePrev)
        {
            if (pnodePrev->__pnodeNext == pnodeToMove)
                break;
            pnodePrev = pnodePrev->__pnodeNext;
        }
    }

    // If not found, then the node was not part of this list.
    if (!pnodePrev && (pnodeToMove != __pnodeHead))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_NodeNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }

    //
    //  Pull the node out of the list. If its the last one, then handle that
    //  specially. We already checked for it already being at the head.
    //
    if (!pnodeToMove->__pnodeNext)
    {
        // Set our previous nodes's next node to 0
        pnodePrev->__pnodeNext = 0;

        // And make it the new tail
        __pnodeTail = pnodePrev;
    }
     else
    {
        // Set our previous node's next node to our next node
        pnodePrev->__pnodeNext = pnodeToMove->__pnodeNext;
    }

    // Bump down the count of nodes
    __c4ElemCount--;

    // And insert the node at the start
    PrependNode(pnodeToMove);
}


tCIDLib::TVoid TSLinkedList::MoveToTail(TSLstNode* const pnodeToMove)
{
    // If there are not at least 2 nodes, then nothng to do
    if (__c4ElemCount < 2)
        return;

    // If this node is already the last, then nothing to do
    if (!pnodeToMove->__pnodeNext)
        return;

    //
    //  Find the previous node of the passed node, which also makes sure
    //  that the node is in the list.
    //
    TSLstNode* pnodePrev = 0;

    if (__pnodeHead && (pnodeToMove != __pnodeHead))
    {
        pnodePrev = __pnodeHead;
        while (pnodePrev)
        {
            if (pnodePrev->__pnodeNext == pnodeToMove)
                break;
            pnodePrev = pnodePrev->__pnodeNext;
        }
    }

    // If not found, then the node was not part of this list.
    if (!pnodePrev && (pnodeToMove != __pnodeHead))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_NodeNotFound
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_AppError
        );
    }

    //
    //  Pull the node out of the list. Handle it being the first node
    //  separately. We already checked whether it was already at the tail.
    //
    if (!pnodePrev)
    {
        // Store the move node's next as the new head
        __pnodeHead = pnodeToMove->__pnodeNext;
    }
     else
    {
        // Set our previous node's next node to our next node
        pnodePrev->__pnodeNext = pnodeToMove->__pnodeNext;
    }

    // Bump down the count of nodes
    __c4ElemCount--;

    // And append the node that we removed
    AppendNode(pnodeToMove);
}


tCIDLib::TVoid TSLinkedList::PrependNode(TSLstNode* const pnodeNew)
{
    // If there are no current nodes, then just call AppendNode()
    if (!__c4ElemCount)
    {
        AppendNode(pnodeNew);
        return;
    }

    // Make sure that we are not overflowing here
    if (__c4ElemCount >= __c4MaxElemCount)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcCol_TooManyNodes
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_OutResource
        );
    }

    // Bump up the count of items
    __c4ElemCount++;

    //
    //  There are only two options here, one was an empty list and was
    //  handled above. The other is that we just patch in the new node.
    //
    //  If there was only 1 node, then __pnodeTail still points to the old
    //  head which is correct.
    //
    TSLstNode* pnodeCurFirst = __pnodeHead;

    __pnodeHead = pnodeNew;
    __pnodeHead->__pnodeNext = pnodeCurFirst;
}
