//
// NAME: CIDLib_ResourceName.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_ResourceName.Cpp module. This module
//  implements the TResourceName class. TResourceName is an abstraction for
//  the naming of shareable resources, such as events, semaphores, mutexes,
//  and memory. Since these things have different naming conventions on
//  each platform, CIDLib uses this abstraction to make such names portable.
//
//  A resource name consists of 3 parts. There is a company name part,
//  a subsystem name part, and a resource name part. This creates a name
//  hiearchy that is easily supported, but which still prevents any serious
//  possibility of name conflicts.
//
//  This class uses a kernel object, of class TKrnlRscName, which does all
//  of the actual work and contains all of the platform specific smarts.
//
//  This class will create a process unique name if you give it a process
//  id.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 05/01/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//  CLASS: TResourceName
// PREFIX: rsn
// ----------------------------------------------------------------------------
class CIDLIBEXP TResourceName :

    public TObject, public MFormattable, public MDuplicable, public MStreamable
{
    public  :
        // --------------------------------------------------------------------
        // Constructors and Destructors
        // --------------------------------------------------------------------
        TResourceName();

        TResourceName
        (
            const   TKrnlRscName&           krsnToUse
        );

        TResourceName
        (
            const   TString&                strCompany
            , const TString&                strSubsystem
            , const TString&                strResource
            , const tCIDLib::TProcessId     pidToUse = kCIDLib::pidInvalid
        );

        TResourceName
        (
            const   TResourceName&          rsnToCopy
        );

        ~TResourceName();


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        tCIDLib::TBoolean operator==
        (
            const   TResourceName&          rsnToCompare
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TResourceName&          rsnToCompare
        )   const;

        TResourceName& operator=
        (
            const   TResourceName&          rsnToAssign
        );


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bValid() const;

        tCIDLib::TProcessId pidOfName() const;

        tCIDLib::TVoid QueryNameParts
        (
                    TString&                strCompany
            ,       TString&                strSubsystem
            ,       TString&                strResource
        )   const;

        const TKrnlRscName& krsnThis() const;

        tCIDLib::TVoid SetName
        (
            const   TString&                strCompany
            , const TString&                strSubsystem
            , const TString&                strResource
            , const tCIDLib::TProcessId     pidToUse = kCIDLib::pidInvalid
        );

        const TString& strFullName
        (
            const   tCIDLib::ENamedRscTypes eType
        )   const;


    protected   :
        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strDest
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToWriteTo
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __eTypeCache
        //      This is the last type of resource we stored the full name
        //      for. If subsequent requests are of the same type, then
        //      we use the name already there.
        //
        //  __krsnThis
        //      This the kernel resource name object that really does all
        //      of the work.
        //
        //  __strFullName
        //      This is used to hold the full name of the resource. Its just
        //      to hold it, since the kernel resource name object creates
        //      it upon request. So it is filled in each time its needed.
        //      However, we cache the last type of resource we were asked
        //      to fill it for and don't bother if its the same as the last.
        // --------------------------------------------------------------------
        mutable tCIDLib::ENamedRscTypes __eTypeCache;
        TKrnlRscName                    __krsnThis;
        mutable TString                 __strFullName;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TResourceName,TObject)
        DefPolyDup(TResourceName)
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  Public operators
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean
TResourceName::operator!=(const TResourceName& rsnToCompare) const
{
    if (this == &rsnToCompare)
        return kCIDLib::False;
    return !operator==(rsnToCompare);
}

inline tCIDLib::TBoolean
TResourceName::operator==(const TResourceName& rsnToCompare) const
{
    if (this == &rsnToCompare)
        return kCIDLib::True;
    return (__krsnThis == rsnToCompare.__krsnThis);
}

inline TResourceName&
TResourceName::operator=(const TResourceName& rsnToAssign)
{
    if (this == &rsnToAssign)
        return *this;

    __krsnThis == rsnToAssign.__krsnThis;
    return *this;
}


// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TResourceName::bValid() const
{
    return __krsnThis.bValid();
}

inline tCIDLib::TProcessId TResourceName::pidOfName() const
{
    return __krsnThis.pidOfName();
}

inline const TKrnlRscName& TResourceName::krsnThis() const
{
    return __krsnThis;
}
