//
// NAME: CIDLib_ResourceName.Cpp
//
// DESCRIPTION:
//
//  This module implements the TResourceName class, an abstraction for the
//  names of shareable resources.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 05/01/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TResourceName,TObject)


// ----------------------------------------------------------------------------
//  CLASS: TResourceName
// PREFIX: rsn
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TResourceName: Constructors and destructors
// ----------------------------------------------------------------------------

TResourceName::TResourceName() :

    __eTypeCache(tCIDLib::ENamedRsc_Event)
{
    strFullName(__eTypeCache);
}

TResourceName::TResourceName(const TKrnlRscName& krsnToUse) :

    __eTypeCache(tCIDLib::ENamedRsc_Event)
    , __krsnThis(krsnToUse)
{
    strFullName(__eTypeCache);
}

TResourceName::TResourceName(   const   TString&            strCompany
                                , const TString&            strSubsystem
                                , const TString&            strResource
                                , const tCIDLib::TProcessId pidToUse) :
    __eTypeCache(tCIDLib::ENamedRsc_Event)
    , __krsnThis
    (
        strCompany.pszData()
        , strSubsystem.pszData()
        , strResource.pszData()
        , pidToUse
    )
{
    strFullName(__eTypeCache);
}

TResourceName::TResourceName(const TResourceName& rsnToCopy) :

    __eTypeCache(rsnToCopy.__eTypeCache)
    , __krsnThis(rsnToCopy.__krsnThis)
    , __strFullName(rsnToCopy.__strFullName)
{
}

TResourceName::~TResourceName()
{
}


// ----------------------------------------------------------------------------
//  TResourceName: Public, non-virtual methods
// ----------------------------------------------------------------------------

inline tCIDLib::TVoid
TResourceName::QueryNameParts(  TString&    strCompany
                                , TString&  strSubsystem
                                , TString&  strResource) const
{
    strCompany = __krsnThis.pszCompanyName();
    strSubsystem = __krsnThis.pszSubsystemName();
    strResource = __krsnThis.pszResourceName();
}


tCIDLib::TVoid
TResourceName::SetName( const   TString&            strCompany
                        , const TString&            strSubsystem
                        , const TString&            strResource
                        , const tCIDLib::TProcessId pidToUse)
{
    // Pass this on to our kernel object
    __krsnThis.SetName
    (
        strCompany.pszData()
        , strSubsystem.pszData()
        , strResource.pszData()
        , pidToUse
    );

    // And update our name cache
    strFullName(__eTypeCache);
}


const TString&
TResourceName::strFullName(const tCIDLib::ENamedRscTypes eType) const
{
    // If this type is the cached name, then just return what we've got
    if (eType == __eTypeCache)
        return __strFullName;

    // Its a different type, so store the cache type
    __eTypeCache = eType;

    // Build the full name into a temp string that should be big enough
    tCIDLib::Tch szTmp[2048];
    __krsnThis.BuildFullName(szTmp, c4MaxBufChars(szTmp), eType);

    // Now update the string memory and return it
    __strFullName = szTmp;
    return __strFullName;
}


// ----------------------------------------------------------------------------
//  TResource: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TResourceName::_FormatTo(TTextStream& strmDest) const
{
    //  Format the separate name parts into the text stream.
    strmDest << __krsnThis.pszCompanyName() << L","
             << __krsnThis.pszSubsystemName() << L","
             << __krsnThis.pszResourceName();
}


tCIDLib::TVoid TResourceName::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Stream in the three name parts
    TString strCompany;
    TString strSubsystem;
    TString strResource;

    strmToReadFrom >> strCompany;
    strmToReadFrom >> strSubsystem;
    strmToReadFrom >> strResource;

    // And stream in the process id
    tCIDLib::TProcessId pidTmp;
    strmToReadFrom >> pidTmp;

    // And set this new info on the resource name object
    __krsnThis = TKrnlRscName
    (
        strCompany.pszData()
        , strSubsystem.pszData()
        , strResource.pszData()
        , pidTmp
    );
}

tCIDLib::TVoid TResourceName::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Get the name parts into string objects
    TString strCompany;
    TString strSubsystem;
    TString strResource;
    QueryNameParts(strCompany, strSubsystem, strResource);

    //
    //  Now stream out the separate string objects that we just got
    //  filled in.
    //
    strmToWriteTo << strCompany;
    strmToWriteTo << strSubsystem;
    strmToWriteTo << strResource;

    // And stream out the process id
    strmToWriteTo << __krsnThis.pidOfName();
}
