//
// NAME: CIDLib_Range.Hpp
//
// DESCRIPTION:
//
//  This is the header for the CIDLib_Range.Cpp module. It provides the
//  TRange class. TRange is a small class that provides min/max values
//  for the range, a current value within that range, and an amount of the
//  range that is 'visible'.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/08/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//


#pragma pack(push, CIDLIBPACK)

// ----------------------------------------------------------------------------
//   CLASS: TRange
//  PREFIX: rng
// ----------------------------------------------------------------------------
class CIDLIBEXP TRange : public TObject, public MStreamable, public MDuplicable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TRange();

        TRange
        (
            const   TRange&                 rngToCopy
        );

        TRange
        (
            const   tCIDLib::TCard4         c4Visible
            , const tCIDLib::TInt4          i4CurValue
            , const tCIDLib::TInt4          i4MinValue
            , const tCIDLib::TInt4          i4MaxValue
        );


        // --------------------------------------------------------------------
        //  Public operator
        // --------------------------------------------------------------------
        tCIDLib::TBoolean operator==
        (
            const   TRange&                 rngToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TRange&                 rngToTest
        )   const;

        TRange& operator=
        (
            const   TRange&                 rngToAssign
        );

        tCIDLib::TInt4 operator--();

        tCIDLib::TInt4 operator--
        (
                    int
        );

        tCIDLib::TInt4 operator++();

        tCIDLib::TInt4 operator++
        (
                    int
        );


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TBoolean bAllVisible() const;

        tCIDLib::TBoolean bAtMax() const;

        tCIDLib::TBoolean bAtLegalMax() const;

        tCIDLib::TBoolean bAtMin() const;

        tCIDLib::TBoolean bAtMinOrLegalMax() const;

        tCIDLib::TBoolean bAtMinOrMax() const;

        tCIDLib::TCard4 c4AbsoluteValue() const;

        tCIDLib::TCard4 c4CalcPercent
        (
            const   tCIDLib::TInt4          i4Value
        )   const;

        tCIDLib::TCard4 c4CurPercent() const;

        tCIDLib::TCard4 c4FullRange() const;

        tCIDLib::TCard4 c4PercentVisible() const;

        tCIDLib::TCard4 c4Visible() const;

        tCIDLib::TCard4 c4Visible
        (
            const   tCIDLib::TCard4         c4New
        );

        tCIDLib::ERangeStates eState() const;

        tCIDLib::TInt4 i4CurValue() const;

        tCIDLib::TInt4 i4CurValue
        (
            const   tCIDLib::TInt4          i4New
        );

        tCIDLib::TInt4 i4MaxLegalValue() const;

        tCIDLib::TInt4 i4MinValue() const;

        tCIDLib::TInt4 i4MinValue
        (
            const   tCIDLib::TInt4          i4New
        );

        tCIDLib::TInt4 i4MaxValue() const;

        tCIDLib::TInt4 i4MaxValue
        (
            const   tCIDLib::TInt4          i4New
        );

        tCIDLib::TVoid SetAll
        (
            const   tCIDLib::TCard4         c4Visible
            , const tCIDLib::TInt4          i4CurValue
            , const tCIDLib::TInt4          i4MinValue
            , const tCIDLib::TInt4          i4MaxValue
        );

        tCIDLib::TVoid SetToMaxLegalValue();


    protected   :
        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private :
        // --------------------------------------------------------------------
        //  Private, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid __Validate
        (
            const   tCIDLib::TCard4         c4Line
        );


        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4Visible
        //      The number of the items in the range that are visible. If this
        //      is set to 1, then it basically has no effect and the range is
        //      just like a range without a 'visible count'. The max legal
        //      range will be the same as the max value.
        //
        //  __i4CurValue
        //      This is the current value within the range.
        //
        //  __i4MinValue
        //  __i4MaxValue
        //      These define the current range. The min is always less than
        //      or equal to the max.
        // --------------------------------------------------------------------
        tCIDLib::TCard4         __c4Visible;
        tCIDLib::TInt4          __i4CurValue;
        tCIDLib::TInt4          __i4MinValue;
        tCIDLib::TInt4          __i4MaxValue;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TRange,TObject)
        DefPolyDup(TRange)
};

#pragma pack(pop)


// ----------------------------------------------------------------------------
//  Public operators
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TRange::operator!=(const TRange& rngToTest) const
{
    return !operator==(rngToTest);
}


// ----------------------------------------------------------------------------
//  Public, non-virtual methods
// ----------------------------------------------------------------------------
inline tCIDLib::TBoolean TRange::bAtMax() const
{
    return (__i4CurValue == __i4MaxValue);
}

inline tCIDLib::TBoolean TRange::bAtLegalMax() const
{
    return (__i4CurValue == i4MaxLegalValue());
}

inline tCIDLib::TBoolean TRange::bAtMin() const
{
    return (__i4CurValue == __i4MinValue);
}

inline tCIDLib::TBoolean TRange::bAtMinOrLegalMax() const
{
    return (bAtMin() || bAtLegalMax());
}

inline tCIDLib::TBoolean TRange::bAtMinOrMax() const
{
    return (bAtMin() || bAtMax());
}

inline tCIDLib::TCard4 TRange::c4AbsoluteValue() const
{
    return tCIDLib::TCard4(__i4CurValue - __i4MinValue);
}

inline tCIDLib::TCard4 TRange::c4FullRange() const
{
    return tCIDLib::TCard4(__i4MaxValue - __i4MinValue)+1;
}

inline tCIDLib::TCard4 TRange::c4Visible() const
{
    return __c4Visible;
}

inline tCIDLib::TCard4 TRange::c4Visible(const tCIDLib::TCard4 c4New)
{
    __c4Visible = c4New;
    return __c4Visible;
}

inline tCIDLib::TInt4 TRange::i4CurValue() const
{
    return __i4CurValue;
}

inline tCIDLib::TInt4 TRange::i4MinValue() const
{
    return __i4MinValue;
}

inline tCIDLib::TInt4 TRange::i4MaxValue() const
{
    return __i4MaxValue;
}

inline tCIDLib::TVoid TRange::SetToMaxLegalValue()
{
    __i4MinValue = i4MaxLegalValue();
}
