//
// NAME: CIDLib_Range.Cpp
//
// DESCRIPTION:
//
//  This module implements the TRange class, which provides a mechanism for
//  dealing with a value that must stay within a range, and to covert it
//  in and out via various conversions (e.g. percent of range.)
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 11/08/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Facility specific includes
// ----------------------------------------------------------------------------
#include    "CIDLib_.Hpp"


// ----------------------------------------------------------------------------
//  Do our standard RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TRange,TObject)



// ----------------------------------------------------------------------------
//   CLASS: TRange
//  PREFIX: rng
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TRange: Constructors and Destructors
// ----------------------------------------------------------------------------

TRange::TRange() :

    __c4Visible(100)
    , __i4CurValue(0)
    , __i4MinValue(0)
    , __i4MaxValue(100)
{
}

TRange::TRange(const TRange& rngToCopy) :

    __c4Visible(rngToCopy.__c4Visible)
    , __i4CurValue(rngToCopy.__i4CurValue)
    , __i4MinValue(rngToCopy.__i4MinValue)
    , __i4MaxValue(rngToCopy.__i4MaxValue)
{
}

TRange::TRange( const   tCIDLib::TCard4 c4Visible
                , const tCIDLib::TInt4  i4CurValue
                , const tCIDLib::TInt4  i4MinValue
                , const tCIDLib::TInt4  i4MaxValue) :

    __c4Visible(c4Visible)
    , __i4CurValue(i4CurValue)
    , __i4MinValue(MinVal(i4MinValue, i4MaxValue))
    , __i4MaxValue(MaxVal(i4MinValue, i4MaxValue))
{
    //
    //  Validate it. It does not matter here that we have already stored
    //  bad stuff because the object will never construct anyway.
    //
    __Validate(__LINE__);
}


// ----------------------------------------------------------------------------
//  TRange: Public operators
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TRange::operator==(const TRange& rngToTest) const
{
    if (rngToTest.__c4Visible != __c4Visible)
        return kCIDLib::False;

    if (rngToTest.__i4CurValue != __i4CurValue)
        return kCIDLib::False;

    if (rngToTest.__i4MinValue != __i4MinValue)
        return kCIDLib::False;

    if (rngToTest.__i4MaxValue != __i4MaxValue)
        return kCIDLib::False;

    return kCIDLib::True;
}


TRange& TRange::operator=(const TRange& rngToAssign)
{
    if (this == &rngToAssign)
        return *this;

    __c4Visible = rngToAssign.__c4Visible;
    __i4CurValue = rngToAssign.__i4CurValue;
    __i4MinValue = rngToAssign.__i4MinValue;
    __i4MaxValue = rngToAssign.__i4MaxValue;

    return *this;
}


tCIDLib::TInt4 TRange::operator--()
{
    if (__i4CurValue > __i4MinValue)
        __i4CurValue--;
    return __i4CurValue;
}

tCIDLib::TInt4 TRange::operator--(int)
{
    if (__i4CurValue > __i4MinValue)
        __i4CurValue--;
    return __i4CurValue;
}

tCIDLib::TInt4 TRange::operator++()
{
    if (__i4CurValue < __i4MaxValue)
        __i4CurValue++;
    return __i4CurValue;
}

tCIDLib::TInt4 TRange::operator++(int)
{
    if (__i4CurValue < __i4MaxValue)
        __i4CurValue++;
    return __i4CurValue;
}


// ----------------------------------------------------------------------------
//  TRange: Public, non-virtual methods
// ----------------------------------------------------------------------------

tCIDLib::TBoolean TRange::bAllVisible() const
{
    if (__c4Visible >= c4FullRange())
        return kCIDLib::True;

    return kCIDLib::False;
}


tCIDLib::TCard4 TRange::c4CalcPercent(const tCIDLib::TInt4 i4Value) const
{
    if ((i4Value < __i4MinValue) || (i4Value > __i4MaxValue))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcRange_PercentValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(i4Value)
            , TInteger(__i4MinValue)
            , TInteger(__i4MaxValue)
        );
    }

    //
    //  Get the two values in question into floating point form. Note that
    //  we don't use c4FullRange() because it returns the range, not the
    //  actual max normalized value which is 1 less.
    //
    tCIDLib::TFloat4 f4Full  = tCIDLib::TFloat4(__i4MaxValue - __i4MinValue);
    tCIDLib::TFloat4 f4Cur   = tCIDLib::TFloat4(i4Value - __i4MinValue);

    // Calculate in floating point and cast to an integral for return
    return tCIDLib::TCard4((f4Cur / f4Full) * 100.0);
}


tCIDLib::TCard4 TRange::c4CurPercent() const
{
    //
    // Get the two values in question into floating point form. Note that
    //  we don't use c4FullRange() because it returns the range, not the
    //  actual max normalized value which is 1 less.
    //
    tCIDLib::TFloat4 f4Full  = tCIDLib::TFloat4(__i4MaxValue - __i4MinValue);
    tCIDLib::TFloat4 f4Cur   = tCIDLib::TFloat4(__i4CurValue - __i4MinValue);

    // Calculate in floating point and cast to an integral for return
    return tCIDLib::TCard4((f4Cur / f4Full) * 100.0);
}


tCIDLib::TCard4 TRange::c4PercentVisible() const
{
    //
    // Get the two values in question into floating point form. Note that
    //  we don't use c4FullRange() because it returns the range, not the
    //  actual max normalized value which is 1 less.
    //
    tCIDLib::TFloat4 f4Full     = tCIDLib::TFloat4(__i4MaxValue - __i4MinValue);
    tCIDLib::TFloat4 f4Visible  = tCIDLib::TFloat4(__c4Visible);

    // Calculate in floating point and cast to an integral for return
    tCIDLib::TCard4 c4Ret = tCIDLib::TCard4((f4Visible / f4Full) * 100.0);

    // Clip it to a percent
    if (c4Ret > 100)
        c4Ret = 100;

    return c4Ret;
}


tCIDLib::ERangeStates TRange::eState() const
{
    if (__i4CurValue == __i4MinValue)
        return tCIDLib::ERangeState_Min;
    else if (__i4CurValue == __i4MaxValue)
        return tCIDLib::ERangeState_Max;
    else if (__i4CurValue == i4MaxLegalValue())
        return tCIDLib::ERangeState_LegalMax;

    return tCIDLib::ERangeState_None;
}


tCIDLib::TInt4 TRange::i4CurValue(const tCIDLib::TInt4 i4New)
{
    if ((i4New < __i4MinValue) || (i4New > i4MaxLegalValue()))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcRange_InvalidValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(i4New)
            , TCardinal(c4FullRange())
            , TCardinal(__c4Visible)
        );
    }

    __i4CurValue = i4New;
    return i4New;
}


tCIDLib::TInt4 TRange::i4MaxLegalValue() const
{
    // Get the full range for quick access
    tCIDLib::TCard4 c4Full = c4FullRange();

    //
    //  If we have more items than are visible, then the max value is 
    //  (full range minus visible) above the min value.
    //
    if (c4Full > __c4Visible)
        return __i4MinValue + (c4Full - __c4Visible);

    //
    //  If there are more visible than there are items, then we never
    //  get off the min value because we never have to scroll.
    //
    return __i4MinValue;
}


tCIDLib::TInt4 TRange::i4MinValue(const tCIDLib::TInt4 i4New)
{
    if (__i4MaxValue < i4New)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcRange_BadMinMax
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(__i4MaxValue)
            , TInteger(i4New)
        );
    }

    __i4MinValue = i4New;

    // Make sure we don't leave the current value hanging out
    if (__i4CurValue < __i4MinValue)
        __i4CurValue = __i4MinValue;

    return i4New;
}

tCIDLib::TInt4 TRange::i4MaxValue(const tCIDLib::TInt4 i4New)
{
    if (i4New < __i4MinValue)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDErrs::errcRange_BadMinMax
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(i4New)
            , TInteger(__i4MinValue)
        );
    }

    __i4MaxValue = i4New;

    // Make sure we don't leave the current value hanging out
    if (__i4CurValue > i4MaxLegalValue())
        __i4CurValue = i4MaxLegalValue();

    return i4New;
}


tCIDLib::TVoid TRange::SetAll(  const   tCIDLib::TCard4 c4Visible
                                , const tCIDLib::TInt4  i4CurValue
                                , const tCIDLib::TInt4  i4MinValue
                                , const tCIDLib::TInt4  i4MaxValue)
{
    //
    //  Create a temp first. If any of the parms were illegal, we will
    //  get an exception and never affect this object.
    //
    TRange  rngTest(c4Visible, i4CurValue, i4MinValue, i4MaxValue);

    // We did not get an exception so copy it in
    __c4Visible = c4Visible;
    __i4CurValue = i4CurValue;
    __i4MinValue = i4MinValue;
    __i4MaxValue = i4MaxValue;
}


// ----------------------------------------------------------------------------
//  TRange: Protected, inherited methods
// ----------------------------------------------------------------------------

tCIDLib::TVoid TRange::_FormatTo(TTextStream& strmToWriteTo) const
{
    strmToWriteTo   << L"Val=" << __i4CurValue
                    << L", Min=" << __i4MinValue
                    << L", Max=" << __i4MaxValue
                    << L", Visible=" << __c4Visible;
}

tCIDLib::TVoid TRange::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Parse the data fields
    strmToReadFrom >> __c4Visible;
    strmToReadFrom >> __i4CurValue;
    strmToReadFrom >> __i4MinValue;
    strmToReadFrom >> __i4MaxValue;

    // Validate it
    __Validate(__LINE__);
}

tCIDLib::TVoid TRange::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Write out the data fields
    strmToWriteTo << __c4Visible;
    strmToWriteTo << __i4CurValue;
    strmToWriteTo << __i4MinValue;
    strmToWriteTo << __i4MaxValue;
}


// ----------------------------------------------------------------------------
//  TRange: Private, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __Validate
//
// DESCRIPTION:
//
//  Validates the current settings for consistency. If not valid, then we
//  log a fatal error. By passing in the name by value, we prevent having
//  to construct strings objects unless there's an error!
// ---------------------------------------
//   INPUT: c4Line is the line number from whence we were called.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TRange::__Validate(const tCIDLib::TCard4 c4Line)
{
    if (__i4MaxValue < __i4MinValue)
    {
        facCIDLib.LogErr
        (
            __FILE__
            , c4Line
            , kCIDErrs::errcRange_BadMinMax
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(__i4MaxValue)
            , TInteger(__i4MinValue)
        );
    }

    if ((__i4CurValue < __i4MinValue) || (__i4CurValue > i4MaxLegalValue()))
    {
        facCIDLib.LogErr
        (
            __FILE__
            , c4Line
            , kCIDErrs::errcRange_InvalidValue
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_BadParms
            , TInteger(__i4CurValue)
            , TCardinal(c4FullRange())
            , TCardinal(__c4Visible)
        );
    }
}
